/**********************************************************************
 ** Item class: These functions manipulate all items in the mud. An item
 **             for the most part is any object that is not alive. Most
 **             are gettable, with a few exceptions like markers and doors
 **
 **
 **    Last reviewed: version 0.14
 **
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **    
 **********************************************************************/

#ifndef ITEM_C
#define ITEM_C

#include "config.h"
#include "sysdep.h"
#include "strings.h"
#include "mudtypes.h"
#include "item.h"
#include "mudobject.h"
#include "objtype.h"
#include "builder.h"
#include "location.h"
#include "flags.h"
#include "newfuncts.h"
#include "global.h"

/***********************************************************************
 ** Item (constructor) - creates the item
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Item::Item()
{
   state = 0;   

   itemflags = new_Flags(2);
}

/***********************************************************************
 ** ~Item (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Item::~Item()
{
   delete_Flags(itemflags);
}


/***********************************************************************
 ** get_itemflags - gets the item flags object
 **
 ** Parameters: Nothing
 **
 ** Returns: a pointer to the item flags structure
 **
 ***********************************************************************/

Flags *Item::get_itemflags()
{
   return itemflags;
}


/***********************************************************************
 ** set_attrib_itemflags - for set attribute, sets the itemflags for any
 **                        item class object
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Item::set_attrib_itemflags(Parse *the_parsed, Builder *the_builder)
{
   Flags *tmp_itemflags;
   int flagnum;
   Strings holder;

   tmp_itemflags = get_itemflags();
      
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("Set which item flag?\n");
      return -1;
   }
   holder.assign_word(the_parsed->get_speech(), 1);
     
   if ((flagnum = 
       tmp_itemflags->get_by_name(holder.str_show(), itemflagnames)) == -1)
   {
      the_builder->send_bldr("That is not an item flag.\n");
      return -1;
   }

   holder.assign_word(the_parsed->get_speech(), 2);
   if (holder.str_show() == NULL)
   {
      the_builder->send_bldr("Set that flag to what?\n"
                                "Valid choices are: On or Off\n");
      return -1;
   }

   if (holder.str_n_cmp("On", holder.str_len()))
      tmp_itemflags->set_flag(flagnum);
   else if (holder.str_n_cmp("Off", holder.str_len()))
      tmp_itemflags->clr_flag(flagnum);
   else
   {
      the_builder->send_bldr("That is not a valid setting.\n"
                             "Valid choices are: On or Off\n");
      return -1;
   }
   the_builder->send_bldr("Flag &+M%s&* has been set to: %s\n",
       itemflagnames[flagnum], (tmp_itemflags->get_flag(flagnum)) ? 
			     "&+GOn&*" : "&+ROff&*");
   return 1;
 
}


/***********************************************************************
 ** get_guards - gets the guards string for this item
 **
 ***********************************************************************/

char *Item::get_guards()
{
   return guards.str_show();
}


/***********************************************************************
 ** set_guards - sets the guard string to what is passed in
 **
 ***********************************************************************/

void Item::set_guards(char *the_guards)
{
   guards = the_guards;
}


/***********************************************************************
 ** set_state - sets the state of the item
 **
 ** Parameters: new_state - the new state we are setting it to
 **
 ** Returns:  Nothing
 **
 ***********************************************************************/
   
void Item::set_state(int new_state)
{
   state = new_state;
}


/***********************************************************************
 ** get_state - gets the state of the item
 **
 ** Parameters: Nothing
 **
 ** Returns: the state of the object
 **
 ***********************************************************************/
   
int Item::get_state(void)
{
   return state;
}


/***********************************************************************
 ** is_guarded - is this object guarded by anyone
 **
 ** Parameters: the_loc - the location to search for guards in
 **
 ** Returns: Pointer to the guard for guarded
 **          NULL if not
 **
 ***********************************************************************/

Mobile *Item::is_guarded(Location *the_loc)
{
   LinkedList *the_contents;
   MudObject  *the_obj;
   Strings    holder;

   if (guards.str_show() == NULL)
      return 0;

   if (the_loc == NULL)
      return NULL;

   the_contents = the_loc->get_inventory();
   the_contents->reset_current();
   the_obj = the_contents->get_next();
   while (the_obj != NULL)
   {
      if (the_obj->get_type() == OBJ_TYPE_MOBILE)
      {
         holder.sprintf("%s@%s", the_obj->get_name(), the_obj->get_area());

         if ((guards.find_in_str(holder.str_show())) || 
             (!STRCASECMP(get_area(), the_obj->get_area()) && 
              guards.find_in_str(the_obj->get_name())))
            return ((Mobile *) the_obj);
      }
      the_obj = the_contents->get_next();
   }
   return NULL;
}



/***********************************************************************
 ** set_attrib_guards - for set_attrib, sets the guards attribute
 **
 ** Parameters: the_parsed - the parsed list from the user
 **             the_builder - the builder doing this command
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Item::set_attrib_guard(Parse *the_parsed, Builder *the_builder)
{
   if (the_parsed->get_speech() == NULL)
   {
      the_builder->send_bldr("You need to specify guards to set to.\n");
      return -1;
   }
   set_guards(the_parsed->get_speech());
   the_builder->send_bldr("Guards on %s set to: %s\n", get_name(), 
                                                        get_guards());
   set_modified(1);
   return 1;
}



/***********************************************************************
 ** write_item_attrib - writes all item attributes to the file
 **
 ** Parameters: the_file - the file that we are writing to
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Item::write_item_attrib(FILE *the_file)
{
   fprintf(the_file, "^%s^\n", (get_guards() == NULL) ? "" : get_guards());

   itemflags->write_flag(the_file);

   return 1;
}



/***********************************************************************
 ** read_item_attrib - reads in item attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area that we are reading
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Item::read_item_attrib(FILE *read_file, char *areaname, 
                                                          ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
      holder.sprintf("Invalid format in mudobject %s, area %s", get_name(), 
		     areaname);
      error_log->log_err(holder.str_show(), "read_item_attrib");
      return -1;
   }

   /* get the guards */
   the_token = get_token(read_file, '^');
   set_guards(the_token->the_string);

   /* read in the item flags */
   if (itemflags->read_flags(read_file, error_log) <= 0)
   {
      holder.sprintf("Error reading itemflags for item '%s@%s'", get_name(),
                                                                 get_area()); 
      error_log->log_err(holder.str_show(), "read_item_attrib");
      return -1;
   }   
   return 1;
}

/***********************************************************************
 ** copy_item_attrib - copies all the item attributes over from 
 **                    another item
 **
 ** Parameters: copy_from - the object we copy attributes from
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Item::copy_item_attrib(Item *copy_from)
{
   itemflags->copy_flags(copy_from->get_itemflags());
   set_guards(copy_from->get_guards());
   set_state(copy_from->get_state());

   return 1;
}


/***********************************************************************
 ** write_object - writes an item to a file.  This write raises an
 **                error as we should not be writing just items ever
 **
 ** Parameters: new_name - the new name that the new object should have
 **
 ***********************************************************************/

void Item::write_object(FILE *the_file, int build_format)
{
   build_format = 0;
   the_file = NULL;
   mainstruct->log_error("Tried to write item to file.", "write_object");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Item::describe(Player *the_player)
{
   the_player = NULL;
   mainstruct->log_error("Tried to describe an item ", "describe");
}


/***********************************************************************
 ** describe - raises an error for trying to describe an entity
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Item::describe(Builder *the_builder)
{
   the_builder = NULL;
   mainstruct->log_error("Tried to describe an item", "describe");
}

/***********************************************************************
 ** set_attrib - raises an error for trying to set attrib on an item
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

int Item::set_attrib(Builder *the_builder, Parse *the_parsed)
{
   the_builder = NULL;
   the_parsed = NULL;
   mainstruct->log_error("Tried to set attrib on an item", "set_attrib");
   return -1;
}


/***********************************************************************
 ** copy_object - copies attributes of another object to this one
 **
 ** Parameters: copy_obj - the object we won't copy cause it is an error
 **
 ** Returns: 1 for success, -1 for failure
 **
 ***********************************************************************/

int Item::copy_object(Entity *copy_obj)
{
   copy_obj = NULL;
   printf("Error!  Attempted to copy an item object\n");
   return -1;
}


/***********************************************************************
 ** get_mem_size_item - gets how much memory is being taken up by the
 **                     item part of this object
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Item::get_mem_size_item()
{
   int size = 0;

   size += itemflags->get_mem_size();
   size += guards.get_mem_size_dynamic();

   return size;
}

#endif



