/**********************************************************************
 ** Boat class:  handles all functions for a boat object, which allows you
 **              to enter it and move around on water
 **
 ** Reviewed through:
 **
 ** Copyright (C) 2000 George Noel (Slate)
 **
 **   This program is free software; you can redistribute it and/or modify
 **   it under the terms of the GNU General Public License as
 **   published by the Free Software Foundation; either version 2 of the 
 **   License, or any later version. 
 **
 **   This program is distributed in the hope that it will be useful, but 
 **   WITHOUT ANY WARRANTY; without even the implied warranty of 
 **   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU 
 **   General Public License for more details. 
 ** 
 **   You should have received a copy of the GNU General Public License 
 **   along with this program (in the docs dir); if not, write to the Free
 **   Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. 
 **
 **********************************************************************/

#ifndef BOAT_C
#define BOAT_C

#include "config.h"
#include "sysdep.h"
#include "moveable.h"
#include "boat.h"
#include "objtype.h"
#include "flags.h"
#include "strings.h"
#include "indflags.h"
#include "player.h"
#include "global.h"
#include "strings.h"
#include "gameflags.h"
#include "locflags.h"

/***********************************************************************
 ** Boat (constructor) - creates the item
 **
 ** Parameters: the_name - the name of the boat
 **             the_area - the area the boat belongs to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Boat::Boat(char *the_name, char *the_area)
{
   /* this should be the only place that this is set */
   obj_type = OBJ_TYPE_BOAT;
   
   if (the_name != NULL)
      set_name(the_name);
   
   if (the_area != NULL)
      set_area(the_area);

   size = weight = state = 0;  
}

/***********************************************************************
 ** ~Boat (destructor) - destroys it
 **
 ** Parameters: None
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

Boat::~Boat()
{
}


/***********************************************************************
 ** set_move_str - sets the portion of the string that will be displayed 
 **                when the player moves around
 **
 ** Parameters: new_str - the new value to set to
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Boat::set_move_str(char *new_str)
{
   move_str = new_str;
}


/***********************************************************************
 ** get_move_str - gets the portion of the string that will be displayed
 **                when the player moves around (in the boat)
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Boat::get_move_str(void)
{
   return move_str.str_show();
}


/***********************************************************************
 ** set_water_brief - sets the brief that will be displayed when the
 **                   boat is in water
 **
 ** Parameters: new_str - the new string
 **
 ** Returns: Nothing
 **
 ***********************************************************************/

void Boat::set_water_brief(char *new_str)
{
   water_brief = new_str;
}


/***********************************************************************
 ** get_water_brief - gets the brief that will be displayed when the
 **                   boat is in water
 **
 ** Parameters: None
 **
 ***********************************************************************/

char *Boat::get_water_brief(void)
{
   return water_brief.str_show();
}


/***********************************************************************
 ** read_boat_attrib - reads in boats attributes from the file
 **
 ** Parameters: read_file - the file to read in from
 **             areaname - the area we are reading from
 **             error_log - the error log to write any errors to
 **
 ** Returns:  1 for successful read
 **          -1 for errors in the read
 **
 ***********************************************************************/

int Boat::read_boat_attrib(FILE *read_file, char *areaname, ErrLog *error_log)
{
   token_record *the_token;
   Strings      holder;

   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_format(get_name(), areaname, "read_boat_attrib");
     return -1;
   }

   /* get the move_str of this boat */
   the_token = get_token(read_file, '^');
   set_move_str(the_token->the_string);   


   /* get the next item, it should be a '^', if not, raise error */
   the_token = get_token(read_file, '\0');
   if (the_token->token_type != T_CARROT)
   {
     error_log->invalid_format(get_name(), areaname, "read_boat_attrib");
     return -1;
   }

   /* get the water_brief of this boat */
   the_token = get_token(read_file, '^');
   set_water_brief(the_token->the_string);   

   return 1;
}


/***********************************************************************
 ** enter_boat - places the player in the boat object
 **
 ** Parameters: the_player - the player to place in the boat
 **
 ** Returns:  1 for success, -1 for failure, -2 for full
 **
 ***********************************************************************/

int Boat::enter_boat(Player *the_player)
{
   Flags *tmp_indflags;

   if ((get_size_held() + the_config.default_indsize) > get_capacity())
      return -2;

   tmp_indflags = the_player->get_indflags();
   tmp_indflags->set_flag(INDFLAG_INBOAT);

   the_player->set_location(this, the_player->get_loc());
   return 1;
}


/***********************************************************************
 ** exit_boat - places the player in the boat object
 **
 ** Parameters: the_player - the player to place in the boat
 **
 ** Returns:  1 for success, -1 for failure, -2 for full
 **
 ***********************************************************************/

int Boat::exit_boat(Player *the_player)
{
   Flags *tmp_indflags;

   tmp_indflags = the_player->get_indflags();
   tmp_indflags->clr_flag(INDFLAG_INBOAT);

   the_player->set_location(get_contained_by(), this);
   return 1;
}


/***********************************************************************
 ** write_object - writes the boat to a specified file in specified
 **                file format
 **
 ** Parameters: the_file - the file to write to
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
void Boat::write_object(FILE *the_file, int build_format)
{

   fprintf(the_file, "\nboat %s\n", get_name());
   if (build_format)
      fprintf(the_file, "%d\n", is_modified());
   write_mudobject_attrib(the_file);
   write_item_attrib(the_file);
   write_moveable_attrib(the_file);

   fprintf(the_file, "^%s^\n", get_move_str());
   fprintf(the_file, "^%s^\n", get_water_brief());
}

/***********************************************************************
 ** describe - describes the boat to a builder
 **
 ** Parameters: the_builder - the person to send all the data to
 **
 ***********************************************************************/

void Boat::describe(Builder *the_builder)
{
   the_builder->send_bldr(_("\n&+GBoat: \t\t&+M%s&*\n"), get_name());
   the_builder->send_bldr(_("&+GTitle: \t\t&+w%s&*\n"), get_title());
   the_builder->send_bldr(_("&+GAltnames: \t&+g%s&*\n"), get_altname());
   the_builder->send_bldr(_("&+GClones: \t&+g%s&*\n"), get_clones());
   the_builder->send_bldr(_("&+GSpecials: \t&+g%s&*\n"), get_special_str());
   the_builder->send_bldr(_("&+GGuards: \t&+g%s&*\n"), get_guards());
   the_builder->send_bldr(_("&+GLocation: \t&+M%s&*\n"), get_location());
   the_builder->send_bldr(_("&+GMoveStr: \t&+g%s&*\n"), get_move_str());
   the_builder->send_bldr(_("&+GSize: \t\t&+w%d&*\n"), get_size());
   the_builder->send_bldr(_("&+GWeight:\t\t&+w%d&*\n"), get_weight());
   the_builder->send_bldr(_("&+GBrief0:\n&*%s\n"), 
                                         ((Moveable *)this)->get_brief(0));
   the_builder->send_bldr(_("&+GBrief1:\n&*%s\n"), 
                                         ((Moveable *)this)->get_brief(1));
   the_builder->send_bldr(_("&+GWaterBrief:\n&*%s\n"), get_water_brief());
   the_builder->send_bldr(_("&+GDesc:&*\n%s\n"), get_desc());

   the_builder->send_bldr("\n");
}


/***********************************************************************
 ** describe - describes the boat to a player
 **
 ** Parameters: the_player - the person to send all the data to
 **
 ***********************************************************************/

void Boat::describe(Player *the_player)
{
   MudObject *tmp_container;
   Strings   container_name;

   the_player->send_plr(_("\n&+GBoat: \t\t&+M%s&*\n"), get_name());
   the_player->send_plr(_("&+GTitle: \t\t&+w%s&*\n"), get_title());
   the_player->send_plr(_("&+GAltnames: \t&+g%s&*\n"), get_altname());
   the_player->send_plr(_("&+GClones: \t&+g%s&*\n"), get_clones());
   the_player->send_plr(_("&+GSpecials: \t&+g%s&*\n"), get_special_str());
   the_player->send_plr(_("&+GGuards: \t&+g%s&*\n"), get_guards());
   the_player->send_plr(_("&+GStartLoc: \t&+M%s&*\n"), get_location());

   tmp_container = get_contained_by();
   if (tmp_container == NULL)
      container_name = "nowhere";
   else
      container_name.sprintf("%s@%s", tmp_container->get_name(), 
                                         tmp_container->get_area());

   the_player->send_plr(_("&+GCurrentLoc: \t&+M%s&*\n"), 
                                                 container_name.str_show());

   the_player->send_plr(_("&+GMoveStr: \t&+g%s&*\n"), get_move_str());
   the_player->send_plr(_("&+GSize: \t\t&+w%d&*\n"), get_size());
   the_player->send_plr(_("&+GWeight:\t\t&+w%d&*\n"), get_weight());
   the_player->send_plr(_("&+GBrief0:\n&*%s\n"), 
                                         ((Moveable *)this)->get_brief(0));
   the_player->send_plr(_("&+GBrief1:\n&*%s\n"), 
                                         ((Moveable *)this)->get_brief(1));
   the_player->send_plr(_("&+GWaterBrief:\n&*%s\n"), get_water_brief());
   the_player->send_plr(_("&+GDesc:&*\n%s\n"), get_desc());
   the_player->send_plr(_("&+YSize: \t\t\t&+W%d&*\n"), get_mem_size());

   list_specials(the_player);

   the_player->send_plr("\n");
}


/***********************************************************************
 ** set_attrib - sets a specified attribute to a specified value
 **
 ** Parameters: the_builder - the builder who is changing this attribute
 **
 ** Returns:  1 if successful
 **          -1 if failed
 **
 ***********************************************************************/
   
int Boat::set_attrib(Builder *the_builder, Parse *the_parsed){

   if (the_parsed->get_target1() == NULL)
   {   the_builder->
        send_bldr(_("You can set the following attributes on a boat.\n"
                  "   title, weight, size, altnames, clones, desc,\n"
                  "   itemflags, brief0, brief1, waterbrief, location\n,"
                  "   specials, movestr, capacity, and guards\n"));
       return -1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("title"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_title(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("weight"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_weight(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("size"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_size(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("altnames"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_altnames(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("guards"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_guard(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("specials"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_special(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("clones"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_clones(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("desc"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_desc(the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("itemflags"),
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_itemflags(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("brief0"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 0);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("brief1"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_brief(the_builder, 1);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("location"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_loc(the_parsed, the_builder);
   }
   if (!STRNCASECMP(the_parsed->get_target1(), _("capacity"), 
                               strlen(the_parsed->get_target1())))
   {
      return set_attrib_capacity(the_parsed, the_builder);
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("movestr"),
                               strlen(the_parsed->get_target1())))
   {
      Strings holder;
      
      if (the_parsed->get_speech() == NULL)
      {
         the_builder->
               send_bldr(_("You need to specify a MoveStr to set to.\n"));
         return -1;
      }
      holder.assign_word(the_parsed->get_speech(), 1);
      set_move_str(holder.str_show());

      the_builder->send_bldr(_("MoveStr on %s set to: %s\n"), get_name(), 
                                                     get_move_str());
      set_modified(1);
      return 1;
   }

   if (!STRNCASECMP(the_parsed->get_target1(), _("waterbrief"),
                               strlen(the_parsed->get_target1())))
   {
      if (the_builder->get_long_input(&water_brief) < 0)
      {
         the_builder->send_bldr(_("Error reading in input, failed!\n"));
         return -1;
      }
      set_modified(1);
      return 1;
   }

   the_builder->send_bldr(_("The attribute '%s' is not a 'boat' attribute.\n"),
                                           the_parsed->get_target1());
   return -1;
}


/***********************************************************************
 ** copy_object - copies the key to another object of a different name
 **
 ** Parameters: copy_obj - copy attributes from this object
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Boat::copy_object(Entity *copy_obj)
{
   Boat *copy_from;

   if (copy_obj->get_type() != OBJ_TYPE_BOAT)
      return 0;

   copy_from = (Boat *) copy_obj;

   /******* set the mudobject attributes *****/
   copy_mudobject_attrib((MudObject *) copy_from);

   /****** set the item attributes ******/
   copy_item_attrib((Item *) copy_from);

   /****** set the moveable attributes ******/
   copy_moveable_attrib((Moveable *) copy_from);

   set_move_str(copy_from->get_move_str());
   set_water_brief(copy_from->get_water_brief());

   return 1;
}

/***********************************************************************
 ** operator = - copies an object to this object
 **
 ** Parameters: None
 **
 ** Returns: a pointer to this object copied to
 **
 ***********************************************************************/

Boat *Boat::operator = (Boat *copy_from)
{
   if (!STRCASECMP(copy_from->get_name(), get_name()))
      return NULL;

   copy_object(copy_from);
   return this;
}


/***********************************************************************
 ** get_passengers - creates a string that tells what passengers are in
 **                  the boat;
 **
 ** Parameters: the_player - the player that we don't count in the string
 **             the_string - the string to place the names into
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Boat::get_passengers(Player *the_player, Strings *the_string)
{
   MudObject *tmp_obj;
   Individual *cur_ind = NULL;
   Individual *next_ind = NULL;
   int        first = 1;

   if (the_string == NULL)
      return -1;

   the_string->truncate(0);
   inventory.reset_current();
   tmp_obj = inventory.get_next();
   while (tmp_obj != NULL)
   {
      if (tmp_obj->is_an_individual())
      {
         if ((tmp_obj->get_type() == OBJ_TYPE_PLAYER) &&
             (((Player *) tmp_obj) == the_player))
	 {
            tmp_obj = inventory.get_next();
            continue;
         }
         cur_ind = next_ind;
         next_ind = ((Individual *) tmp_obj);

         if (cur_ind != NULL)
	 {
            if (first)
	    {
               the_string->str_copy(cur_ind->get_title());
               first = 0;
	    }
            else
	    {
               the_string->str_cat(", ");
               the_string->str_cat(cur_ind->get_title());
	    }
         }
      }
      tmp_obj = inventory.get_next();
   }

   if (next_ind == NULL)
      return 0;

   if (!first)
      the_string->str_cat(", ");

   if (cur_ind != NULL)
   {
      the_string->str_cat(cur_ind->get_title());
      the_string->str_cat(_(" and "));
   }
   the_string->str_cat(next_ind->get_title());
   return 1;
}


/***********************************************************************
 ** stop_passengers_fighting - stops all the passengers in the boat from
 **                            fighting since the boat fleed
 **
 ** Parameters: None
 **
 ** Returns:  1 if succeeded 
 **           0 if failed
 **
 ***********************************************************************/
int Boat::stop_passengers_fighting(void)
{
   MudObject *tmp_obj;
   Individual *the_ind;

   inventory.reset_current();
   tmp_obj = inventory.get_next();
   while (tmp_obj != NULL)
   {
      if (tmp_obj->is_an_individual())
      {
         the_ind = (Individual *) tmp_obj;
 
         the_ind->stop_fighting();
         mainstruct->clear_fighter(the_ind);
      }
      tmp_obj = inventory.get_next();
   }
   return 1;
}


/***********************************************************************
 ** move_location - moves the location of the players in the boat as well as
 **            the boat location to the new location
 **
 ** Parameters: new_loc - the new loc to move to
 **
 ** Returns:   1 if succeeded 
 **           -1 if failed
 **
 ***********************************************************************/
int Boat::move_location(Location *new_loc)
{
   MudObject  *tmp_obj;
   Individual *the_ind;
   Flags      *tmp_gameflags;
   Player     *the_player;
   LinkedNode *tmp_node;
   Strings    holder;

   if (set_location(new_loc, get_contained_by()) == -1)
   {
     holder.sprintf(_("Could not set boat location %s@%s for boat '%s@%s'"), 
	       new_loc->get_name(), new_loc->get_area(), get_name(), get_area());
      mainstruct->log_error(holder.str_show(), "move_location");
      return -1;
   }

   inventory.reset_current();
   tmp_obj = inventory.get_next();

   while (tmp_obj != NULL)
   {
      if (tmp_obj->is_an_individual())
      {
         the_ind = (Individual *) tmp_obj;
 
         the_ind->set_curr_loc(new_loc);

         if (the_ind->get_type() == OBJ_TYPE_PLAYER)
	 {
            the_player = (Player *) the_ind;
            tmp_gameflags = the_player->get_gameflags();

            tmp_node = inventory.store_current();
            new_loc->show_location(the_player, 
                                    tmp_gameflags->get_flag(GAMEFLAG_BRIEF));
            inventory.restore_current(tmp_node);
         }

      }
      tmp_obj = inventory.get_next();
   }
   return 1;
}


/***********************************************************************
 ** contains_players - contains players so clone it, don't lose it
 **
 ** Parameters: None
 **
 ** Returns:  1 if players found 
 **           0 if no players found
 **
 ***********************************************************************/
int Boat::contains_players(void)
{
   MudObject *tmp_obj;

   inventory.reset_current();
   tmp_obj = inventory.get_next();
   while (tmp_obj != NULL)
   {
      if (tmp_obj->get_type() == OBJ_TYPE_PLAYER)
      {
         return 1;
      }
      tmp_obj = inventory.get_next();
   }
   return 0;
}


/***********************************************************************
 ** get_brief - gets the brief description of the object, specialized for
 **             boats to take into account if they are on water
 **
 ** Parameters: none.
 **
 ** Returns:  1 if success
 **          -1 if failure
 **
 ***********************************************************************/
char *Boat::get_brief()
{   
   MudObject *tmp_obj;
   Location *tmp_loc;
   Flags    *tmp_locflags;

   tmp_obj = get_contained_by();
   while ((tmp_obj != NULL) && (tmp_obj->get_type() != OBJ_TYPE_LOCATION))
      tmp_obj = tmp_obj->get_contained_by();

   if (tmp_obj == NULL)
      return NULL;
 
   tmp_loc = (Location *) tmp_obj;
   tmp_locflags = tmp_loc->get_locflags();

   if (tmp_locflags->get_flag(LOCFLAG_AQUATIC))
      return water_brief.str_show();
   else if (tmp_locflags->get_flag(LOCFLAG_SHORE))
      return brief_desc0.str_show();
   else  
      return brief_desc1.str_show();
}


/***********************************************************************
 ** get_mem_size - gets how much memory this special is taking up
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Boat::get_mem_size()
{
   int size = 0;

   size = sizeof(this);
   size += get_mem_size_dynamic();
   return size;
}

/***********************************************************************
 ** get_mem_size_dynamic - gets how much memory is taken up by pointers
 **                        pointing to other objects, not including the
 **                        sizeof(this)
 **
 ** Returns: mem size in bytes
 **
 ***********************************************************************/

int Boat::get_mem_size_dynamic()
{
   int  size = 0;
   
   size += move_str.get_mem_size_dynamic();
   size += water_brief.get_mem_size_dynamic();
   size += get_mem_size_moveable();
   size += get_mem_size_item();
   size += get_mem_size_mudobj();
   size += get_mem_size_entity();

   return size;
}


#endif









