/*******************************************************************************
 *  PROJECT: GNOME Colorscheme
 *
 *  AUTHOR: Jonathon Jongsma
 *
 *  Copyright (c) 2005 Jonathon Jongsma
 *
 *  License:
 *    This program is free software; you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation; either version 2 of the License, or
 *    (at your option) any later version.
 *
 *    This program is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this program; if not, write to the 
 *    Free Software Foundation, Inc., 59 Temple Place, Suite 330, 
 *    Boston, MA  02111-1307  USA
 *
 *******************************************************************************/

#ifndef __GCS_TEXTSWATCH_WIDGET_H
#define __GCS_TEXTSWATCH_WIDGET_H

#include <memory>   // for std::auto_ptr
#include <gtkmm/uimanager.h>
#include <gtkmm/menu.h>
#include <gdkmm/colormap.h>
#include <gdkmm/window.h>
#include "core/gcs-types.h"
#include "core/gcs-color.h"
#include "gcs-colorswatch.h"

namespace gcs
{
    namespace Widgets
    {
        /** A widget which displays a colorswatch with a text description
         */
        class TextSwatch : public ColorSwatch
        {
            public:
                TextSwatch(ColorPtr bg);
                TextSwatch(const TextSwatch& c);
                virtual ~TextSwatch(void);

                virtual void set_color(ColorPtr bg);
                virtual bool on_button_press_event(GdkEventButton *e);

            protected:
                /** Text to be displayed on a ColorSwatch
                 *
                 * This is the text vthat displays the hex string, RGB, and HSV
                 * values of the color that is currently being represented by
                 * the colorswatch
                 */
                class SwatchText
                {
                    public:
                        SwatchText(Glib::RefPtr<Pango::Context> context, 
                                Glib::ustring txt = "");
                        virtual ~SwatchText(void);
                        inline Glib::RefPtr<Pango::Layout> get_layout(void)
                        { return m_refLayout;}

                        /** Sets the text to be displayed on the colorswatch 
                         */
                        void set_text(Glib::ustring text);

                        inline int get_width(void) {return m_width;}
                        inline int get_height(void) {return m_height;}

                    protected:
                        /** Updates the internal values of height and width
                         * whenever the text is changed, etc.
                         */
                        void update_extents(void);

                    private:
                        Glib::RefPtr<Pango::Layout> m_refLayout;
                        int m_width;
                        int m_height;
                };

                /** Initialize stuff when the object is realized.  Before
                 * realization (i.e. in the constructor), we can't call
                 * get_window() or anything because the Gdk::Window doesn't yet
                 * exist
                 */
                virtual void on_realize(void);
                /* Actions to take whenever the drawing area needs to be redrawn
                 * (i.e. after return from being minimized or being brought in
                 * front of another window.  This is where all of the drawing takes
                 * place.
                 */
                virtual bool on_expose_event(GdkEventExpose *e);
                /** Allocates the background color that will be displayed in
                 * the ColorSwatch
                 */
                virtual void redraw(void);
                /** Handles the action of pressing a mouse button on a
                 * ColorSwatch object.  Will pop up a context menu if one is
                 * defined, else will do nothing
                 */
                /** Handler for 'copy' item in the popup menu */
                virtual void on_action_copy(void);

            private:
                /** The graphics context for the ColorSwatch
                */
                Glib::RefPtr<Gdk::GC> m_textGC;
                /** The colormap of the ColorSwatch
                */
                Glib::RefPtr<Pango::Context> m_refContext;
                Glib::RefPtr<Gtk::UIManager> m_refUIManager;
                std::auto_ptr<SwatchText> m_pMinText;
                std::auto_ptr<SwatchText> m_pMaxText;
                /** The foreground color of the ColorSwatch
                */
                ColorPtr m_foreground;

                /** The right-click context menu for the colorswatch
                 */
                Gtk::Menu *m_pContextMenu;

                /** Called when the clipboard requests the contents that were
                 * copied
                 */
                void on_clipboard_get(Gtk::SelectionData& data, guint info);
                /** Called to clear the clipboard buffer
                 */
                void on_clipboard_clear(void);
                /** stores values that have been copied to the clipbard
                 */
                Glib::ustring m_clipboardBuffer;
        };

    } // namespace Widgets
} // namespace gcs

#endif // __GCS_TEXTSWATCH_WIDGET_H
