// ---------------------------------------------------------------------------
// - Xaas.cpp                                                                -
// - afnix:csm service - all as a service class implementation               -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2016 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Xaas.hpp"
#include "Vector.hpp"
#include "Boolean.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {
 
  // -------------------------------------------------------------------------
  // - private section                                                       -
  // -------------------------------------------------------------------------

  // the default rid
  static const String XAAS_XRID_XDEF = "default@localdomain";
  // the default active mode
  static const bool   XAAS_MODE_XDEF = true;

  // the rid property name
  static const String XAAS_XRID_NAME = "XAAS-RID";
  static const String XAAS_XRID_INFO = "XAAS REGISTRATION ID";

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create a default service

  Xaas::Xaas (void) {
    d_mode = XAAS_MODE_XDEF;
    setrid (XAAS_XRID_XDEF);
  }

  // create a service by name

  Xaas::Xaas (const String& name) : Plist (name) {
    d_mode = XAAS_MODE_XDEF;
    setrid (XAAS_XRID_XDEF);
  }

  // create a service by name and info

  Xaas::Xaas (const String& name, const String& info) : Plist (name, info) {
    d_mode = XAAS_MODE_XDEF;
    setrid (XAAS_XRID_XDEF);
  }

  // create a service by plist

  Xaas::Xaas (const Plist& plst) : Plist (plst) {
    d_mode = XAAS_MODE_XDEF;
    if (plst.exists (XAAS_XRID_NAME) == false) setrid (XAAS_XRID_XDEF);
  }

  // copy construct this account

  Xaas::Xaas (const Xaas& that) {
    that.rdlock ();
    try {
      // copy the base list
      Plist::operator = (that);
      // copy locally
      d_mode = that.d_mode;
      that.unlock ();
    } catch (...) {
      that.unlock ();
      throw;
    }
  }

  // assign a service to this one

  Xaas& Xaas::operator = (const Xaas& that) {
    // check for self-assignation
    if (this == &that) return *this;
    // lock and assign
    wrlock ();
    that.rdlock ();
    try {
      // copy the base list
      Plist::operator = (that);
      // copy locally
      d_mode = that.d_mode;
      unlock ();
      that.unlock ();
      return *this;
    } catch (...) {
      unlock ();
      that.unlock ();
      throw;
    }
  }
 
  // set the registration id

  void Xaas::setrid (const String& rid) {
    wrlock ();
    try {
      if (Plist::exists (XAAS_XRID_NAME) == true) {
	Plist::set (XAAS_XRID_NAME, rid);
      } else {
	Plist::add (XAAS_XRID_NAME, XAAS_XRID_INFO, rid);
      }
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the registration id

  String Xaas::getrid (void) const {
    rdlock ();
    try {
      String result = Plist::getpval (XAAS_XRID_NAME);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // set the service mode

  void Xaas::setmode (const bool mode) {
    wrlock ();
    try {
      d_mode = mode;
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the service port

  bool Xaas::getmode (void) const {
    rdlock ();
    try {
      bool result = d_mode;
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------
  
  // the quark zone
  static const long QUARK_ZONE_LENGTH = 4;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_SETRID  = zone.intern ("set-rid");
  static const long QUARK_GETRID  = zone.intern ("get-rid");
  static const long QUARK_SETMODE = zone.intern ("set-mode");
  static const long QUARK_GETMODE = zone.intern ("get-mode");

  // return true if the given quark is defined

  bool Xaas::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Plist::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* Xaas::apply (Runnable* robj, Nameset* nset, const long quark,
		       Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // dispatch 0 argument
    if (argc == 0) {
      if (quark == QUARK_GETRID)  return new String  (getrid  ());
      if (quark == QUARK_GETMODE) return new Boolean (getmode ());
    }
    if (argc == 1) {
      if (quark == QUARK_SETRID) {
	String rid = argv->getstring (0);
	setrid (rid);
	return nilp;
      }
      if (quark == QUARK_SETMODE) {
	long mode = argv->getbool (0);
	setmode (mode);
	return nilp;
      }
    }
    // call the plist methods
    return Plist::apply (robj, nset, quark, argv);
  }
}
