// ---------------------------------------------------------------------------
// - CartSet.cpp                                                             -
// - afnix:csm service - cart set class implementation                       -
// ---------------------------------------------------------------------------
// - This program is free software;  you can redistribute it  and/or  modify -
// - it provided that this copyright notice is kept intact.                  -
// -                                                                         -
// - This program  is  distributed in  the hope  that it will be useful, but -
// - without  any  warranty;  without  even   the   implied    warranty   of -
// - merchantability or fitness for a particular purpose.  In no event shall -
// - the copyright holder be liable for any  direct, indirect, incidental or -
// - special damages arising in any way out of the use of this software.     -
// ---------------------------------------------------------------------------
// - copyright (c) 1999-2016 amaury darsch                                   -
// ---------------------------------------------------------------------------

#include "Vector.hpp"
#include "Boolean.hpp"
#include "Integer.hpp"
#include "CartSet.hpp"
#include "Runnable.hpp"
#include "QuarkZone.hpp"
#include "Exception.hpp"

namespace afnix {

  // -------------------------------------------------------------------------
  // - class section                                                         -
  // -------------------------------------------------------------------------

  // create an empty cart set

  CartSet::CartSet (void) {
    d_acf = false;
    reset ();
  }

  // create an anonymous cart set

  CartSet::CartSet (const bool acf) {
    d_acf = acf;
    reset ();
  }

  // return the object class name

  String CartSet::repr (void) const {
    return "CartSet";
  }

  // reset this cart
  
  void CartSet::reset (void) {
    wrlock ();
    try {
      d_lht.reset ();
      d_aco.reset ();
      unlock ();
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get the number of cart in the cart set

  long CartSet::length (void) const {
    rdlock ();
    try {
      long result = d_lht.length ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }
	
  // return true if the cart set is empty

  bool CartSet::empty (void) const {
    rdlock ();
    try {
      bool result = d_lht.empty ();
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // check if a cart exists by rid

  bool CartSet::exists (const String& rid) const {
    rdlock ();
    try {
      bool result = d_lht.exists (rid);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a cart by index

  Cart* CartSet::get (const long idx) const {
    rdlock ();
    try {
      Cart* lco = dynamic_cast <Cart*> (d_lht.getobj (idx));
      unlock ();
      return lco;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a cart by rid

  Cart* CartSet::lookup (const String& rid) const {
    rdlock ();
    try {
      Cart* lco = dynamic_cast <Cart*> (d_lht.lookup (rid));
      unlock ();
      return lco;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // add a rco in the cart set

  t_long CartSet::addrco (const Rco& rco) {
    wrlock ();
    try {
      // get the rco rid
      String rid = rco.getrid ();
      if (rid.isnil () == true) {
	// check for valid anonymous
	if (d_acf == false) {
	  throw Exception ("cart set-error", "invalid anonymous rco to add");
	}
	// add the rco and return
	t_long result = d_aco.addrco (rco);
	unlock ();
	return result;
      }
      // check if the cart exists in the hashtable
      Cart* lco = dynamic_cast <Cart*> (d_lht.get (rid));
      if (lco == nilp) {
	lco = new Cart (rid);
	d_lht.add (rid, lco);
      }
      // add the rco in the cart
      t_long result = lco->addrco (rco);
      unlock ();
      return result;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // get a rco by rid and index

  Rco CartSet::getrco (const String& rid, const t_long kid) const {
    rdlock ();
    try {
      // check for anonymous
      if (rid.isnil () == true) {
	// check for valid anonymous
	if (d_acf == false) {
	  throw Exception ("cart set-error", "invalid anonymous rco query");
	}
	Rco rco = d_aco.getrco (kid);
	unlock ();
	return rco;
      }
      // find the cart by rid
      Cart* lco = dynamic_cast <Cart*> (d_lht.lookup (rid));
      if (lco == nilp) {
	throw Exception ("internal-error", "invalid nil cart in lookup");
      }
      // get the rco by index
      Rco rco = lco->getrco (kid);
      unlock ();
      return rco;
    } catch (...) {
      unlock ();
      throw;
    }
  }

  // -------------------------------------------------------------------------
  // - object section                                                        -
  // -------------------------------------------------------------------------

  // the quark zone
  static const long QUARK_ZONE_LENGTH = 6;
  static QuarkZone  zone (QUARK_ZONE_LENGTH);

  // the object supported quarks
  static const long QUARK_GET     = zone.intern ("get");
  static const long QUARK_ADDRCO  = zone.intern ("add-rco");
  static const long QUARK_GETRCO  = zone.intern ("get-rco");
  static const long QUARK_LOOKUP  = zone.intern ("lookup");
  static const long QUARK_LENGTH  = zone.intern ("length");
  static const long QUARK_EMPTYP  = zone.intern ("empty-p");
  static const long QUARK_EXISTSP = zone.intern ("exists-p");

  // create a new object in a generic way

  Object* CartSet::mknew (Vector* argv) {
    long argc = (argv == nilp) ? 0 : argv->length ();
    // check for 0 argument
    if (argc == 0) return new CartSet;
    // check for 1 argument
    if (argc == 1) {
      bool acf = argv->getbool (0);
      return new CartSet (acf);
    }
    throw Exception ("argument-error",
                     "too many argument with cart set constructor");
  }

  // return true if the given quark is defined

  bool CartSet::isquark (const long quark, const bool hflg) const {
    rdlock ();
    if (zone.exists (quark) == true) {
      unlock ();
      return true;
    }
    bool result = hflg ? Object::isquark (quark, hflg) : false;
    unlock ();
    return result;
  }
  
  // apply this object with a set of arguments and a quark
  
  Object* CartSet::apply (Runnable* robj, Nameset* nset, const long quark,
			  Vector* argv) {
    // get the number of arguments
    long argc = (argv == nilp) ? 0 : argv->length ();
    
    // check for 0 argument
    if (quark == QUARK_EMPTYP) return new Boolean (empty  ());
    if (quark == QUARK_LENGTH) return new Integer (length ());
    // check for 1 argument
    if (argc == 1) {
      if (quark == QUARK_ADDRCO) {
	Object* obj = argv->get (0);
	Rco*    rco = dynamic_cast <Rco*> (obj);
	if (rco == nilp) {
	  throw Exception ("type-error", "invalid object with cart set add",
			   Object::repr (obj));
	}
	t_long result = addrco (*rco);
	return new Integer (result);
      }
      if (quark == QUARK_GET) {
	rdlock ();
	try {
	  long idx = argv->getlong (0);
	  Cart* lco = get (idx);
	  robj->post (lco);
	  unlock ();
	  return lco;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_LOOKUP) {
	rdlock ();
	try {
	  String rid = argv->getstring (0);
	  Cart* lco = lookup (rid);
	  robj->post (lco);
	  unlock ();
	  return lco;
	} catch (...) {
	  unlock ();
	  throw;
	}
      }
      if (quark == QUARK_EXISTSP) {
	String rid = argv->getstring (0);
	return new Boolean (exists (rid));
      }
    }
    // check for 2 arguments
    if (argc == 2) {
      if (quark == QUARK_GETRCO) {
	String rid = argv->getstring (0);
	t_long kid = argv->getlong (1);
	return new Rco (getrco (rid, kid));
      }
    }
    // call the object method
    return Object::apply (robj, nset, quark, argv);
  }
}
