/*
 * This file is part of Magellan <http://www.kAlliance.org/Magellan>
 *
 * Copyright (c) 1998-2000 Teodor Mihai <teddy@ireland.com>
 * Copyright (c) 1998-2000 Laur Ivan <laur.ivan@ul.ie>
 * Copyright (c) 1999-2000 Virgil Palanciuc <vv@ulise.cs.pub.ro>
 *
 * Requires the Qt widget libraries, available at no cost at
 * http://www.troll.no/
 *
 * Also requires the KDE libraries, available at no cost at
 * http://www.kde.org/
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell 
 * copies of the Software, and to permit persons to whom the Software is 
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in 
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE 
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS 
 * IN THE SOFTWARE.
 */

#include <smtphandler.h>
#include <indexclass.h>
#include <outgoingmessage.h>
#include <qsocketnotifier.h>
#include <unistd.h>
#include <iosynchandler.h>
#include <stdio.h>
#include <orb.h>
#include <localmailfolder.h>
#include <messagedescriptor.h>
#include <messagedevice.h>
#include <accountmanager.h>
#include <servernotifier.h>

SMTPHandler *SMTPHandler::thisInstance;
sem_t SMTPHandler::threadLock;
sem_t SMTPHandler::msgLock;
int SMTPHandler::statusSync[2];
queue<OutgoingMessage *> SMTPHandler::msgQueue;
queue<IndexClass *> SMTPHandler::idxQueue;
string SMTPHandler::status;

SMTPHandler *SMTPHandler::ref()
{
	if( !thisInstance)
	{
	  thisInstance=new SMTPHandler();
	  assert( thisInstance->init() );
	}
	return thisInstance;
}

SMTPHandler::SMTPHandler()
{
//nothing
}

bool SMTPHandler::init()
{
	if( ::pipe(statusSync) )
	  return false;
	
	if( sem_init(&threadLock, 0, 0) ||
	    sem_init(&msgLock, 0, 0) )
	  return false;
	
	QSocketNotifier *statusNotifier=new QSocketNotifier(statusSync[0], QSocketNotifier::Read, this);
	connect(statusNotifier, SIGNAL(activated(int)), this, SLOT(__ackStatus(int)));
	
	pthread_t thread;
	pthread_attr_t attr;
	pthread_attr_init(&attr);
	pthread_attr_setdetachstate(&attr, PTHREAD_CREATE_DETACHED);
	pthread_create(&thread, &attr, smtp_thread, 0);
	
	currentMessage=0;
	totalMessages=0;
	return true;
}

void SMTPHandler::startSend(Account *account)
{
	QString accountName=account->accname;
	MailFolder *outbox=ObjectRequestBroker::thisInstance()->mailfolderReference("sysmail_outbox");
	
	if(!outbox)
	{
		printf("A fatal error has occured, outbox interface not found. Terminated.\n");
		exit(1);
	}
	
	// get listing
	QStringList messages=outbox->getEntryList();
	
	if(messages.count()) ServerNotifier::thisInstance()->ioProgress(QString("smtp:task:")+account->accname);
	
	for(unsigned i=0;i<messages.count();i++)
	{
		IndexClass *index=outbox->indexIDict()[messages[i]];
		
		MessageDevice *dev=new MessageDevice(index);
		dev->loadDescriptor(false);
		
		Account *acc=AccountManager::ref()->getAccount(dev->getDescriptor().account);
		if(!acc) acc=AccountManager::ref()->getDefaultAccount();
		
		delete dev;
		
		if(!index->isLocked() && acc) send(index, acc);
	}
}

void SMTPHandler::send(IndexClass *idx, Account *account)
{
	// debug
	printf("smtphandler: trying to send message %s...\n", (const char *)idx->getID());

	// debug
	printf("smtphandler: locking message...\n");
		
	idx->lock();
	
	// debug
	printf("smtphandler: creating outgoing structure...\n");
	
	OutgoingMessage *m=new OutgoingMessage(idx, account);
	
	// debug
	printf("smtphandler: queueing message...\n");
	
	msgQueue.push(m);
	idxQueue.push(idx);

	// debug
	printf("smtphandler: computing new progress indication...\n");
		
	// signal progress indication
	int sval;
	sem_getvalue(&msgLock, &sval);
	totalMessages++;
	if(!sval) currentMessage++;
	
	QString progressMessage;
	progressMessage=QString("smtp:sending:%1:%2").arg(currentMessage).arg(totalMessages);
		
	ServerNotifier::thisInstance()->ioProgress(progressMessage);
	// debug
	printf("smtphandler: signaling thread about a new message...\n");
	
	// signal thread that a new message is to be sent
	sem_post(&msgLock);
}

void SMTPHandler::__ackStatus(int)
{
	// empty sync pipe
	char buf;
	read(statusSync[0], &buf, 1);

	IndexClass *lastIdx=idxQueue.front();
	idxQueue.pop();
	
	// debug
	printf("smtphandler: status update received: %s\n", status.c_str());
	
	// send status update to the IOSyncHandler
	lastIdx->unlock();
	if(status=="sent")
	{
		IOSyncHandler::ref()->unlinkOutgoing(lastIdx);
	}
	else
	{
		IOSyncHandler::ref()->updateOutgoing(lastIdx, status);
		ServerNotifier::thisInstance()->ioProgress(QString("smtp:")+status.c_str());
	}
	
	// signal progress indication
	int sval;
	sem_getvalue(&msgLock, &sval);
	if(!sval) currentMessage=totalMessages=0;
	else currentMessage++;
	
	// send info to the notifier
	QString progressMessage;
	if(totalMessages) progressMessage=QString("smtp:sent:%1:%2").arg(currentMessage).arg(totalMessages);
	else progressMessage="smtp:done";
		
	if(status=="sent") ServerNotifier::thisInstance()->ioProgress(progressMessage);
	
	// unlock thread
	sem_post(&threadLock);
}
