/* 
   Project: UL

   Copyright (C) 2007 Michael Johnston & Jordi Villa-Freixa

   Author: Michael Johnston

   This application is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
 
   This application is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.
 
   You should have received a copy of the GNU General Public
   License along with this library; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111 USA.
*/
#include "ULExportController.h"
#include <MolTalk/MolTalk.h>

id sharedExportController = nil;

/**
Category containing method wrappers around ULFramework functions
*/
@interface ULExportController (PrivateExportWrappers)
- (id) exportDataSetAsCSV: (id) anObject toFile: (NSString*) filename;
- (id) exportDataSourceAsPDB: (id) anObject toFile: (NSString*) filename;
- (id) exportTemplateAsPropertyList: (id) template toFile: (NSString*) fileName;
@end

@implementation ULExportController

+ (id) sharedExportController
{
	if(sharedExportController == nil)
		sharedExportController = [self new];

	return sharedExportController;
}

- (id) init
{
	if(sharedExportController != nil)
		return sharedExportController;

	if(self = [super init])
	{
		exportPanel = [ULExportPanel exportPanel];
		pasteboard = [ULPasteboard appPasteboard];
		sharedExportController = self;
		knownFormats = [NSDictionary dictionaryWithObjectsAndKeys:
				[NSArray arrayWithObjects: 
					@"pdb",
					@"binary archive",
					nil],
				@"AdDataSource",
				[NSArray arrayWithObjects: 
					@"csv",
					@"binary archive",
					nil],
				@"AdDataSet",
				[NSArray arrayWithObject: @"binary archive"],
				@"AdSimulationData",
				[NSArray arrayWithObjects: 
					@"AdunCore template",
					@"binary archive",
					nil],
				@"ULTemplate", nil];
		[knownFormats retain];		
		displayStrings = [NSDictionary dictionaryWithObjectsAndKeys:
				@"System", @"AdDataSource",
				@"Simulation", @"AdSimulationData",
				@"DataSet", @"AdDataSet",
				@"Template", @"ULTemplate", nil];
		[displayStrings retain];	
		exportMethods = [NSMutableDictionary new];
		[exportMethods setObject: 
			[NSDictionary dictionaryWithObjectsAndKeys:
				NSStringFromSelector(@selector(exportDataSetAsCSV:toFile:)),
				@"csv", nil]	
			forKey: @"AdDataSet"];	
		[exportMethods setObject: 
			[NSDictionary dictionaryWithObjectsAndKeys:
				NSStringFromSelector(@selector(exportDataSourceAsPDB:toFile:)),
				@"pdb", nil]	
			forKey: @"AdDataSource"];	
		[exportMethods setObject: 
			[NSDictionary dictionaryWithObjectsAndKeys:
				NSStringFromSelector(@selector(exportTemplateAsPropertyList:toFile:)),
				@"AdunCore template", nil]	
			forKey: @"ULTemplate"];	

	}

	return self;
}

- (void) dealloc
{
	[knownFormats release];
	[displayStrings release];
	[exportMethods release];
	[super dealloc];
}

- (void) _exportObjectAsBinaryArchive: (id) anObject toFile: (NSString*) filename
{
	BOOL retVal;
	int selectedRow;
	NSError *error;
	NSMutableData *data;
	NSKeyedArchiver* archiver;
	NSString* storagePath, *destinationPath;

	//If its a simulation we also have to export the simulation
	//data directory. This involves copying the directory to the
	//chosen location. 	
	//FIXME: We should define path extensions for the different types 
	//of adun objects.
	
	if([anObject isKindOfClass: [AdSimulationData class]])
	{
		storagePath = [[anObject dataStorage] storagePath];
		destinationPath =  [filename stringByAppendingString: @"_Data"];
		retVal = [[NSFileManager defaultManager]
				copyPath: storagePath
				toPath: destinationPath
				handler: nil];

		if(!retVal)
		{
			//Abort
			NSRunAlertPanel(@"Error",
				@"Unable to extract simulation data - Aborting",
				@"Dismiss", 
				nil,
				nil);

			return;
		}	
	}
	data = [NSMutableData new];
	archiver = [[NSKeyedArchiver alloc] initForWritingWithMutableData: data];
	[archiver setOutputFormat: NSPropertyListXMLFormat_v1_0];
	[archiver encodeObject: anObject forKey: @"root"];
	[archiver finishEncoding];
	[archiver release];
	
	retVal = [[ULIOManager appIOManager]
			writeObject: data 
			toFile: filename 
			error: &error];

	[data release];
	if(!retVal)
	{
		NSRunAlertPanel(@"Error",
			[[error userInfo] objectForKey:NSLocalizedDescriptionKey],
			@"Dismiss", 
			nil,
			nil);
	}
}

- (void) export: (id) sender
{
	int result, index;
	id exportObject; //the actual unarchived object
	id savePanel;
	NSString* filename, *format, *type, *extension;

	//Get the type of the object to be exported
	
	type = [[pasteboard availableTypes] objectAtIndex: 0];
	[exportPanel setObjectType: [displayStrings objectForKey: type]];
	[exportPanel setChoices: [knownFormats objectForKey: type]];
	result = [exportPanel runModal];
	format = [exportPanel choice];

	if(result == NSOKButton)
	{
		savePanel = [NSSavePanel savePanel];	
		if([[exportPanel choice] isEqual: @"pdb"])
			[savePanel setRequiredFileType: @"pdb"];
			
		[savePanel setTitle: @"Export Data"];
		[savePanel setDirectory: NSHomeDirectory()];
		result = [savePanel runModal];
		filename = [savePanel filename];

		if(result == NSOKButton)
		{
			//Take any extension off filename 
			//This just makes it easier to work with it later
			extension = [filename pathExtension];
			exportObject = [pasteboard objectForType: type];
			[self exportObject: exportObject 
				toFile: filename
				format: format];
		}		
	}			
}

- (void) exportObject: (id) anObject toFile: (NSString*) filename format: (NSString*) format 
{
	SEL selector;
	NSString* selectorString;
	
	if([format isEqual: @"binary archive"])
		[self _exportObjectAsBinaryArchive: anObject toFile: filename];
	else
	{
		selectorString = [[exportMethods objectForKey: 
						NSStringFromClass([anObject class])]
					objectForKey: format];
		[self performSelector: NSSelectorFromString(selectorString)
			withObject: anObject
			withObject: filename];		
	}
}

@end

@implementation ULExportController (PrivateExportWrappers)

- (id) exportDataSetAsCSV: (id) dataSet toFile: (NSString*) filename
{
	int i;
	NSArray* dataMatrices;
	AdDataMatrix* matrix;
	id name, cvsString;

	dataMatrices = [dataSet dataMatrices];
	//Remove any path extension specified so we can add the filenumber.
	filename = [filename stringByDeletingPathExtension];
	for(i=0; i<(int)[dataMatrices count];i++)
	{
		name = [NSString stringWithFormat: @"%@%d.csv", filename, i+1];
		matrix = [dataMatrices objectAtIndex: i];
		cvsString = [matrix stringRepresentation];
		[cvsString writeToFile: name atomically: NO];
	}

	return nil;
}

- (id) exportDataSourceAsPDB: (id) dataSource toFile: (NSString*) filename
{
	id structure;
	MTFileStream* fileStream;

 	structure = ULConvertDataSourceToPDBStructure(dataSource);		
	fileStream = [MTFileStream streamToFile: 
			[NSString stringWithFormat: @"%@", filename]];
	[structure writePDBToStream: fileStream];
	[fileStream close];

	return nil;
}

- (id) exportTemplateAsPropertyList: (id) template toFile: (NSString*) filename
{
	[[template coreRepresentation] writeToFile: filename
		atomically: NO];

	return nil;	
}

@end

