# Analysis report and its manipulation utility.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/message"
require "adlint/code"
require "adlint/metric"

module AdLint #:nodoc:

  # == DESCRIPTION
  # Report information collector.
  class Report
    # === DESCRIPTION
    # Constructs an empty report.
    def initialize(msg_fpath, met_fpath, &block)
      @msg_fpath = msg_fpath
      @msg_file = open_message_file(msg_fpath)

      @met_fpath = met_fpath
      @met_file = open_metric_file(met_fpath)

      @unique_messages = Set.new

      yield(self)
    ensure
      @msg_file.close if @msg_file
      @met_file.close if @met_file
    end

    attr_reader :msg_fpath
    attr_reader :met_fpath

    # === DESCRIPTION
    # Adds a message to this report.
    #
    # === PARAMETER
    # _message_:: Message -- Message to be added.
    #
    # === RETURN VALUE
    # Report -- Self.
    def add_message(message)
      @msg_file.puts(message.to_csv)
      $stderr.puts(message.to_s) unless verbose?
      self
    end

    def add_unique_message(message)
      if @unique_messages.add?(message)
        add_message(message)
      end
      self
    end

    # === DESCRIPTION
    # Adds code structure information to this report.
    #
    # === PARAMETER
    # _code_struct_:: CodeStructure -- Code structure information to be added.
    #
    # === RETURN VALUE
    # Report -- Self.
    def add_code_struct(code_struct)
      @met_file.puts(code_struct.to_csv)
      self
    end

    def add_code_metric(code_metric)
      @met_file.puts(code_metric.to_csv)
      self
    end

    private
    def open_message_file(fpath)
      File.open(fpath, "w").tap do |io|
        io.set_encoding(Encoding.default_external)
        io.puts(["V", SHORT_VERSION, Time.now.to_s, Dir.getwd].to_csv)
      end
    end

    def open_metric_file(fpath)
      File.open(fpath, "w").tap do |io|
        io.set_encoding(Encoding.default_external)
        io.puts(["VER", SHORT_VERSION, Time.now.to_s, Dir.getwd].to_csv)
      end
    end
  end

  module MessageUniqueness
    @@messages_to_be_unique = Set.new

    def ensure_uniqueness_of(message_id)
      @@messages_to_be_unique.add(message_id)
      nil
    end

    def must_be_unique?(message_id)
      @@messages_to_be_unique.include?(message_id)
    end
    module_function :must_be_unique?
  end

  # == DESCRIPTION
  # Report manipulation utility.
  module ReportUtil
    # === DESCRIPTION
    # Adds an error message to the report.
    #
    # Abbreviation below is available.
    #  add_error_message(id, location, ...) => E(id, location, ...)
    #
    # === PARAMETER
    # _id_:: String -- Message id string.
    # _location_:: Location -- Location where the message detected.
    # _parts_:: Array< Object > -- Message formatting parts.
    #
    # === RETURN VALUE
    # None.
    def add_error_message(id, location, *parts)
      add_message(ErrorMessage.new(id, location, *parts))
    end
    alias :E :add_error_message
    module_function :add_error_message, :E

    # === DESCRIPTION
    # Adds a warning message to the report.
    #
    # Abbreviation below is available.
    #  add_warning_message(id, location, ...) => W(id, location, ...)
    #
    # === PARAMETER
    # _id_:: String -- Message id string.
    # _location_:: Location -- Location where the message detected.
    # _parts_:: Array< Object > -- Message formatting parts.
    #
    # === RETURN VALUE
    # None.
    def add_warning_message(id, location, *parts)
      add_message(WarningMessage.new(id, location, *parts))
    end
    alias :W :add_warning_message
    module_function :add_warning_message, :W

    # === DESCRIPTION
    # Adds a context message to the report.
    #
    # Abbreviation below is available.
    #  add_context_message(id, location, ...) => C(id, location, ...)
    #
    # === PARAMETER
    # _id_:: String -- Message id string.
    # _location_:: Location -- Location where the message detected.
    # _parts_:: Array< Object > -- Message formatting parts.
    #
    # === RETURN VALUE
    # None.
    def add_context_message(id, location, *parts)
      add_message(ContextMessage.new(id, location, *parts))
    end
    alias :C :add_context_message
    module_function :add_context_message, :C

    # === DESCRIPTION
    # Adds a message to the report.
    #
    # === PARAMETER
    # _message_:: Message -- Message to be added.
    #
    # === RETURN VALUE
    # None.
    def add_message(message)
      if MessageUniqueness.must_be_unique?(message.id)
        report.add_unique_message(message)
      else
        report.add_message(message)
      end
      nil
    end
    module_function :add_message

    # === DESCRIPTION
    # Adds type declaration information to the report.
    #
    # Abbreviation below is available.
    #  add_typedcl(...) => TYPEDCL(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _typedcl_type_:: String -- Type declaration type.
    # _type_name_:: String -- Type name.
    # _type_rep_:: String -- Type representation string.
    #
    # === RETURN VALUE
    # None.
    def add_typedcl(location, typedcl_type, type_name, type_rep)
      add_code_struct(TypeDcl.new(location, typedcl_type, type_name, type_rep))
    end
    alias :TYPEDCL :add_typedcl
    module_function :add_typedcl, :TYPEDCL

    # === DESCRIPTION
    # Adds global variable decaration information to the report.
    #
    # Abbreviation below is available.
    #  add_gvardcl(...) => GVARDCL(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _variable_name_:: String -- Global variable name.
    # _type_rep_:: String -- Type representation string.
    #
    # === RETURN VALUE
    # None.
    def add_gvardcl(location, variable_name, type_rep)
      add_code_struct(GVarDcl.new(location, variable_name, type_rep))
    end
    alias :GVARDCL :add_gvardcl
    module_function :add_gvardcl, :GVARDCL

    # === DESCRIPTION
    # Adds function declaration information to the report.
    #
    # Abbreviation below is available.
    #  add_funcdcl(...) => FUNCDCL(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the declaration appears.
    # _function_linkage_type_:: String -- Function linkage type string.
    # _function_scope_type_:: String -- Declaration scope type string.
    # _function_identifier_:: String -- Function identifier.
    #
    # === RETURN VALUE
    # None.
    def add_funcdcl(location, function_linkage_type, function_scope_type,
                    function_identifier)
      add_code_struct(FuncDcl.new(location, function_linkage_type,
                                  function_scope_type, function_identifier))
    end
    alias :FUNCDCL :add_funcdcl
    module_function :add_funcdcl, :FUNCDCL

    # === DESCRIPTION
    # Adds variable definition information to the report.
    #
    # Abbreviation below is available.
    #  add_vardef(...) => VARDEF(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _var_linkage_type_:: String -- Variable linkage type string.
    # _var_scope_type_:: String -- Variable scope type string.
    # _storage_class_type_:: String -- Variable storage class type.
    # _variable_name_:: String -- Variable name.
    # _type_rep_:: String -- Variable type representation string.
    #
    # === RETURN VALUE
    # None.
    def add_vardef(location, var_linkage_type, var_scope_type,
                   storage_class_type, variable_name, type_rep)
      add_code_struct(VarDef.new(location, var_linkage_type, var_scope_type,
                                 storage_class_type, variable_name, type_rep))
    end
    alias :VARDEF :add_vardef
    module_function :add_vardef, :VARDEF

    # === DESCRIPTION
    # Adds function definition information to the report.
    #
    # Abbreviation below is available.
    #  add_funcdef(...) => FUNCDEF(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _function_linkage_type_:: String -- Function linkage type string.
    # _function_scope_type_:: String -- Definition scope type string.
    # _function_identifier_:: String -- Function identifier.
    # _lines_:: Integer -- Physical lines.
    #
    # === RETURN VALUE
    # None.
    def add_funcdef(location, function_linkage_type, function_scope_type,
                    function_identifier, lines)
      add_code_struct(FuncDef.new(location, function_linkage_type,
                                  function_scope_type, function_identifier,
                                  lines))
    end
    alias :FUNCDEF :add_funcdef
    module_function :add_funcdef, :FUNCDEF

    # === DESCRIPTION
    # Adds macro definition information to the report.
    #
    # Abbreviation below is available.
    #  add_macrodef(...) => MACRODEF(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _macro_name_:: String -- Macro name.
    # _macro_type_:: String -- Macro type string.
    #
    # === RETURN VALUE
    # None.
    def add_macrodef(location, macro_name, macro_type)
      add_code_struct(MacroDef.new(location, macro_name, macro_type))
    end
    alias :MACRODEF :add_macrodef
    module_function :add_macrodef, :MACRODEF

    # === DESCRIPTION
    # Adds label definition information to the report.
    #
    # Abbreviation below is available.
    #  add_labeldef(...) => LABELDEF(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the definition appears.
    # _label_name_:: String -- Label name.
    #
    # === RETURN VALUE
    # None.
    def add_labeldef(location, label_name)
      add_code_struct(LabelDef.new(location, label_name))
    end
    alias :LABELDEF :add_labeldef
    module_function :add_labeldef, :LABELDEF

    # === DESCRIPTION
    # Adds initialization information to the report.
    #
    # Abbreviation below is available.
    #  add_initialization(...) => INITIALIZATION(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the variable appears.
    # _variable_name_:: String -- Initialized variable name.
    # _initializer_rep_:: String -- Initializer representation.
    #
    # === RETURN VALUE
    # None.
    def add_initialization(location, variable_name, initializer_rep)
      add_code_struct(Initialization.new(location, variable_name,
                                         initializer_rep))
    end
    alias :INITIALIZATION :add_initialization
    module_function :add_initialization, :INITIALIZATION

    # === DESCRIPTION
    # Adds assignment information to the report.
    #
    # Abbreviation below is available.
    #  add_assignment(...) => ASSIGNMENT(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the variable appears.
    # _variable_name_:: String -- Assigned variable name.
    # _assignment_rep_:: String -- Assignment expression representation.
    #
    # === RETURN VALUE
    # None.
    def add_assignment(location, variable_name, assignment_rep)
      add_code_struct(Assignment.new(location, variable_name,
                                     assignment_rep))
    end
    alias :ASSIGNMENT :add_assignment
    module_function :add_assignment, :ASSIGNMENT

    # === DESCRIPTION
    # Adds header include information to the report.
    #
    # Abbreviation below is available.
    #  add_include(...) => INCLUDE(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the directive appears.
    # _included_fpath_:: Pathname -- Path name of the included file.
    #
    # === RETURN VALUE
    # None.
    def add_include(location, included_fpath)
      add_code_struct(Include.new(location, included_fpath))
    end
    alias :INCLUDE :add_include
    module_function :add_include, :INCLUDE

    # === DESCRIPTION
    # Adds function call information to the report.
    #
    # Abbreviation below is available.
    #  add_call(...) => CALL(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the function call appears.
    # _caller_function_:: FunctionIdentifier -- Calling function identifier.
    # _callee_function_:: FunctionIdentifier -- Called function identifier.
    def add_call(location, caller_function, callee_function)
      add_code_struct(Call.new(location, caller_function, callee_function))
    end
    alias :CALL :add_call
    module_function :add_call, :CALL

    # === DESCRIPTION
    # Adds variable cross reference information to the report.
    #
    # Abbreviation below is available.
    #  add_xref_variable(...) => XREF_VAR(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the cross-ref appears.
    # _accessor_function_:: FunctionIdentifier -- Accessing function
    #                       identifier.
    # _access_type_:: String -- Access type string.
    # _accessee_variable_:: String -- Accessed variable name.
    #
    # === RETURN VALUE
    # None.
    def add_xref_variable(location, accessor_function, access_type,
                          accessee_variable)
      add_code_struct(XRefVar.new(location, accessor_function, access_type,
                                  accessee_variable))
    end
    alias :XREF_VAR :add_xref_variable
    module_function :add_xref_variable, :XREF_VAR

    # === DESCRIPTION
    # Adds function cross reference information to the report.
    #
    # Abbreviation below is available.
    #  add_xref_function(...) => XREF_FUNC(...)
    #
    # === PARAMETER
    # _location_:: Location -- Location where the cross-ref appears.
    # _accessor_function_:: FunctionIdentifier -- Accessing function
    #                       identifier.
    # _access_type_:: String -- Access type string.
    # _accessee_function_:: FunctionIdentifier -- Accessed function identifier.
    #
    # === RETURN VALUE
    # None.
    def add_xref_function(location, accessor_function, access_type,
                          accessee_function)
      add_code_struct(XRefFunc.new(location, accessor_function, access_type,
                                   accessee_function))
    end
    alias :XREF_FUNC :add_xref_function
    module_function :add_xref_function, :XREF_FUNC

    def add_literal(location, literal_type, literal_prefix, literal_suffix,
                    literal_value)
      add_code_struct(Literal.new(location, literal_type, literal_prefix,
                                  literal_suffix, literal_value))
    end
    alias :LIT :add_literal
    module_function :add_literal, :LIT

    def add_pp_directive(location, pp_directive, pp_tokens)
      add_code_struct(PPDirective.new(location, pp_directive, pp_tokens))
    end
    alias :PP_DIRECTIVE :add_pp_directive
    module_function :add_pp_directive, :PP_DIRECTIVE

    # === DESCRIPTION
    # Adds code structure information to the report.
    #
    # === PARAMETER
    # _code_struct_:: CodeStructure -- Code structure information to be added.
    #
    # === RETURN VALUE
    # None.
    def add_code_struct(code_struct)
      report.add_code_struct(code_struct)
      nil
    end
    module_function :add_code_struct

    def add_FL_STMT(fpath, statement_count)
      add_code_metric(FL_STMT_Metric.new(fpath, statement_count))
    end
    alias :FL_STMT :add_FL_STMT
    module_function :add_FL_STMT, :FL_STMT

    def add_FL_FUNC(fpath, function_count)
      add_code_metric(FL_FUNC_Metric.new(fpath, function_count))
    end
    alias :FL_FUNC :add_FL_FUNC
    module_function :add_FL_FUNC, :FL_FUNC

    def add_FN_STMT(function_identifier, location, statement_count)
      add_code_metric(FN_STMT_Metric.new(function_identifier, location,
                                         statement_count))
    end
    alias :FN_STMT :add_FN_STMT
    module_function :add_FN_STMT, :FN_STMT

    def add_FN_UNRC(function_identifier, location, unreached_statement_count)
      add_code_metric(FN_UNRC_Metric.new(function_identifier, location,
                                         unreached_statement_count))
    end
    alias :FN_UNRC :add_FN_UNRC
    module_function :add_FN_UNRC, :FN_UNRC

    def add_FN_LINE(function_identifier, location, total_lines)
      add_code_metric(FN_LINE_Metric.new(function_identifier, location,
                                         total_lines))
    end
    alias :FN_LINE :add_FN_LINE
    module_function :add_FN_LINE, :FN_LINE

    def add_FN_PARA(function_identifier, location, parameter_count)
      add_code_metric(FN_PARA_Metric.new(function_identifier, location,
                                         parameter_count))
    end
    alias :FN_PARA :add_FN_PARA
    module_function :add_FN_PARA, :FN_PARA

    def add_FN_UNUV(function_identifier, location, useless_variable_count)
      add_code_metric(FN_UNUV_Metric.new(function_identifier, location,
                                         useless_variable_count))
    end
    alias :FN_UNUV :add_FN_UNUV
    module_function :add_FN_UNUV, :FN_UNUV

    def add_FN_CSUB(function_identifier, location, function_call_count)
      add_code_metric(FN_CSUB_Metric.new(function_identifier, location,
                                         function_call_count))
    end
    alias :FN_CSUB :add_FN_CSUB
    module_function :add_FN_CSUB, :FN_CSUB

    def add_FN_GOTO(function_identifier, location, goto_count)
      add_code_metric(FN_GOTO_Metric.new(function_identifier, location,
                                         goto_count))
    end
    alias :FN_GOTO :add_FN_GOTO
    module_function :add_FN_GOTO, :FN_GOTO

    def add_FN_RETN(function_identifier, location, return_count)
      add_code_metric(FN_RETN_Metric.new(function_identifier, location,
                                         return_count))
    end
    alias :FN_RETN :add_FN_RETN
    module_function :add_FN_RETN, :FN_RETN

    def add_FN_UELS(function_identifier, location, if_statement_count)
      add_code_metric(FN_UELS_Metric.new(function_identifier, location,
                                         if_statement_count))
    end
    alias :FN_UELS :add_FN_UELS
    module_function :add_FN_UELS, :FN_UELS

    def add_FN_NEST(function_identifier, location, max_nest_level)
      add_code_metric(FN_NEST_Metric.new(function_identifier, location,
                                         max_nest_level))
    end
    alias :FN_NEST :add_FN_NEST
    module_function :add_FN_NEST, :FN_NEST

    def add_FN_PATH(function_identifier, location, path_count)
      add_code_metric(FN_PATH_Metric.new(function_identifier, location,
                                         path_count))
    end
    alias :FN_PATH :add_FN_PATH
    module_function :add_FN_PATH, :FN_PATH

    def add_FN_CYCM(function_identifier, location, cyclomatic_complexity)
      add_code_metric(FN_CYCM_Metric.new(function_identifier, location,
                                         cyclomatic_complexity))
    end
    alias :FN_CYCM :add_FN_CYCM
    module_function :add_FN_CYCM, :FN_CYCM

    def add_FN_CALL(function_signature, location, caller_count)
      add_code_metric(FN_CALL_Metric.new(function_signature, location,
                                         caller_count))
    end
    alias :FN_CALL :add_FN_CALL
    module_function :add_FN_CALL, :FN_CALL

    def add_code_metric(code_metric)
      report.add_code_metric(code_metric)
      nil
    end
    module_function :add_code_metric
  end

  CodeExtraction.class_eval { include ReportUtil }
  MessageDetection.class_eval { extend MessageUniqueness; include ReportUtil }
  MetricMeasurement.class_eval { include ReportUtil }

end
