# Phases of cross module analysis.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/phase"
require "adlint/monitor"
require "adlint/ld/object"
require "adlint/ld/typedef"
require "adlint/ld/message"
require "adlint/ld/metric"
require "adlint/ld/util"

module AdLint #:nodoc:
module Ld #:nodoc:

  class MapTypedefPhase < Phase
    private
    def do_execute(context)
      monitored_region("mtd") do |monitor|
        mapper = TypedefMapper.new
        context[:metric_fpaths].each do |fpath|
          mapper.execute(fpath)
          monitor.progress += 1.0 / context[:metric_fpaths].size
        end
        context[:ld_typedef_mapping] = mapper.result
      end
    end
  end

  class MapFunctionPhase < Phase
    private
    def do_execute(context)
      monitored_region("mfn") do |monitor|
        mapper = FunctionMapper.new
        context[:metric_fpaths].each do |fpath|
          mapper.execute(fpath)
          monitor.progress += 1.0 / context[:metric_fpaths].size
        end
        context[:ld_function_mapping] = mapper.result
      end
    end
  end

  class MapVariablePhase < Phase
    private
    def do_execute(context)
      monitored_region("mvr") do |monitor|
        mapper = VariableMapper.new
        context[:metric_fpaths].each do |fpath|
          mapper.execute(fpath)
          monitor.progress += 1.0 / context[:metric_fpaths].size
        end
        context[:ld_variable_mapping] = mapper.result
      end
    end
  end

  class LinkFunctionPhase < Phase
    private
    def do_execute(context)
      monitored_region("lfn") do |monitor|
        builder = FunctionCallGraphBuilder.new(context[:ld_function_mapping])
        context[:metric_fpaths].each do |fpath|
          builder.execute(fpath)
          monitor.progress += 1.0 / context[:metric_fpaths].size
        end
        context[:ld_function_call_graph] = builder.result
      end
    ensure
      DebugUtil.dump_function_call_graph(context)
    end
  end

  class LinkVariablePhase < Phase
    private
    def do_execute(context)
      monitored_region("lvr") do |monitor|
        builder = VariableReferenceGraphBuilder.new(
          context[:ld_variable_mapping], context[:ld_function_mapping],
          context[:ld_function_call_graph])
        context[:metric_fpaths].each do |fpath|
          builder.execute(fpath)
          monitor.progress += 1.0 / context[:metric_fpaths].size
        end
        context[:ld_variable_reference_graph] = builder.result
      end
    ensure
      DebugUtil.dump_variable_reference_graph(context)
    end
  end

  class PreparePhase < Phase
    private
    def do_execute(context)
      monitored_region("pre") do
        context[:ld_typedef_traversal] =
          TypedefTraversal.new(context[:ld_typedef_mapping])
        context[:ld_function_traversal] =
          FunctionTraversal.new(context[:ld_function_mapping])
        context[:ld_variable_traversal] =
          VariableTraversal.new(context[:ld_variable_mapping])

        context[:ld_commands] = setup_metric_measurements(context) +
                                setup_message_detections(context)
      end
    end

    def setup_metric_measurements(context)
      [
        FN_CALL.new(context)
      ]
    end

    def setup_message_detections(context)
      [
        W0555.new(context),
        W0586.new(context),
        W0589.new(context),
        W0591.new(context),
        W0593.new(context),
        W0628.new(context),
        W0770.new(context),
        W0791.new(context),
        W1037.new(context)
      ]
    end
  end

  class TypedefReviewPhase < Phase
    private
    def do_execute(context)
      monitored_region("rtd") do
        context[:ld_typedef_traversal].execute
      end
    end
  end

  class FunctionReviewPhase < Phase
    private
    def do_execute(context)
      monitored_region("rfn") do
        context[:ld_function_traversal].execute
      end
    end
  end

  class VariableReviewPhase < Phase
    private
    def do_execute(context)
      monitored_region("rvr") do
        context[:ld_variable_traversal].execute
      end
    end
  end

  class CommandPhase < Phase
    private
    def do_execute(context)
      monitored_region("com") do
        context[:ld_commands].each { |command| command.execute }
      end
    end
  end

end
end
