# Preprocessed pure C language source.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/token"
require "adlint/util"
require "adlint/cpp/asm"
require "adlint/cpp/subst"

module AdLint #:nodoc:
module Cpp #:nodoc:

  class PreprocessedSource
    include InlineAssemblyDefinition

    def initialize(root_fpath)
      @root_fpath = root_fpath
      @tokens = []
    end

    extend Pluggable

    def_plugin :on_language_extension
    def_plugin :on_inline_assembly

    attr_reader :root_fpath

    def add_token(token)
      @tokens.push(token)
    end

    def pp_tokens
      @tokens.select { |token| token.type == :PP_TOKEN }
    end

    def substitute_code_blocks
      substitutions = Traits.instance.of_compiler.extension_substitution
      substitutions.each do |pattern, replacement|
        substitution = create_extension_substitution(pattern, replacement)
        @tokens = substitution.execute(@tokens)
      end

      substitutions = Traits.instance.of_compiler.arbitrary_substitution
      substitutions.each do |pattern, replacement|
        substitution = create_arbitrary_substitution(pattern, replacement)
        @tokens = substitution.execute(@tokens)
      end

      create_inline_assembly_substitutions(self).each do |substitution|
        @tokens = substitution.execute(@tokens)
      end

      self
    end

    def to_s
      @io = StringIO.new
      @io.set_encoding(Encoding.default_external)
      @last_fpath = nil
      @last_line_no = 0
      @last_column_no = 1
      @last_token = nil
      @tokens.each { |token| print(token) }
      @io.string
    end

    private
    def create_extension_substitution(pattern, replacement)
      CodeSubstitution.new(pattern, replacement).tap do |subst|
        subst.on_substitution += lambda { |matched_tokens|
          on_language_extension.invoke(matched_tokens)
        }
      end
    end

    def create_arbitrary_substitution(pattern, replacement)
      CodeSubstitution.new(pattern, replacement)
    end

    def print(token)
      return if @last_column_no == 1 && token.type == :NEW_LINE

      if token.location.fpath == @last_fpath
        if @last_line_no < token.location.line_no
          if (vspace = token.location.line_no - @last_line_no) > 3
            insert_line_marker(token)
          else
            vspace.times { @io.puts }
          end
        end
        if (hspace = token.location.appearance_column_no - @last_column_no) > 0
          @io.print(" " * hspace)
        elsif need_hspace?(token)
          @io.print(" ")
        end
        if token.type == :NEW_LINE
          @io.puts
          @last_line_no = token.location.line_no + 1
          @last_column_no = 1
        else
          @io.print(token.value.to_default_external)
          @last_line_no = token.location.line_no
          @last_column_no =
            token.location.appearance_column_no + token.value.length
        end
      else
        insert_line_marker(token)
        print(token)
      end

      @last_token = token
    end

    def need_hspace?(token)
      return false unless @last_token
      if keyword_or_identifier?(@last_token.value)
        !start_with_punctuator?(token.value)
      else
        !(end_with_punctuator?(@last_token.value) ||
          start_with_punctuator?(token.value))
      end
    end

    def start_with_punctuator?(str)
      str !~ /\A[a-z_0-9]/i
    end

    def end_with_punctuator?(str)
      str !~ /[a-z_0-9]\z/i
    end

    def keyword_or_identifier?(str)
      str =~ /\A[a-z_][a-z_0-9]*\z/i
    end

    def insert_line_marker(token)
      if @last_column_no > 1
        @io.puts
      end
      line_marker =
        "# #{token.location.line_no.to_s.to_default_external} " +
        "\"#{token.location.fpath.to_s.to_default_external}\""
      @io.puts(line_marker.to_default_external)
      @last_fpath = token.location.fpath
      @last_line_no = token.location.line_no
      @last_column_no = 1
    end
  end

end
end
