# C conversion semantics.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

module AdLint #:nodoc:
module C #:nodoc:

  # Host class of this module must include InterpreterMediator.
  module Conversion
    def do_conversion(original, to_type)
      if original.type.coercible?(to_type)
        # NOTE: Value will be coerced into the destination type in
        #       VariableTableMediator#temporary_variable.
        temporary_variable(to_type, wrap_around_value(original, to_type))
      else
        nil
      end
    end

    def do_integer_promotion(original)
      return original unless original.type.integer?

      promoted_type = original.type.integer_promoted_type
      if original.type.same_as?(promoted_type)
        original
      else
        do_conversion(original, promoted_type) || original
      end
    end

    def do_usual_arithmetic_conversion(lhs_original, rhs_original)
      if lhs_original.type.pointer? && rhs_original.type.pointer?
        return lhs_original, rhs_original
      end

      arith_type = lhs_original.type.arithmetic_type_with(rhs_original.type)

      if lhs_original.type.same_as?(arith_type)
        lhs_converted = lhs_original
      else
        lhs_converted = do_conversion(lhs_original, arith_type) || lhs_original
      end

      if rhs_original.type.same_as?(arith_type)
        rhs_converted = rhs_original
      else
        rhs_converted = do_conversion(rhs_original, arith_type) || rhs_original
      end

      return lhs_converted, rhs_converted
    end

    def do_default_argument_promotion(original)
      promoted_type = original.type.argument_promoted_type
      if original.type.same_as?(promoted_type)
        original
      else
        do_conversion(original, promoted_type) || original
      end
    end

    private
    def wrap_around_value(original, to_type)
      return original.value unless original.type.scalar? && to_type.scalar?

      case
      when original.type.signed? && to_type.unsigned?
        min = ScalarValue.of(to_type.min_value)
        if (original.value < min).may_be_true?
          return min - original.value + ScalarValue.of(1)
        end
      when original.type.unsigned? && to_type.signed?
        max = ScalarValue.of(to_type.max_value)
        if (original.value > max).may_be_true?
          return max - original.value + ScalarValue.of(1)
        end
      end

      original.value
    end
  end

  # Host class of this module must include StandardTypeCatalogAccessor.
  module UsualArithmeticTypeConversion
    def do_usual_arithmetic_type_conversion(lhs_type, rhs_type)
      # NOTE: The ISO C99 standard saids;
      #
      # 6.3.1.8 Usual arithmetic conversions
      #
      # 1 Many operators that except operands of arithmetic type cause
      #   conversions and yield result types in a similar way.  The purpose is
      #   to determine a common real type for the operands and result.  For the
      #   specified operands, each operand is converted, without change of type
      #   domain, to a type whose corresponding real type is the common real
      #   type.  Unless explicitly stated otherwise, the common real type is
      #   also the corresponding real type of the result, whose type domain is
      #   the type domain of the operands if they are the same, and complex
      #   otherwise.  This pattern is called the usual arithmetic conversions:
      #
      #     First, if the corresponding real type of either operand is long
      #     double, the other operand is converted, without change of type
      #     domain, to a type whose corresponding real type is long double.
      #
      #     Otherwise, if the corresponding real type of either operand is
      #     double, the other operand is converted, without change of type
      #     domain, to a type whose corresponding real type is double.
      #
      #     Otherwise, if the corresponding real type of either operand is
      #     float, the other operand is converted, without change of type
      #     domain, to a type whose corresponding real type is float.
      #
      #     Otherwise, the integer promotions are performed on both operands.
      #     Then the following rules are applied to the promoted operands:
      #
      #       If both operands have the same type, then no further conversion
      #       is needed.
      #
      #       Otherwise, if both operands have signed integer types or both
      #       have unsigned integer types, the operand with the type of lesser
      #       integer conversion rank is converted to the type of the operand
      #       with greater rank.
      #
      #       Otherwise, if the operand that has unsigned integer type has rank
      #       greater or equal to the rank of the type of the other operand,
      #       then the operand with signed integer type is converted to the
      #       type of the operand with unsigned integer type.
      #
      #       Otherwise, if the type of the operand with signed integer type
      #       can represent all of the values of the type of the operand with
      #       unsigned integer type, then the operand with unsigned integer
      #       type is converted to the type of the operand with signed integer
      #       type.
      #
      #       Otherwise, both operands are converted to the unsigned integer
      #       type corresponding to the type of the operand with signed integer
      #       type.

      if lhs_type.same_as?(long_double_type) ||
          rhs_type.same_as?(long_double_type)
        return long_double_type
      end

      if lhs_type.same_as?(double_type) || rhs_type.same_as?(double_type)
        return double_type
      end

      if lhs_type.same_as?(float_type) || rhs_type.same_as?(float_type)
        return float_type
      end

      lhs_promoted_type = lhs_type.integer_promoted_type
      rhs_promoted_type = rhs_type.integer_promoted_type

      return lhs_promoted_type if lhs_promoted_type.same_as?(rhs_promoted_type)

      lhs_rank = lhs_promoted_type.integer_conversion_rank
      rhs_rank = rhs_promoted_type.integer_conversion_rank

      case
      when lhs_promoted_type.signed? && rhs_promoted_type.signed?
        return lhs_rank < rhs_rank ? rhs_promoted_type : lhs_promoted_type
      when lhs_promoted_type.unsigned? && rhs_promoted_type.unsigned?
        return lhs_rank < rhs_rank ? rhs_promoted_type : lhs_promoted_type
      when lhs_promoted_type.unsigned? && lhs_rank >= rhs_rank
        return lhs_promoted_type
      when rhs_promoted_type.unsigned? && lhs_rank <= rhs_rank
        return rhs_promoted_type
      when lhs_promoted_type.signed? &&
           rhs_promoted_type.compatible?(lhs_promoted_type)
        return lhs_promoted_type
      when rhs_promoted_type.signed? &&
           lhs_promoted_type.compatible?(rhs_promoted_type)
        return rhs_promoted_type
      when lhs_promoted_type.signed?
        return lhs_promoted_type.corresponding_unsigned_type
      when rhs_promoted_type.signed?
        return rhs_promoted_type.corresponding_unsigned_type
      end

      raise TypeError, "cannot do usual arithmetic conversion."
    end
  end

end
end
