# C runtime branch of execution path.
#
# Author::    Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>
# Copyright:: Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
# License::   GPLv3+: GNU General Public License version 3 or later
#
# Owner::     Yutaka Yanoh <mailto:yanoh@users.sourceforge.net>

#--
#     ___    ____  __    ___   _________
#    /   |  / _  |/ /   / / | / /__  __/           Source Code Static Analyzer
#   / /| | / / / / /   / /  |/ /  / /                   AdLint - Advanced Lint
#  / __  |/ /_/ / /___/ / /|  /  / /
# /_/  |_|_____/_____/_/_/ |_/  /_/   Copyright (C) 2010-2012, OGIS-RI Co.,Ltd.
#
# This file is part of AdLint.
#
# AdLint is free software: you can redistribute it and/or modify it under the
# terms of the GNU General Public License as published by the Free Software
# Foundation, either version 3 of the License, or (at your option) any later
# version.
#
# AdLint is distributed in the hope that it will be useful, but WITHOUT ANY
# WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR
# A PARTICULAR PURPOSE.  See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License along with
# AdLint.  If not, see <http://www.gnu.org/licenses/>.
#
#++

require "adlint/c/ctrlexpr"
require "adlint/c/option"

module AdLint #:nodoc:
module C #:nodoc:

  class Branch
    include BranchOptions

    def initialize(branch_group, *options)
      @group = branch_group
      @options = options
      @break_event = nil
      @controlling_expression = nil
    end

    attr_reader :group
    attr_reader :break_event
    attr_reader :controlling_expression

    def add_options(*options)
      @options = (@options + options).uniq
      @group.add_options(*options)
    end

    def narrowing?
      @options.include?(NARROWING)
    end

    def widening?
      @options.include?(WIDENING)
    end

    def first?
      @options.include?(FIRST)
    end

    def final?
      @options.include?(FINAL)
    end

    def smother_break?
      @options.include?(SMOTHER_BREAK)
    end

    def implicit_condition?
      @options.include?(IMPLICIT_CONDITION)
    end

    def execute(interpreter, expression = nil, &block)
      environment.enter_versioning_group if first?
      environment.begin_versioning

      @controlling_expression = ControllingExpression.new(interpreter, self)
      if ensure_condition(@controlling_expression, expression)
        @break_event = BreakEvent.catch { yield(self) }
      end

    ensure
      environment.end_versioning(break_with_return?)
      environment.leave_versioning_group(!@group.complete?) if final?

      if final? && @group.complete?
        rethrow_break_event
      end
    end

    def restart_versioning(&block)
      @controlling_expression.save_affected_variables
      environment.end_versioning(false)
      environment.leave_versioning_group(true)
      environment.enter_versioning_group
      environment.begin_versioning
      yield
      @controlling_expression.restore_affected_variables
    end

    def break_with_break?
      @break_event && @break_event.break?
    end

    def break_with_continue?
      @break_event && @break_event.continue?
    end

    def break_with_return?
      @break_event && @break_event.return?
    end

    private
    def ensure_condition(controlling_expression, expression)
      case
      when narrowing?
        controlling_expression.ensure_true_by_narrowing(expression).commit!
      when widening?
        controlling_expression.ensure_true_by_widening(expression).commit!
      else
        raise "branch must be for narrowing or widening."
      end
      @group.all_controlling_variables_value_exist?
    end

    def rethrow_break_event
      case
      when @group.all_branches_break_with_break?
        BreakEvent.of_break.throw unless smother_break?
      when @group.all_branches_break_with_return?
        BreakEvent.of_return.throw
      end
    end

    def environment
      @group.environment
    end
  end

  class BranchGroup
    include BranchOptions
    include BranchGroupOptions

    def initialize(environment, *options)
      @environment = environment
      @options = options
      @branches = []
    end

    attr_reader :environment

    def add_options(*options)
      @options = (@options + options).uniq
    end

    def complete?
      @options.include?(COMPLETE)
    end

    def create_first_branch(*options)
      new_branch = Branch.new(self, FIRST, *options)
      @branches.push(new_branch)
      new_branch
    end

    def create_trailing_branch(*options)
      new_branch = Branch.new(self, *options)
      @branches.push(new_branch)
      new_branch
    end

    def all_controlling_variables
      @branches.map { |branch|
        branch.controlling_expression ?
          branch.controlling_expression.affected_variables : []
      }.flatten.uniq
    end

    def all_controlling_variables_value_exist?
      all_controlling_variables.all? { |variable| variable.value.exist? }
    end

    def all_branches_break_with_break?
      @branches.all? { |branch| branch.break_with_break? }
    end

    def all_branches_break_with_continue?
      @branches.all? { |branch| branch.break_with_continue? }
    end

    def all_branches_break_with_return?
      @branches.all? { |branch| branch.break_with_return? }
    end
  end

  class BreakEvent
    def self.catch(&block)
      Kernel.catch(:break) { yield; nil }
    end

    def self.of_break
      @break ||= new(:break)
    end

    def self.of_continue
      @continue ||= new(:continue)
    end

    def self.of_return
      @return ||= new(:return)
    end

    def initialize(type)
      @type = type
    end
    private_class_method :new

    def break?
      @type == :break
    end

    def continue?
      @type == :continue
    end

    def return?
      @type == :return
    end

    def throw
      Kernel.throw(:break, self)
    end
  end

end
end
