; CUTIL - Centaur Basic Utilities
; Copyright (C) 2008-2011 Centaur Technology
;
; Contact:
;   Centaur Technology Formal Verification Group
;   7600-C N. Capital of Texas Highway, Suite 300, Austin, TX 78731, USA.
;   http://www.centtech.com/
;
; This program is free software; you can redistribute it and/or modify it under
; the terms of the GNU General Public License as published by the Free Software
; Foundation; either version 2 of the License, or (at your option) any later
; version.  This program is distributed in the hope that it will be useful but
; WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
; FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
; more details.  You should have received a copy of the GNU General Public
; License along with this program; if not, write to the Free Software
; Foundation, Inc., 51 Franklin Street, Suite 500, Boston, MA 02110-1335, USA.
;
; Original author: Jared Davis <jared@centtech.com>

(in-package "CUTIL")
(include-book "defprojection")

(defxdoc defmapappend
  :parents (cutil)
  :short "Append transformations of list elements."

  :long "<p>Defmapappend allows you to quickly introduce a function like</p>

<code>
 (loop for elem in x append (f elem))
</code>

<p>and produces some basic theorems about this new function.</p>

<p>General form:</p>

<code>
 (defmapappend name formals
    transform
    &amp;key guard                   ; t by default
         verify-guards           ; t by default
         transform-exec          ; nil by default
         transform-true-list-p   ; nil by default
         mode                    ; default defun-mode by default
         parents                 ; '(undocumented) by default
         short                   ; nil by default
         long                    ; nil by default
         )
</code>

<p>For instance,</p>

<code>
 (defmapappend append-square-lists (x)
    (square-lists x)
    :guard (integer-list-listp x))
</code>

<p>would introduce a new function, <tt>append-square-lists</tt>, that applies
<tt>square-lists</tt> to every element of its argument and appends together
all of the results.</p>

<p>Note that <b>x</b> is treated in a special way: it refers to the whole list
in the formals and guards, but refers to the individual elements of the list in
the <tt>element</tt> portion.  This is similar to how other macros like @(see
deflist), @(see defalist), and @(see defprojection) handle <tt>x</tt>.</p>



<h3>Usage and Arguments</h3>

<p>Let <tt>pkg</tt> be the package of <tt>name</tt>.  All functions, theorems,
and variables are created in this package.  One of the formals must be
<tt>pkg::x</tt>, and this argument represents the list that will be
transformed.  Otherwise, the only restriction on formals is that you may not
use the names <tt>pkg::a</tt>, <tt>pkg::y</tt>, and <tt>pkg::acc</tt>, because
we use these variables in the theorems we generate.</p>

<p>The <tt>transform</tt> should indicate an element transforming function that
produces a list of some kind as its output.  Adopting an ML-like syntax,
<tt>transform</tt> should have a signature such as the following:</p>

<code>
  transform : elem -&gt; A list
</code>

<p>We produce a new function of the given <tt>name</tt>, which has the
signature:</p>

<code>
  name : elem list -&gt; A list
</code>

<p>Our new function applies <tt>transform</tt> to every element in its input
list, and appends together all of the results.  That is, the logical definition
of the new function we introduce is as follows:</p>

<code>
 (defun name (x)
   (if (atom x)
       nil
     (append (transform (car x))
             (name (cdr x)))))
</code>

<p>The new function will be more efficient than the above.  In particular, we
write a <tt>mappappend-exec</tt> function that builds the answer in reverse
using revappend and reverses it at the end.  An even more efficient version is
possible when the <tt>:transform-exec</tt> option is provided; see below for
details.</p>

<p>The optional <tt>:guard</tt> and <tt>:verify-guards</tt> are given to the
<tt>defund</tt> event that we introduce.  Often @(see deflist) is convenient
for introducing the necessary guard.</p>

<p>The optional <tt>:mode</tt> keyword can be set to <tt>:logic</tt> or
<tt>:program</tt> to introduce the recognizer in logic or program mode.  The
default is whatever the current default defun-mode is for ACL2, i.e., if you
are already in program mode, it will default to program mode, etc.</p>

<p>The optional <tt>:transform-true-list-p</tt> argument can be set to
<tt>t</tt> whenever the transformation is known to unconditionally produce a
true list, and allows us to slightly optimize our function.</p>

<h4>The :transform-exec argument</h4>

<p>When provided, the optional <tt>:transform-exec</tt> argument should be the
name of a function that satisfies the following property:</p>

<code>
  (implies (true-listp acc)
           (equal (transform-exec x acc)
                  (append (rev (transform x)) acc)))
</code>

<p>Note that such functions are automatically introduced by @(see
defprojection).  For instance,</p>

<code>
 (defprojection square-list (x)
   (square x))
</code>

<p>generates a suitable function named <tt>square-list-exec</tt>.  Amusingly,
suitable functions are also generated by defmapappend, itself.</p>

<p>When such a function is provided, we can use it to generate a more efficient
implementation, which uses the tail-recursive function to build the answer in
reverse, and then reverses it at the very end, avoiding even the intermediate
computation of the lists emitted by <tt>transform</tt>.</p>")

(defun defmapappend-fn (name formals transform
                             guard verify-guards
                             transform-exec transform-true-list-p
                             mode
                             parents short long
                             )
  (declare (xargs :mode :program))
  (b* (((unless (symbolp name))
        (er hard? 'defmapappend "Name must be a symbol, but is ~x0." name))

       (mksym-package-symbol name)

       ;; Special variables that are reserved by defmapappend
       (x   (intern-in-package-of-symbol "X" name))
       (a   (intern-in-package-of-symbol "A" name))
       (n   (intern-in-package-of-symbol "N" name))
       (y   (intern-in-package-of-symbol "Y" name))
       (acc (intern-in-package-of-symbol "ACC" name))

       ((unless (and (symbol-listp formals)
                     (no-duplicatesp formals)))
        (er hard 'defmapappend
            "The formals must be a list of unique symbols, but the ~
            formals are ~x0." formals))

       ((unless (member x formals))
        (er hard 'defmapappend
            "The formals must contain X, but are ~x0.~%" formals))

       ((unless (and (not (member a formals))
                     (not (member n formals))
                     (not (member y formals))
                     (not (member acc formals))))
        (er hard 'defmapappend
            "As a special restriction, formals may not mention a, n, ~
            or y, but the formals are ~x0." formals))

       ((unless (and (consp transform)
                     (symbolp (car transform))))
        (er hard 'defmapappend
            "The transform must be a function applied to the formals, ~
             but is ~x0." transform))

       (exec-fn        (mksym name '-exec))
       (transform-fn   (car transform))
       (transform-args (cdr transform))

       ((unless (and (subsetp-equal transform-args formals)
                     (subsetp-equal formals transform-args)))
        (er hard 'defmapappend
            "The transform's formals do not agree with the defmapappend ~
             function's formals."))

       ((unless (booleanp verify-guards))
        (er hard 'defmapappend
            ":verify-guards must be a boolean, but is ~x0."
            verify-guards))

       ((unless (booleanp transform-true-list-p))
        (er hard 'defmapappend
            ":transform-true-list-p must be a boolean, but is ~x0."
            transform-true-list-p))

       ((unless (symbolp transform-exec))
        (er hard 'defmapappend
            ":transform-exec must be a symbol, but is ~x0."
            transform-exec))

       ((unless (member mode '(:logic :program)))
        (er hard 'defmapappend
            ":mode must be :logic or :program, but is ~x0."
            mode))

       (short (or short
                  (and parents
                       (str::cat "@(call " (symbol-name name)
                                 ") applies @(see " (symbol-name transform-fn)
                                 ") to every member of the list <tt>x</tt>, "
                                 "and appends together all the resulting lists."))))

       (long (or long
                 (and parents
                      (str::cat "<p>This is an ordinary @(see defmapappend).</p>"
                                "@(def " (symbol-name name) ")"))))

       (doc (if (or parents short long)
                `((defxdoc ,name :parents ,parents :short ,short :long ,long))
              nil)))

    `(encapsulate
       ()

       ,(if (eq mode :program)
            '(program)
          '(logic))

       ,@doc

       (defund ,exec-fn (,@formals ,acc)
         (declare (xargs :guard ,guard
                         :verify-guards nil))
         (if (consp ,x)
             (,exec-fn ,@(subst `(cdr ,x) x formals)
                       ,(if transform-exec
                            `(,transform-exec ,@(subst `(car ,x) x transform-args) ,acc)
                          `(,(if transform-true-list-p
                                 'revappend
                               'revappend-without-guard)
                            (,transform-fn . ,(subst `(car ,x) x transform-args))
                            ,acc)))
           ,acc))

       (defund ,name (,@formals)
         (declare (xargs :guard ,guard
                         :verify-guards nil))
         (mbe :logic (if (consp ,x)
                         (append (,transform-fn . ,(subst `(car ,x) x transform-args))
                                 (,name . ,(subst `(cdr ,x) x formals)))
                       nil)
              :exec (reverse (,exec-fn ,@formals nil))))

       ,@(and
          (eq mode :logic)
          `((defthm ,(mksym name '-when-not-consp)
              (implies (not (consp ,x))
                       (equal (,name . ,formals)
                              nil))
              :hints(("Goal" :in-theory (enable ,name))))

            (defthm ,(mksym name '-of-cons)
              (equal (,name . ,(subst `(cons ,a ,x) x formals))
                     (append (,transform-fn . ,(subst a x transform-args))
                             (,name . ,formals)))
              :hints(("Goal" :in-theory (enable ,name))))

            (local (defthm lemma
                     (implies (true-listp ,acc)
                              (true-listp (,exec-fn ,@formals ,acc)))
                     :hints(("Goal" :in-theory (enable ,exec-fn)))))

            (defthm ,(mksym exec-fn '-removal)
              (equal (,exec-fn ,@formals ,acc)
                     (append (rev (,name ,@formals)) ,acc))
              :hints(("Goal" :in-theory (enable ,exec-fn))))

            ,@(if verify-guards
                  `((verify-guards ,exec-fn)
                    (verify-guards ,name))
                nil)

            (defthm ,(mksym name '-of-list-fix)
              (equal (,name . ,(subst `(list-fix ,x) x formals))
                     (,name . ,formals))
              :hints(("Goal" :induct (len ,x))))

            (defthm ,(mksym name '-of-append)
              (equal (,name . ,(subst `(append ,x ,y) x formals))
                     (append (,name . ,formals)
                             (,name . ,(subst y x formals))))
              :hints(("Goal" :induct (len ,x))))

            )))))

(defmacro defmapappend (name formals transform
                             &key
                             transform-exec
                             (transform-true-list-p 't)
                             (guard 't)
                             (verify-guards 't)
                             mode
                             (parents '(acl2::undocumented))
                             (short 'nil)
                             (long 'nil))
  `(make-event (let ((mode (or ',mode (default-defun-mode (w state)))))
                 (defmapappend-fn ',name ',formals ',transform
                   ',guard ',verify-guards
                   ',transform-exec ',transform-true-list-p
                   mode
                   ',parents ',short ',long))))


