// -*- C++ -*-

// ===================================================================
/**
 *  @file   SSLIOP_Current.h
 *
 *  SSLIOP_Current.h,v 1.12 2004/07/31 13:06:27 ossama Exp
 *
 *  @author Ossama Othman <ossama@dre.vanderbilt.edu>
 */
// ===================================================================

#ifndef TAO_SSLIOP_CURRENT_H
#define TAO_SSLIOP_CURRENT_H

#include /**/ "ace/pre.h"

#include "SSLIOP_Export.h"

#if !defined (ACE_LACKS_PRAGMA_ONCE)
# pragma once
#endif /* ACE_LACKS_PRAGMA_ONCE */

#include "SSLIOP_Current_Impl.h"

#include "orbsvcs/orbsvcs/SSLIOPC.h"
#include "tao/ORB_Core.h"

// This is to remove "inherits via dominance" warnings from MSVC.
// MSVC is being a little too paranoid.
#if defined(_MSC_VER)
#if (_MSC_VER >= 1200)
#pragma warning(push)
#endif /* _MSC_VER >= 1200 */
#pragma warning(disable:4250)
#endif /* _MSC_VER */


namespace TAO
{
  namespace SSLIOP
  {
    class Current;
    class Current_var;
    typedef Current * Current_ptr;

    /**
     * @class Current
     *
     * @brief Implementation of the TAO SSLIOP::Current extension.
     *
     * This object can be used to obtain SSL session related
     * information about the current execution context.  For example,
     * SSL peer certificate chains for the current request can be
     * obtained from this object.
     */
    class TAO_SSLIOP_Export Current
      : public ::SSLIOP::Current,
        public TAO_Local_RefCounted_Object
    {
    public:

      /// Constructor.
      Current (TAO_ORB_Core *orb_core);

      /// Return the peer certificate associated with the current
      /// request.
      virtual ::SSLIOP::ASN_1_Cert * get_peer_certificate (
          ACE_ENV_SINGLE_ARG_DECL)
        ACE_THROW_SPEC ((CORBA::SystemException,
                         ::SSLIOP::Current::NoContext));

      /**
       * Return the certificate chain associated with the current
       * execution context.  If no SSL session is being used for the
       * request or upcall, then the NoContext exception is raised.
       * On the client side, the chain does include the peer (server)
       * certficate.  However, the certificate chain on the server
       * side does NOT contain the peer (client) certificate.
       */
      virtual ::SSLIOP::SSL_Cert * get_peer_certificate_chain (
          ACE_ENV_SINGLE_ARG_DECL)
        ACE_THROW_SPEC ((CORBA::SystemException,
                         ::SSLIOP::Current::NoContext));

      /**
       * This method is mostly useful as an inexpensive means of
       * determining whether or not SSL session state is available.
       *
       * @return @c true if the current execution context is not
       *         within a SSL session.
       */
      virtual CORBA::Boolean no_context (ACE_ENV_SINGLE_ARG_DECL)
        ACE_THROW_SPEC ((CORBA::SystemException));

      /// Set the TSS slot ID assigned to this object.
      void tss_slot (size_t slot);

      /// Setup the Current.
      void setup (Current_Impl *& prev_impl,
                  Current_Impl * new_impl,
                  bool &setup_done);

      /// Teardown the Current for this request.
      void teardown (Current_Impl *prev_impl,
                     bool &setup_done);

      /**
       * @name Downcast and Reference Counting Methods
       *
       * These are basically the same methods generated by the IDL
       * compiler for all IDL interfaces.
       */
      //@{
#if !defined(__GNUC__) || !defined (ACE_HAS_GNUG_PRE_2_8)
      typedef Current_ptr _ptr_type;
      typedef Current_var _var_type;
#endif /* ! __GNUC__ || g++ >= 2.8 */

      static int _tao_class_id;

      // The static operations.
      static Current_ptr _duplicate (Current_ptr obj);

      static Current_ptr _narrow (CORBA::Object_ptr obj
                                  ACE_ENV_ARG_DECL);


      static Current_ptr _unchecked_narrow (CORBA::Object_ptr obj
                                            ACE_ENV_ARG_DECL);


      static Current_ptr _nil (void)
      {
        return (Current_ptr)0;
      }

      virtual const char* _interface_repository_id (void) const;
      //@}

    protected:

      /// Destructor
      ~Current (void);

      /// Set the TSS SSLIOP::Current implementation.
      int implementation (Current_Impl *impl);

      /// Return the TSS SSLIOP::Current implementation.
      Current_Impl *implementation (void);

    private:

      /// Prevent copying through the copy constructor and the assignment
      /// operator.
      //@{
      ACE_UNIMPLEMENTED_FUNC (Current (const Current &))
      ACE_UNIMPLEMENTED_FUNC (void operator= (const Current &))
      //@}

        private:

      /// TSS slot assigned to this object.
      size_t tss_slot_;

      /// Pointer to the ORB Core corresponding to the ORB with which this
      /// object is registered.
      TAO_ORB_Core *orb_core_;

    };

    class Current_var : public TAO_Base_var
    {
    public:
      Current_var (void); // default constructor
      Current_var (Current_ptr p) : ptr_ (p) {}
      Current_var (const Current_var &); // copy constructor
      ~Current_var (void); // destructor

      Current_var &operator= (Current_ptr);
      Current_var &operator= (const Current_var &);
      Current_ptr operator-> (void) const;

      operator const Current_ptr &() const;
      operator Current_ptr &();
      // in, inout, out, _retn
      Current_ptr in (void) const;
      Current_ptr &inout (void);
      Current_ptr &out (void);
      Current_ptr _retn (void);
      Current_ptr ptr (void) const;

      // Hooks used by template sequence and object manager classes
      // for non-defined forward declared interfaces.
      static Current_ptr tao_duplicate (Current_ptr);
      static void tao_release (Current_ptr);
      static Current_ptr tao_nil (void);
      static Current_ptr tao_narrow (CORBA::Object *
                                                ACE_ENV_ARG_DECL);
      static CORBA::Object * tao_upcast (void *);

    private:
      Current_ptr ptr_;
      // Unimplemented - prevents widening assignment.
      Current_var (const TAO_Base_var &rhs);
      Current_var &operator= (const TAO_Base_var &rhs);
    };
  }  // End SSLIOP namespace.
}  // End TAO namespace.

#if defined (__ACE_INLINE__)
# include "SSLIOP_Current.inl"
#endif /* __ACE_INLINE__ */

#if defined(_MSC_VER) && (_MSC_VER >= 1200)
#pragma warning(pop)
#endif /* _MSC_VER */

#include /**/ "ace/post.h"

#endif  /* TAO_SSLIOP_CURRENT_H */
