/* ------------------------------------------------------------------------
 * $Id: TriangleGeometryImpl.cc,v 1.24 2001/08/01 15:09:00 elm Exp $
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-06-21 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- System Includes
#include <algorithm>

// -- 3Dwm Includes
#include "Celsius/debug.hh"
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Nobel/Appearance.hh"
#include "Nobel/Renderer.hh"
#include "Polhem/VolumeImpl.hh"
#include "Polhem/TriangleGeometryImpl.hh"

using namespace Nobel;

#define MIDPOINT(v1, v2, out) out.x = (v2.x + v1.x) / 2; \
                              out.y = (v2.y + v1.y) / 2; \
                              out.z = (v2.z + v1.z) / 2;

// -- Code Segment 

TriangleGeometryImpl::TriangleGeometryImpl()
    : FaceGeometryImpl()
{
    // empty
}

TriangleGeometryImpl::~TriangleGeometryImpl()
{
    // empty
}

void TriangleGeometryImpl::setVertexNumber(CORBA::Long n)
{
    Guard<Mutex> guard(_mutex);

    // Make sure the value is legal 
    if (n > 0) {

	// Update the limit and the sequence length
	_vertexNumber = n;
	_mesh.vertexList.length(n);
    }
}

void TriangleGeometryImpl::setNormalNumber(CORBA::Long n)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the value is legal 
    if (n > 0) {
	
	// Update the limit and the sequence length
	_normalNumber = n;
	_mesh.normalList.length(n);
    }
}

void TriangleGeometryImpl::setTexCoordNumber(CORBA::Long n)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the value is legal 
    if (n > 0) {
	
	// Update the limit and the sequence length
	_texCoordNumber = n;
	_mesh.texCoordList.length(n);
    }
}

void TriangleGeometryImpl::setFaceNumber(CORBA::Long n)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the value is legal 
    if (n > 0) {

	// Update the limit and the sequence lengths
	_faceNumber = n;
	_mesh.vertexIndexList.length(n * 3);
	_mesh.texCoordIndexList.length(n * 3);
	_mesh.normalIndexList.length(n * 3);
    }
}

void TriangleGeometryImpl::setVertex(CORBA::Long index, const Vertex3D &v)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the index is within legal range
    if (index >= 0 && index < _vertexNumber) {

	// Update the value and mark the geometry as dirty
	_mesh.vertexList[index] = v;
	_dirty = true;
    }
}

Vertex3D TriangleGeometryImpl::getVertex(CORBA::Long index)
{
    // Make sure the index is with legal range
    if (index < 0 || index >= _vertexNumber)
	throw Nobel::Geometry::OutOfRange();

    // It is, so return the value
    return _mesh.vertexList[index];
}

void TriangleGeometryImpl::setTexCoord(CORBA::Long index, const TexCoord &tc)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the index is within legal range
    if (index >= 0 && index < _texCoordNumber) {

	// Update the value and mark the geometry as dirty
	_mesh.texCoordList[index] = tc;
	_dirty = true;
    }
}

TexCoord TriangleGeometryImpl::getTexCoord(CORBA::Long index)
{
    // Make sure the index is with legal range
    if (index < 0 || index >= _texCoordNumber)
	throw Nobel::Geometry::OutOfRange();

    // It is, so return the value
    return _mesh.texCoordList[index];
}

void TriangleGeometryImpl::setNormal(CORBA::Long index, const Vertex3D &n)
{    
    Guard<Mutex> guard(_mutex);
    
    // Make sure the index is within legal range
    if (index >= 0 && index < _normalNumber) {
	
	// Update the value and mark the geometry as dirty
	_mesh.normalList[index] = n;
	_dirty = true;
    }
}

Vertex3D TriangleGeometryImpl::getNormal(CORBA::Long index)
{
    // Make sure the index is with legal range
    if (index < 0 || index >= _normalNumber)
	throw Nobel::Geometry::OutOfRange();

    // It is, so return the value
    return _mesh.normalList[index];
}

void TriangleGeometryImpl::setVertexIndex(CORBA::Long index,
					  const Triangle &f)
{
    Guard<Mutex> guard(_mutex);
    
    // Make sure the index is with legal range
    if (index >= 0 && index < _faceNumber) {
	_mesh.vertexIndexList[a(index)] = f.a;
	_mesh.vertexIndexList[b(index)] = f.b;
	_mesh.vertexIndexList[c(index)] = f.c;
    }
}

Triangle TriangleGeometryImpl::getVertexIndex(CORBA::Long index)
{
    // Make sure the index is with legal range
    if (index < 0 || index >= _faceNumber)
	throw Nobel::Geometry::OutOfRange();

    // It is, so return the value
    return face(_mesh.vertexIndexList, index);
}

void TriangleGeometryImpl::setTexCoordIndex(CORBA::Long index,
					    const Triangle &f)
{
    Guard<Mutex> guard(_mutex);
    
    if (index >= 0 && index < _faceNumber) {
	_mesh.texCoordIndexList[a(index)] = f.a;
	_mesh.texCoordIndexList[b(index)] = f.b;
	_mesh.texCoordIndexList[c(index)] = f.c;
    }
}

Triangle TriangleGeometryImpl::getTexCoordIndex(CORBA::Long index)
{
    // @@@ Don't know whether we need to duplicate here!
    if (index >= 0 && index < _faceNumber)
	return face(_mesh.texCoordIndexList, index);
    throw Nobel::Geometry::OutOfRange();
}

void TriangleGeometryImpl::setNormalIndex(CORBA::Long index, const Triangle &f)
{
    Guard<Mutex> guard(_mutex);
    
    if (index >= 0 && index < _faceNumber) {
	_mesh.normalIndexList[a(index)] = f.a;
	_mesh.normalIndexList[b(index)] = f.b;
	_mesh.normalIndexList[c(index)] = f.c;
    }
}

Triangle TriangleGeometryImpl::getNormalIndex(CORBA::Long index)
{
    // @@@ Don't know whether we need to duplicate here!
    if (index >= 0 && index < _faceNumber)
	return face(_mesh.normalIndexList, index);
    throw Nobel::Geometry::OutOfRange();
}

void TriangleGeometryImpl::render(Renderer_ptr r)
{
    // Send the entire mesh to the renderer
    Guard<Mutex> guard(_mutex);
    r->renderTriangles(_mesh);
}

TriangleMesh *TriangleGeometryImpl::getMesh()
{
    // Create a new triangle mesh and copy contents from the existing mesh
    Guard<Mutex> guard(_mutex);
    TriangleMesh *mesh_ptr = new TriangleMesh(_mesh);
    return mesh_ptr;
}

void TriangleGeometryImpl::setMesh(const TriangleMesh &mesh)
{
    Guard<Mutex> guard(_mutex);
    
    // Copy contents of the new mesh into the internal mesh
    _mesh = mesh; 
    
    // Update geometry limits
    _vertexNumber = mesh.vertexList.length();
    _normalNumber = mesh.normalList.length();
    _texCoordNumber = mesh.texCoordList.length();
    _faceNumber = mesh.vertexIndexList.length() / 3;
    
    // Update mesh list lengths
    _mesh.normalIndexList.length(_faceNumber * 3);
    _mesh.texCoordIndexList.length(_faceNumber * 3);
}

void TriangleGeometryImpl::recomputeBoundingVolume()
{
    Vertex3D minimum, maximum;

    // Find the extremes of the vertex data
    findMinMax(_mesh.vertexList, _vertexNumber, minimum, maximum);
    
    // Update the bounding volume
    _bounds->setBounds(minimum, maximum);

    // Okay, the geometry isn't dirty anymore
    _dirty = false;
}
