/* ------------------------------------------------------------------------
 * AppearanceImpl.cc
 *
 * This file is part of 3Dwm: The Three-Dimensional User Environment.
 *
 * 3Dwm: The Three-Dimensional User Environment:
 *	<http://www.3dwm.org>
 *
 * Chalmers Medialab
 * 	<http://www.medialab.chalmers.se>
 * 
 * ------------------------------------------------------------------------
 * File created 2000-08-07 by Niklas Elmqvist.
 *
 * Copyright (c) 2000 Niklas Elmqvist <elm@3dwm.org>.
 * Copyright (c) 2000 Steve Houston <shouston@programmer.net>.
 * ------------------------------------------------------------------------
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 * ------------------------------------------------------------------------
 */

// -- 3Dwm Includes
#include "Celsius/Mutex.hh"
#include "Celsius/Guard.hh"
#include "Polhem/RendererImpl.hh"
#include "Polhem/AppearanceImpl.hh"

using namespace Nobel;

// -- Code Segment

AppearanceImpl::AppearanceImpl()
    : //_shading(ShadingAttributes::ShadingNone),
    _use_lighting(false)
{
    const Color white = { 1.0f, 1.0f, 1.0f, 1.0f };
    _material.shininess = 0.0f;
    _material.ambient = white;
    _material.diffuse = white;
    _material.specular = white;
    _material.emissive = white;
}
    
AppearanceImpl::~AppearanceImpl()
{
    // empty
}

Nobel::Appearance::ShadingAttributes AppearanceImpl::shading()
{
    return _shading;
}

void AppearanceImpl::shading(Nobel::Appearance::ShadingAttributes sa)
{
    Guard<Mutex> guard(_mutex);
    _shading = sa;
}

CORBA::Boolean AppearanceImpl::lighting() 
{
    return _use_lighting;
}

void AppearanceImpl::lighting(CORBA::Boolean l)
{
    Guard<Mutex> guard(_mutex);
    _use_lighting = l;
}

MaterialAttributes AppearanceImpl::material()
{
    return _material;
}

void AppearanceImpl::material(const MaterialAttributes &ma)
{
    Guard<Mutex> guard(_mutex);
    _material = ma;
}

void AppearanceImpl::setTexture(Texture_ptr tex)
{
    Guard<Mutex> guard(_mutex);
    _tex = Texture::_duplicate(tex);
}

Texture_ptr AppearanceImpl::getTexture()
{
    return Texture::_duplicate(_tex);
}

void AppearanceImpl::apply(Renderer_ptr r)
{
    Guard<Mutex> guard(_mutex);

    // Apply the texture (if present)
    if (!CORBA::is_nil(_tex)) {

	// Make sure the texture is initialized
	if (!_tex->isReady())
	    _tex->init(r);
	
	// Apply it
	_tex->apply(r);	
    }
    else
	// No texture, disable texture-mapping
	r->disableTexture();
    
    // TODO: apply all other sorts of render states (lighting etc. etc.)
    
    // Apply material attributes
    r->applyMaterial(_material);
}
