/* Stars -- Displays a Map of the Night Sky
    Copyright (C) September 22, 2002  Walter Brisken

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA*/

#include <stdio.h>
#include "glade/glade.h"
#include "configure.h"
#include "configurelocation.h"
#include "configurevista.h"
#include "configurecoords.h"
#include "viewer.h"

struct configurator *configinfo = 0;

/* Protos */

void  c_accept_click(GtkWidget *widget, struct configurator *ci);
void     c_try_click(GtkWidget *widget, struct configurator *ci);
void  c_revert_click(GtkWidget *widget, struct configurator *ci);
void c_dismiss_click(GtkWidget *widget, struct configurator *ci);


struct configurator *newconfigurator()
{
	static GladeXML *configxml = 0;
	struct configurator *ci;

	GtkWidget *acceptbut;
	GtkWidget *trybut;
	GtkWidget *revertbut;
	GtkWidget *dismissbut;

	if(configxml == 0) configxml = glade_xml_new(PREFIXDIR "/share/stars/glade/configure.glade", 0);
	if(configxml == 0)
	{
		fprintf(stderr, "newconfigurator: configure.glade not found\n");
		return 0;
	}

	ci = g_new(struct configurator, 1);
	ci->configwindow = glade_xml_get_widget(configxml, "configwindow");
	if(!ci)
	{
		printf("configwindow not allocated\n");
	}
	ci->v = 0;

	acceptbut = glade_xml_get_widget(configxml, "acceptbutton");
	gtk_signal_connect(GTK_OBJECT(acceptbut), "clicked",
		GTK_SIGNAL_FUNC(c_accept_click), ci);

	trybut = glade_xml_get_widget(configxml, "trybutton");
	gtk_signal_connect(GTK_OBJECT(trybut), "clicked",
		GTK_SIGNAL_FUNC(c_try_click), ci);

	revertbut = glade_xml_get_widget(configxml, "revertbutton");
	gtk_signal_connect(GTK_OBJECT(revertbut), "clicked",
		GTK_SIGNAL_FUNC(c_revert_click), ci);

	dismissbut = glade_xml_get_widget(configxml, "dismissbutton");
	gtk_signal_connect(GTK_OBJECT(dismissbut), "clicked",
		GTK_SIGNAL_FUNC(c_dismiss_click), ci);

	gtk_signal_connect(GTK_OBJECT(ci->configwindow), "delete_event",
		GTK_SIGNAL_FUNC(c_dismiss_click), ci);
		
	ci->locinfo = newlocationconfig(configxml);
	ci->viewinfo = newvistaconfig(configxml);
	ci->coordsinfo = newcoordsconfig(configxml);
	
	return ci;
}

void delteconfigurator(struct configurator *ci)
{
	if(!ci)
	{
		fprintf(stderr, "Trying to delete null configurator\n");
		return;
	}
	deletelocationconfig(ci->locinfo);
	deletevistaconfig(ci->viewinfo);
	deletecoordsconfig(ci->coordsinfo);
}

/* ------------------- Functions ----------------- */

void setupconfiginfo()
{
	if(!configinfo)
	{
		fprintf(stderr, "setupconfig: configinfo = 0\n");
		return;
	}
	locationconfigsetup(configinfo);
	vistaconfigsetup(configinfo);
	coordsconfigsetup(configinfo);
}

void hideconfigwindow()
{
	if(!configinfo)
	{
		fprintf(stderr, "Error, hiding null configwindow\n");
		return;
	}
	if(configinfo->v) 
	{
		gtk_widget_hide(configinfo->configwindow);
		configinfo->v = 0;
	}
}

void showconfigwindow(struct viewer *v)
{
	if(!configinfo)
	{
		configinfo = newconfigurator();
		if(!configinfo) 
		{
			fprintf(stderr, "Could not make configurator\n");
			return;
		}
	}
	if(!configinfo->v) 
	{
		configinfo->v = v;
		setupconfiginfo(v);
		gtk_widget_show(configinfo->configwindow);
	}
}

/* temporary ... */
void configmatrixchanged(struct viewer *v)
{
	if(!configinfo) return;
	if(v != configinfo->v) return;
	vistaconfigrefresh(configinfo);
}

/* ------------------- Callbacks ----------------- */

void c_accept_click(GtkWidget *widget, struct configurator *ci)
{
	c_try_click(widget, ci);
	hideconfigwindow();
}

void c_try_click(GtkWidget *widget, struct configurator *ci)
{
	locationconfigtry(ci);
	vistaconfigtry(ci);
	coordsconfigtry(ci);
}

void c_revert_click(GtkWidget *widget, struct configurator *ci)
{
	locationconfigrevert(ci);
	vistaconfigrevert(ci);
	coordsconfigrevert(ci);
}

void c_dismiss_click(GtkWidget *widget, struct configurator *ci)
{
	hideconfigwindow();
}

/* ------------------- End of Callbacks ----------- */

