/*
 * snes9express
 * joystick.cpp
 * (C) 1998, David Nordlund
 * Licensed under the "Artistic" license
 * To view the details of the Artistic license, see the file
 * ./ARTISTIC, or go to http://www.opensource.org/artistic-license.html
 */

#include <stdio.h>
#include <stdlib.h>
#include <glib.h>
#include <gdk/gdk.h>
#include "frend.h"
#include "interface.h"
#ifdef S9X_INCLUDE_JOYSTICK
# include <fcntl.h>
# include <unistd.h>
# include <sys/ioctl.h>
# include <linux/joystick.h>
#endif

#include "pix/joystick.xpm"
#include "pix/controller.xpm"
#include "pix/def.xpm"
#include "pix/del.xpm"

static char*Default_JS_Devices[] = {
   "/dev/js0", "/dev/js1", "/dev/js2", "/dev/js3"
};
static char*JS_Labels[] = {
   "Joystick 1", "Joystick 2", "Joystick 3", "Joystick 4"
};
static char*devJSargs[] = {
   "-joydev1", "-joydev2", "-joydev3", "-joydev4"
};

static char	*MapValues[16] = {
   "0",  "1",  "2",  "3",  "4",  "5",  "6",  "7",  "8",  "9",
     "10", "11", "12", "13", "14", "15"
};
static char	*DefaultMap[8] = {
   "1", "0", "4", "3", "6", "7", "8", "9"
};
const int ButtonOffset = 2;
static char	*MapHeaders[] = {
   "js", "Device Name", "A", "B", "X", "Y", "L", "R", "Start", "Select"
};
static int	MapHeaderWidths[] = {
   18, 128, 24, 24, 24, 24, 24, 24, 40, 40
};
static char	*EmptyRow[] = {
   "", "", "", "", "", "", "", "", "", ""
};
static char*MapArgs[] = {
   "-joymap1", "-joymap2", "-joymap3", "-joymap4"
};
void s9x_Joystick__DeviceEvent(void*JMptr, int fd, GdkInputCondition IC);

fr_Image *BtnInd, *BtnX;


s9x_Joystick::s9x_Joystick(fr_Element*parent):
fr_Notepage(parent, "Joystick"),
DevWindow(this, "Joystick Devices", joystick_xpm, "js devs"),
MapWindow(this, "Joystick Maps", joystick_xpm, "js maps"),
devJS0(this, JS_Labels[0], (char*)0, Default_JS_Devices[0], joystick_xpm),
devJS1(this, JS_Labels[1], (char*)0, Default_JS_Devices[1], joystick_xpm),
devJS2(this, JS_Labels[2], (char*)0, Default_JS_Devices[2], joystick_xpm),
devJS3(this, JS_Labels[3], (char*)0, Default_JS_Devices[3], joystick_xpm),
DevBtn(this, "Devices ..."),
DevClose(this, "Close"),
MapBtn(this, "Joystick Maps"),
MapCloseBtn(this, "Close"),
Buttons(this, "Buttons"),	
Old(this, "Old-style", false),
UseJoystick(this, "Use Joystick", true),
Swap(this, "Swap", false),
MapDiagram(this, "Controller", controller_xpm),
MapInstructions(this, "Press a button on your joystick to map it."),
MapTable(this, 10, false),
ImgInd(this, def_xpm),
ImgX(this, del_xpm),
MapButtonBox(this),
Map0(this, 0, MapTable), Map1(this, 1, MapTable),
Map2(this, 2, MapTable), Map3(this, 3, MapTable)
{
   SetGridSize(4, 2, true);
   SetStretch(Grow, Fill);
   Label = new fr_Label(this, Name, joystick_xpm);
	
   UseJoystick.NotArgs << "-j" <<  fr_CaseInsensitive << "-nojoy";
   UseJoystick.SetTooltip("Read input from joystick(s). (analog joystick polling can slow things down)");
   UseJoystick.AddListener(this);
   Pack(UseJoystick);

   Swap.Args << fr_CaseInsensitive << "-swapjoypads" << "-sw" << "-s";
   Swap.SetKeyStroke("6");
   Swap.SetTooltip("Cross Joystick 1 over to Joystick 2 and vice-versa");
   Pack(Swap);

   Old.Args << fr_CaseInsensitive << "-old" << "-o";
   Old.SetTooltip("Use old-style joystick emulation");   
   Pack(Old);

   fr_MenuItem *MI;
   MI = new fr_MenuItem(&Buttons, "2 buttons");
   MI->SetTooltip("for a 2-button joystick/gamepad");
   MI = new fr_MenuItem(&Buttons, "4 buttons");
   MI->Args << fr_CaseInsensitive << "-4" << "-four";
   MI->SetTooltip("for a 4-button joystick/gamepad");
   MI = new fr_MenuItem(&Buttons, "6 buttons");
   MI->Args << fr_CaseInsensitive << "-6" << "-six";
   MI->SetTooltip("for a 6-button joystick/gamepad");
        
   Pack(Buttons, 3, 0, 4, 1);
	
   DevBtn.SetTooltip("Alter the joystick devices.");
   DevBtn.AddListener(this);
   Pack(DevBtn, 3, 0, 4, 1);

   DevWindow.SetGridSize(1, S9X_JS_COUNT + 1, false);
   DevWindow.SetPadding(4, 4);
   DevWindow.Pad();
   DevWindow.AddListener(this);
   
   devJS[0] = &devJS0;	Map[0] = &Map0;
   devJS[1] = &devJS1;	Map[1] = &Map1;
   devJS[2] = &devJS2;	Map[2] = &Map2;
   devJS[3] = &devJS3;	Map[3] = &Map3;
   
   for(int d=0; d<S9X_JS_COUNT; d++) {
      devJS[d]->AddLabel(JS_Labels[d]);
      devJS[d]->Args << fr_CaseInsensitive << devJSargs[d];
      devJS0.SetTooltip("Select the device to use for this joystick");
      DevWindow.Pack(*(devJS[d]));
   }
   DevWindow.Pack(DevClose, 2, 1);
   DevClose.AddListener(this);

   MapBtn.SetTooltip("Map the buttons for your joystick(s)");
   MapBtn.AddListener(this);
   SetStretch(Fill, Fill);
   Pack(MapBtn, 4, 1);

   MapWindow.SetGridSize(4, 4, false);
   MapWindow.SetPadding(4, 4);
   MapWindow.Pad();
   MapWindow.AddListener(this);

   BtnInd = &ImgInd;
   BtnX = &ImgX;
   MapTable.SetHeaders(MapHeaders);
   int i;
   for(i=0; i<10; i++) {
      MapTable.SetColumnWidth(i, MapHeaderWidths[i]);
      if(i<ButtonOffset) MapTable.SetColumnAlign(i, fr_DataTable::Left);
      else		 MapTable.SetColumnAlign(i, fr_DataTable::Center);
      
   }
   for(i=0; i<2*S9X_JS_COUNT; i++)
     MapTable.AddRow(EmptyRow);
   SetTableSelection();
   MapTable.SetSize(360, S9X_JS_COUNT * 40);
   
   MapCloseBtn.AddListener(this);
   MapButtonBox.AddButton(MapCloseBtn, true);
   
   MapWindow.Pack(MapDiagram, 0, 0, 4, 1);
   MapWindow.Pack(MapInstructions, 0, 3, 2, 4);
   MapWindow.Pack(MapTable, 0, 1, 4, 3);
   MapWindow.Pack(MapButtonBox, 2, 3, 4, 4);
   
   Frame();
}

void s9x_Joystick::SetToDefaults() {
   UseJoystick.SetToDefault();
   Swap.SetToDefault();
   Old.SetToDefault();
   Buttons.SetToDefault();
   for(int i=0; i<S9X_JS_COUNT; i++) {
      devJS[i]->SetToDefault();
      Map[i]->SetToDefault();
   }
}

void s9x_Joystick::Set9xVersion(float version) {
   bool cond = (version >= 1.10);
   Old.EnableIf(!cond);
   Buttons.EnableIf(!cond);
   Old.SetVisibility(!cond);
   Buttons.SetVisibility(!cond);
   DevBtn.SetVisibility(cond);
	
   for(int i=0; i<S9X_JS_COUNT; i++)
     devJS[i]->EnableIf(cond);
   MapBtn.SetEditable(cond);
}

void s9x_Joystick::SiftArgs(fr_ArgList& L) {
   UseJoystick.SiftArgs(L);
   Swap.SiftArgs(L);
   Old.SiftArgs(L);
   Buttons.SiftArgs(L);
   for(int i=0; i<S9X_JS_COUNT; i++) {
      devJS[i]->SiftArgs(L);
      Map[i]->SiftArgs(L);
   }
}

void s9x_Joystick::CompileArgs(fr_ArgList& L) {
   if(!UseJoystick.GetState()) {
      UseJoystick.CompileArgs(L);
      return;
   };
   Swap.CompileArgs(L);
   Old.CompileArgs(L);
   Buttons.CompileArgs(L);
   for(int i=0; i<S9X_JS_COUNT; i++) {
      devJS[i]->CompileArgs(L);
      Map[i]->CompileArgs(L);
   }
}

void s9x_Joystick::EventOccurred(fr_Event*e) {
   bool s;
   int i;

   if(e->Is(UseJoystick)) {
      s = UseJoystick.GetState();
      Swap.SetEditable(s);
      Buttons.SetEditable(s);
      Old.SetEditable(s);
      DevBtn.SetEditable(s);
      MapBtn.SetEditable(s);
   } else if(e->Is(DevBtn)) {
      DevWindow.SetVisibility(true);
   } else if(e->Is(DevClose)||e->Is(DevWindow, fr_Destroy)) {
      DevWindow.SetVisibility(false);
   } else if(e->Is(MapBtn)) {
      MapWindow.SetVisibility(true);
      for(i=0; i<S9X_JS_COUNT; i++)
	 Map[i]->OpenDevice(devJS[i]->GetFileName());
   } else if(e->Is(MapCloseBtn)||e->Is(MapWindow, fr_Destroy)) {
      MapWindow.SetVisibility(false);
      for(i=0; i<S9X_JS_COUNT; i++)
	Map[i]->CloseDevice();
   };
}

void s9x_Joystick::SetTableSelection() {
   for(int r=0; r<S9X_JS_COUNT; r++) {
      MapTable.Select(2*r);
      MapTable.Deselect(2*(r+1));
   }
}

/* ############################# s9x_JoyMap ############################### */

s9x_JoyMap::s9x_JoyMap(s9x_Joystick*parent, int j, fr_DataTable& m):
fr_Option(parent, JS_Labels[j]),
Index(j), Row(2*j),
MapTable(m)
{
   device_fd = device_id = -1;
   IsVisible = false;
   Args << MapArgs[j];
   SelectColumn(ButtonOffset);
}

/// Set the mapped value for a button
void s9x_JoyMap::SetBtn(int btn, int val) {
   MapTable.SetCell(Row, ButtonOffset+btn, MapValues[val<16?val:15]);
}

/// @return the mapped value for a button
int s9x_JoyMap::GetBtn(int btn) {
   char *valstr = MapTable.GetCell(Row, ButtonOffset+btn);
   if(valstr)
     return atoi(valstr);
   return -1;
}

/// Set the mapped value for the selected button
void s9x_JoyMap::SetSelectedBtnVal(int val) {
   SetBtn(SelectedBtn, val);
}

/// @return the mapped value for the selected button
int s9x_JoyMap::GetSelectedBtnVal() {
   return GetBtn(SelectedBtn);
}

/// Reset the button map to the snes9x default map
void s9x_JoyMap::SetToDefault() {
   for(int i=0; i<8; i++)
     SetBtn(i, atoi(DefaultMap[i]));
}

/// Check to see if the current map is the default map
bool s9x_JoyMap::IsDefault() {
   for(int i=0; i<S9X_JS_COUNT; i++)
     if(GetBtn(i) != atoi(DefaultMap[i]))
       return false;
   return true;
}

void s9x_JoyMap::SiftArgs(fr_ArgList& L) {
   char *a;
   int m = Args.MatchArgs(L) + 1;
   if (m<1) return;
   for(int i=0; i<8; i++) {
      a = L[m+i];
      if(a) {
	 SetBtn(i, atoi(a));
	 L.Mark(m+i);
      } else {
	 cerr << "not enough parameters (8) for " << Args.GetPreferredArg();
	 break;
      };
   };
}

void s9x_JoyMap::CompileArgs(fr_ArgList& L) {
   if (IsDefault()) return;
   L << Args.GetPreferredArg();
   for(int i=0; i<8; i++)
     L << MapTable.GetCell(Row, i+ButtonOffset);
}

void s9x_JoyMap::SelectColumn(int col) {
   if(col < ButtonOffset) return;
   SelectedBtn = col - ButtonOffset;
   fr_Image *I;
   if(device_fd>=0)	I = BtnInd;
   else			I = BtnX;
   for(int i=2; i<10; i++) {
      if(i==col)
	MapTable.SetCell(Row+1, i, "^", I);
      else
	MapTable.SetCell(Row+1, i, "");
   };
   MapTable.ShowRow(0);
}

void s9x_JoyMap::MoveSelector(int dir) {
   SelectedBtn += dir;
   if(SelectedBtn>7)
     SelectedBtn = 0;
   if(SelectedBtn<0)
     SelectedBtn = 0;
   SelectColumn(SelectedBtn + ButtonOffset);
}

void s9x_JoyMap::AlterSelectedBtn(int dir) {
   int b;
   b = GetSelectedBtnVal();
   b += dir;
   if(b<0)
     b = 0;
   if(b>15)
     b = 15;
   SetSelectedBtnVal(b);
}

void s9x_JoyMap::EventOccurred(fr_Event*e) {
   
}

int s9x_JoyMap::OpenDevice(char *dev) {
#ifdef S9X_INCLUDE_JOYSTICK   
   int version, v1, v2;
   if( !dev || !fr_Exists(dev)) {
      MapTable.SetCell(Row, 1, "does not exist");
      SelectColumn(ButtonOffset);
      return -1;
   };
   axes = buttons = version = v1 = v2 = 0;
   CloseDevice();
   device_fd = open(dev, O_RDONLY);
   if(device_fd >= 0) {
      axes = buttons = 2;
      version = 0x000800;
      ioctl(device_fd, JSIOCGVERSION, &version);
      ioctl(device_fd, JSIOCGAXES, &axes);
      ioctl(device_fd, JSIOCGBUTTONS, &buttons);
      ioctl(device_fd, JSIOCGNAME(40), JSname);
      v1 = (version >> 16);
      v2 = (version >> 8) & 0xFF;
   } else
     strcpy(JSname, "n/a");
   if( (v1 < 1) || (v2 < 1) ) { // joystick driver too old
      if(device_fd >= 0)
	close(device_fd);
      device_fd = -1;
   };
   if(device_fd >= 0) {
      device_id = gdk_input_add(device_fd, GDK_INPUT_READ,
				s9x_Joystick__DeviceEvent, this);
      SelectColumn(ButtonOffset);
   }
#else
   strcpy(JSname, "N/A");
#endif
   MapTable.SetCell(Row, 0, MapValues[Index]);
   MapTable.SetCell(Row, 1, JSname);
   SelectColumn(ButtonOffset);
   return device_fd;
}

void s9x_JoyMap::CloseDevice() {
#ifdef S9X_INCLUDE_JOYSTICK   
   if(device_id >= 0)
     gdk_input_remove(device_id);
   if(device_fd >= 0)
     close(device_fd);
#endif
   device_fd = device_id = -1;
}

void s9x_Joystick__DeviceEvent(void*JMptr, int fd, GdkInputCondition IC) {
#ifdef S9X_INCLUDE_JOYSTICK   
   static __u32 last_time = 0;
   struct js_event js;
   int size, dir;
   s9x_JoyMap*JM;
	
   if (IC != GDK_INPUT_READ) return;
   JM = (s9x_JoyMap*)JMptr;
   size = sizeof(struct js_event);
   if (read(fd, &js, size) != size) return;
   
   switch(js.type) {
    case JS_EVENT_AXIS:
      if(js.time - last_time < 80) //enforce a delay
	js.value = 0;
      last_time = js.time;
      js.value /= (1<<14);
      if(js.value)
	dir = js.value / abs(js.value);
      else
	dir = 0;
      if(js.number == 0)
	JM->MoveSelector(dir);
      else if(js.number == 1)
	JM->AlterSelectedBtn(-dir);
      break;
    case JS_EVENT_BUTTON:
      if(js.value)
	JM->SetSelectedBtnVal(js.number);
      else
	JM->MoveSelector(1);
      break;
   };
#endif
}
