// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/common/class_loader/Class.cpp,v 1.15 2002/01/10 22:50:53 ssubram5 Exp $
//


#include "platform.h"
#include <string.h>
#include <assert.h>
#include <stdlib.h>

#ifdef ORP_POSIX
#include "Loader_Result.h"
#endif

#include "Class.h"
#include "environment.h"
 
#include "gc_for_orp.h"
 
#include "orp_utils.h"
#include "nogc.h"
#include "compile.h"



//
// private static variable containing the id of the next class
// access to this needs to be thread safe; also, this will have to 
// change if we want to reuse class ids
//
// an id of 0 is reserved to mean null; therefore, ids start from 1
//
unsigned class_next_id = 1;

//#ifdef CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH

bool bootstrapped = false;
Class *class_class = NULL;
unsigned sizeof_class_class = 0;

#endif

Class *init_class_fields(Class *clss, const String *name);

Class *new_class(const String *name)
{
//#if CLASS_LOADER_BOOTSTRAP
#ifndef OLD_VERSION_CLASSPATH
	Class *clss = NULL;
	unsigned int sizeof_class_in_heap = 0;
	// It's possible that name is null when defining class on the fly
	if (name && (strcmp(name->bytes, "java/lang/Class") == 0) ) { 
	
		if (!bootstrapped) {
			// Forcible bootstrap.
			assert(ORP_Global_State::loader_env->JavaLangClass_Class == 0);
			assert(class_class == NULL);
			clss = (Class *) malloc(sizeof(Class) + 500);
			assert(clss);
			assert(sizeof_class_class == 0);
			class_class = clss;
			return init_class_fields(clss, name);
		} else { 
			assert(class_class);
			free(class_class);
			class_class = NULL;
		}
	}
	assert(bootstrapped == true);
	sizeof_class_in_heap = sizeof_class_class;
	assert(sizeof_class_class > 0);
#else
	unsigned int sizeof_class_in_heap = 
			(( (sizeof(Class) + (GC_OBJECT_ALIGNMENT - 1)) 
				/ GC_OBJECT_ALIGNMENT) * GC_OBJECT_ALIGNMENT) + OBJECT_HEADER_SIZE;
	Class *clss;
#endif

	//
	// New: Put these objects in the GC Heap:
	//

    if(ORP_Global_State::loader_env->JavaLangClass_Class) {
 
        // the class property determines if it is a pinned object.
        clss = 
            (Class *)gc_malloc (
                ORP_Global_State::loader_env->JavaLangClass_Class->instance_data_size, 
                (Partial_Reveal_VTable *)ORP_Global_State::loader_env->JavaLangClass_Class->p_vtable);
//#ifdef NEW_JAVA_LANG_CLASS_LAYOUT
#ifndef OLD_VERSION_CLASSPATH
#else
        assert (sizeof_class_in_heap
                == get_instance_data_size(ORP_Global_State::loader_env->JavaLangClass_Class));
#endif
        // Make sure the high bit is set.
        assert (sizeof_class_in_heap != ORP_Global_State::loader_env->JavaLangClass_Class->instance_data_size);
        assert (get_prop_pinned(ORP_Global_State::loader_env->JavaLangClass_Class->class_properties));
 
    } else {
        // Classes loaded before java.lang.Class must use this API, because the
        // vtable for java.lang.Class hasn't been constructed yet.
        // Currently only three classes should be allocated through this call:
        // java.lang.Object, java.io.Serializable and java.lang.Class.
 

        clss = (Class *)gc_pinned_malloc_noclass(sizeof_class_in_heap);
 
    }
    if (ORP_Global_State::loader_env->JavaLangClass_Class != NULL) {
        // ww -- during sys init, this field is zero
        clss->p_vtable =
            ORP_Global_State::loader_env->JavaLangClass_Class->p_vtable;
    }
	return init_class_fields(clss, name);
}

Class *init_class_fields(Class *clss, const String *name) {

    clss->n_instance_refs = 0;

#ifdef POINTER64
	clss->alignment = 8;
#else
#ifdef EIGHT_BYTE_ALIGN
	clss->alignment = 8;
#else
    clss->alignment = 4;
#endif
#endif
 
    clss->class_properties = 0;
    clss->gc_information = 0;
 
    clss->id = class_next_id++;
	clss->class_loader = 0;
	clss->has_finalizer = 0;
	clss->access_flags = 0;
	clss->name = name;
	clss->state = ST_Start;
	clss->table_next=NULL;
	clss->super_class = NULL;
//#ifdef FAST_INSTOF //::
	clss->depth = 0;
/*	memset(&clss->superclasses,
		   0,
		   sizeof(clss->superclasses));*/
//#endif
	clss->superinterfaces = NULL;
    clss->static_data_size = 0;
    clss->static_method_size = 0;
	clss->instance_data_size = 0;
	clss->n_virtual_method_entries = 0;
    clss->n_intfc_method_entries = 0;
	clss->vtable_descriptors = NULL;
	clss->intfc_table_descriptors = NULL;
	clss->n_intfc_table_entries = 0;
	clss->finalize_method = NULL;
	clss->static_initializer = NULL;
	clss->package = NULL;
	clss->next_class_in_package = NULL;
    clss->class_object = 0;  // added this to support hello world.
	clss->n_fields = clss->n_methods = clss->n_superinterfaces = clss->cp_size = 0;
	clss->const_pool = NULL;
	clss->methods = NULL;
	clss->n_dimensions = 0;
	clss->is_primitive = 0;
	clss->is_array = 0;
	clss->is_array_of_primitives = 0;

	clss->array_base_class = NULL;
	clss->array_element_class = NULL;
	clss->class_file_name = NULL;
	clss->src_file_name = NULL;
	clss->static_data_block = NULL;
	clss->static_method_block = NULL;
	clss->vtable = NULL;

#ifdef CLI_TESTING
    clss->is_value_type = FALSE;
    clss->is_enum = FALSE;
    clss->cli_type_def = NULL;
    clss->super_type = NULL;
#ifdef CLI_OCL
	clss->system_type = NULL;
#endif
#endif

#ifdef _DEBUG
    clss->printed_in_dump_jit = FALSE;
#endif

#ifdef ORP_STATS
    clss->num_class_init_checks = 0;

    // For subclasses of java.lang.Throwable only.
    clss->num_throws = 0;
    clss->num_instanceof_slow = 0;
#endif

    clss->registered_natives = NULL;
	return clss;
} //new_class



//
// This function doesn't check for fields inherited from superclasses.
//
Field *class_lookup_field(Class *clss,Signature *sig)
{
	for (unsigned i=0; i<clss->n_fields; i++) {
		if (clss->fields[i].get_signature() == sig)
			return &clss->fields[i];
	}
	return NULL;
} //class_lookup_field



Field *class_lookup_field_recursive(Class *clss,
                                    const char *name,
                                    const char *descr)
{
    String *field_name =
        ORP_Global_State::loader_env->string_pool.lookup(name);
    String *field_descr =
        ORP_Global_State::loader_env->string_pool.lookup(descr);
	Signature *sig =
        ORP_Global_State::loader_env->sig_table.lookup(field_name, field_descr);

    Field *f = 0;
    for(; clss && !f; clss = clss->super_class) {
        f = class_lookup_field(clss, sig);
        if(f) {
            return f;
        }

        unsigned n_intf = clss->n_superinterfaces;
		for(unsigned i = 0; i < n_intf; i++) {
            Class *intf = clss->superinterfaces[i].clss;
            assert(intf);
            assert(class_is_interface(intf));
            f = class_lookup_field(intf, sig);
            if(f) {
                return f;
            }
        }
    }
    return f;
} //class_lookup_field_recursive



Method *class_lookup_method(Class *clss,Signature *sig)
{
	for (unsigned i=0; i<clss->n_methods; i++) {
		if (clss->methods[i].get_signature() == sig)
			return &clss->methods[i];
	}
	return NULL;
} //class_lookup_method



Method *class_lookup_method_recursive(Class *clss, Signature *sig)
{
    Method *m = 0;
    for(; clss && !m; clss = clss->super_class) {
        m = class_lookup_method(clss, sig);
    }
    return m;
} //class_lookup_method_recursive



Method *class_lookup_method_recursive(Class *clss,
                                      const char *name,
                                      const char *descr)
{
    String *field_name =
        ORP_Global_State::loader_env->string_pool.lookup(name);
    String *field_descr =
        ORP_Global_State::loader_env->string_pool.lookup(descr);
	Signature *sig =
        ORP_Global_State::loader_env->sig_table.lookup(field_name, field_descr);

    return class_lookup_method_recursive(clss, sig);
} //class_lookup_method_recursive



Method::Method()
{
    _code_block = 0;
    _code_block_size = 0;
    _jit_info_block  = NULL;
    _jit_info_block_size = 0;

    //
    // _vtable_patch may be in one of three states:
    // 1. NULL -- before a vtable is initialized or after all patching is done.
    // 2. Points to a vtable entry which must be patched.  
    //    This state can be recognized because *_vtable_patch == _code
    // 3. Otherwise it points to a list containing multiple vtable patch info.
    //
    _vtable_patch = 0;
    // Create a stub which will JIT the method on its first invocation.

    char *stub = create_compile_me_stub(this);

    assert(stub);
    _compile_me_stub = stub;
    _code = stub;

    _state = ST_NotCompiled;
    _jit = NULL;
    _jits = NULL;
    _side_effects = MSE_Unknown;
    inline_records = 0;

	_index = 0;
	_max_stack=_max_locals=_n_exceptions=_n_handlers=0;
	_exceptions = NULL;
	_byte_code_length = 0;
	_byte_codes = NULL;
	_handlers = NULL;
	_flags.is_init = 0;
    _flags.is_clinit = 0;
    _flags.is_overridden = 0;
    _flags.is_finalize = 0;
    _flags.is_nop = FALSE;

    _line_number_table = NULL;
	_line_number_entries = NULL;
	_local_vars_table = NULL;
} //Method::Method



////////////////////////////////////////////////////////////////////
// begin Class_Table

Class_Table::Class_Table()
{
	// initialize table to empty
	for (unsigned i=0; i<CLASS_TABLE_SIZE; i++) {
		_table[i] = NULL;
	}
} //Class_Table::Class_Table



Class *Class_Table::lookup(const String *name)
{
	// hash on string name address
	unsigned h = (((unsigned)name)>>2)%CLASS_TABLE_SIZE;
    for(Class *clss = _table[h]; clss != NULL; clss = clss->table_next) {
        if(clss->name == name)
			return clss;
	}
	// signature not found in table
	return NULL;
} //Class_Table::lookup



void Class_Table::insert(Class *clss)
{
	const String *name = clss->name;
    if(!lookup(name)) {
    	// hash on string name address
    	unsigned h = (((unsigned)name)>>2) % CLASS_TABLE_SIZE;
	    clss->table_next = _table[h];
    	_table[h] = clss;
    }
} //Class_Table::insert



Class *Class_Table::_remove(Class *curr,Class *elem)
{
	if (curr == NULL)
		return NULL;
	if (curr == elem)
		return elem->table_next;
	curr->table_next = _remove(curr->table_next,elem);
	return curr;
} //Class_Table::_remove



void Class_Table::remove(Class *clss)
{
	if (clss == NULL)
		return;
	// hash on string name address
	const String *name = clss->name;
	unsigned h = (((unsigned)name)>>2)%CLASS_TABLE_SIZE;
	_table[h] = _remove(_table[h],clss);
} //Class_Table::remove



// A simple iterator.
// It is used for GC at the moment, but it should be replaced with
// something that would allow GC of classes.
Class **Class_Table::get_first_class()
{
    for(unsigned i = 0; i < CLASS_TABLE_SIZE; i++) {
        if(_table[i] != NULL)
            return &(_table[i]);
    }
    return NULL;
} //Class_Table::get_first_class



Class **Class_Table::get_next_class(Class *prev)
{
    assert(prev);
    // hash on string name address
    const String *name = prev->name;
    unsigned h = (((unsigned)name)>>2)%CLASS_TABLE_SIZE;
    for (Class *clss = _table[h]; clss != NULL; clss = clss->table_next) {
        if (clss == prev) {
            if(clss->table_next) {
                return &(clss->table_next);
            } else {
                for(unsigned i = h + 1; i < CLASS_TABLE_SIZE; i++) {
                    if(_table[i] != NULL)
                        return &(_table[i]);
                }
                return NULL;
            }
        }
    }
    assert(0);
    return NULL;
} //Class_Table::get_next_class



// end Class_Table
////////////////////////////////////////////////////////////////////



//
// !!! fix this !!!
//
Intfc_Table *create_intfc_table(unsigned n_entries)
{
	unsigned size = INTFC_TABLE_OVERHEAD + (n_entries * sizeof(Intfc_Table_Entry));
	Intfc_Table *table = (Intfc_Table*)gc_malloc_fixed_data_C(size);
	memset(table,0,size);
	return table;
}



#ifdef POINTER64
#define VTABLE_GC_HEADER 16
#else
#define VTABLE_GC_HEADER 8
#endif

VTable *create_vtable(Class *p_class, unsigned n_vtable_entries)
{
    unsigned vtable_size = VTABLE_GC_HEADER + VTABLE_OVERHEAD + n_vtable_entries * sizeof(void *);

    void *p_gc_hdr = malloc(vtable_size);
    memset(p_gc_hdr, 0, vtable_size);
    *((int *)p_gc_hdr) = p_class->n_instance_refs;

    VTable *vtable = (VTable *)(((char *)p_gc_hdr) + VTABLE_GC_HEADER);

    gc_class_loaded((Partial_Reveal_VTable *) vtable);

//#ifdef FAST_INSTOF //::
	if(p_class && p_class->super_class) {
		p_class->depth = p_class->super_class->depth + 1;
		memcpy(&vtable->superclasses,
			   &p_class->super_class->vtable->superclasses,
			   sizeof(vtable->superclasses));
		for(int i = 0; i < MAX_FAST_INSTOF_DEPTH; i++) {
			if(vtable->superclasses[i] == NULL) {
				vtable->superclasses[i] = p_class;
				break;
			}
		}
	} 
//#endif

	return vtable;
} //create_vtable


