// Copyright (C)  2000 Intel Corporation.  All rights reserved.
//
// $Header: /usr/development/orp/orp/arch/ia32/ia32_o1_jit/jcfg.h,v 1.2 2001/08/13 09:59:58 xhshi Exp $
//


/* Goal: Produce a simple flow graph for the purposes of global
   register allocation.  It will not be hierarchical in any way.  We
   want to construct a graph where each node contains the start and
   length of the corresponding basic block, as well as a list of the
   incoming and outgoing edges.  For the purposes of the register
   allocation algorithm, we will only really care about the backwards
   (incoming) edges.

   Along with that information, we also need to know whether a given
   variable is defined in the block, whether it is used, and whether
   the first def happens before the first use (if the variable is both
   defined and used in the basic block).

   The algorithm for building the flow graph is the following.
   Iterate through the byte code and locate the beginning of each
   basic block.  The beginning of a basic block is the statement
   following a branch, or the target of a branch, or the first byte
   code.  After this traversal, we will also know the total number of
   basic blocks.  For each statement, we also record whether it is a
   branch statement and/or a branch target.  At the same time, we get
   a count of the number of edges going out of each statement and the
   number of edges coming into each statement.  Note that all edges
   will come into the first statement of a basic block, and all edges
   will go out of the last statement of a basic block.

   After we get the count of the number of edges and the number of
   basic blocks, we can allocate an array of CFG nodes, an array of
   forward edges, and an array of backward edges.  The idea is to
   avoid using linked lists by putting information into arrays
   instead.

   Finally, we iterate through each basic block.  We quickly find the
   branches (which are always at the last statement of the basic
   block), and set up the forward and backward edges in the flow
   graph.

   I'm doing a few tricks to keep from having to malloc lots of space.
   First, I get a count of the number of basic blocks, so that I can
   allocate all of them at once in an array.  Second, I do the same
   sort of thing to avoid allocating an array of edges in each node of
   the flow graph.  Instead, I keep a single array of forward edges
   and a single array of backward edges, and each flow graph node
   points to a range within that array.  Note that each of these
   arrays is "sorted" so that a flow graph node can encode its edges
   with a simple range.  It's easy to set this up for the forward
   edges, but harder for the backward edges.

   ================================================================

   Pass 1: Count basic blocks, mark the beginning of each basic block,
   count edges.  Non-recursive algorithm.

   If the previous statement is an unconditional branch, mark this
   statement as a new basic block (provided of course that this
   statement is not already marked as the beginning of a basic block).
   Unconditional branches include RETURN and THROW type statements,
   but not INVOKE statements.

   If this statement is a branch statement, then add all necessary
   edges in the following manner:

     If it's a forward branch, and the target is not already marked as
     the beginning of a basic block, then mark the target as a new
     basic block.  If it's a backward branch, and the target is not
     already marked as the beginning of a basic block, then mark the
     target as a new basic block, and add an edge from the previous
     statement, since we MUST be splitting an existing basic block.

   In the summary[], mark the beginning of each statement, so that
   it's easy to add the extra edge when splitting a basic block.

  */

#ifndef _JCFG_H_
#define _JCFG_H_

#include "Mem_Manager.h"
#include "bit_vector.h"
#include "jit.h"

#define CFG_FORWARD_EDGES 0  // whether to record forward edges in CFG

// calculate a signed integer offset based on an array of unsigned chars.
#define JCFG_OFFSET2(array,idx) ((((char*)array)[idx] << 8) | (array)[(idx)+1])
#define JCFG_OFFSET4(array,idx) (((array)[idx] << 24) | ((array)[(idx)+1] << 16) | ((array)[(idx)+2] << 8) | (array)[(idx)+3])

struct Jcfg_node
{
  const unsigned char *bc_start;
  // bc_end is the beginning of the last instruction of the BB.
  const unsigned char *bc_end;
  unsigned bc_length;
#if CFG_FORWARD_EDGES
  int *out_edges;  // points somewhere within forward_edge_array
  int num_out_edges;
#endif // CFG_FORWARD_EDGES
  int *in_edges;  // points somewhere within backward_edge_array
  int num_in_edges;
  Bit_Vector *loaded_vars, *stored_vars;
  short allocation[4];  // local variable assigned to each callee-save register

  unsigned char registers_available;  // a bit vector
  unsigned char visited;  // whether this node has been visited in DFS

// cur_out_edge_idx and cur_in_edge_idx are used only temporarily
// while constructing the CFG.
#if CFG_FORWARD_EDGES
  int cur_out_edge_idx;
#endif // CFG_FORWARD_EDGES
  int cur_in_edge_idx;
};

struct Jcfg_bytecode_info
{
  unsigned short num_in_edges;
#if CFG_FORWARD_EDGES
  unsigned short num_out_edges;
#endif // CFG_FORWARD_EDGES
  unsigned is_stmt_start:1;
  unsigned is_basic_block_start:1;
  unsigned is_load_or_store:1;  // but not counting floating point
};

class Jcfg
{
public:
  Jcfg(const unsigned char *bc, unsigned code_length, unsigned maxLocals,
       Mem_Manager &m, void *methodHandle, void *classHandle);
  ~Jcfg() {}
  int ok_to_register_allocate() { return success; }
  void register_allocate();
  unsigned char callee_save_registers_used();
  unsigned char available_registers_in_BB(const unsigned char *);
  unsigned char register_assigned_to_var(int varnum);
  void set_call_site_info(Jit_Method_Info *mi, unsigned size);
  unsigned var_in_register(unsigned reg_no, const unsigned char *where);
  unsigned var_in_register(unsigned reg_no, int where);
  unsigned char registers_available_in_range(const unsigned char *bc,
                                             unsigned size, int &hint);
  unsigned char registers_available_in_codegen_bb(const unsigned char *bc, int &hint);
  int var_is_live_on_entry(int varnum);

  void *operator new(size_t sz,Mem_Manager& m) { return m.alloc(sz); }

private:
  struct Jcfg_node *cfg_node_array;  // size is basic_block_count
#if CFG_FORWARD_EDGES
  int *forward_edge_array;
#endif // CFG_FORWARD_EDGES
  int *backward_edge_array;
  int edge_array_size;

  int *refcount;  // reference count for each local variable
  Bit_Vector *is_long;  // whether a local variable needs 2 registers

  int is_jsr_or_ret;
  unsigned basic_block_count;
  int success;
  Mem_Manager *mem_manager;
  unsigned num_locals;
  unsigned bc_length;
  const unsigned char *bytecodes;

  void *m_handle, *c_handle;

  struct Jcfg_bytecode_info *summary;

  /* register_allocations is an array of bit vectors.  The number of
     array elements is equal to the number of local variables.  For
     each local variable, the bit vector gives the register (or
     registers) allocated to that variable.  */
  unsigned char *register_allocations;
  /* registers_used is the union of all registers used for global
     register allocation.  These registers should be saved at the
     beginning and restored at the end of the method.  The compiler
     can use this value ANDed with the available registers in a CFG
     node to find which registers can be given to the register manager
     for local register allocation in that basic block.  */
  unsigned char registers_used;
  /* var_is_register_allocated tells whether the variable is
     register-allocated in the method.  It is needed in the method_info for
     garbage collection. */
  Bit_Vector *var_is_register_allocated;

  void count_edge(unsigned src, unsigned dest);
  void add_edge(unsigned src, unsigned dest);
  void add_edge(unsigned src, const unsigned char *dest);
  void create_summary(const unsigned char *first_bc, unsigned code_length);
  unsigned assign_register(unsigned result, unsigned varnum,
                           unsigned char &rbitvec,
                           unsigned basic_block_count,
                           struct Jcfg_node *cfg_node_array);
  void fixup_allocation(int varnum);
  void print_cfg();
  int find_node(const unsigned char *where);
};

#endif // _JCFG_H
