/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.j2ee.refactoring.changeparam;

import java.io.IOException;
import java.text.MessageFormat;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import javax.jmi.reflect.RefObject;
import org.netbeans.api.project.FileOwnerQuery;
import org.netbeans.api.project.Project;
import org.netbeans.jmi.javamodel.ClassDefinition;
import org.netbeans.jmi.javamodel.JavaClass;
import org.netbeans.jmi.javamodel.Method;
import org.netbeans.jmi.javamodel.Type;
import org.netbeans.modules.j2ee.dd.api.ejb.EjbJar;
import org.netbeans.modules.j2ee.dd.api.ejb.EnterpriseBeans;
import org.netbeans.modules.j2ee.dd.api.ejb.Entity;
import org.netbeans.modules.j2ee.dd.api.ejb.MethodParams;
import org.netbeans.modules.j2ee.dd.api.ejb.Query;
import org.netbeans.modules.j2ee.dd.api.ejb.QueryMethod;
import org.netbeans.modules.j2ee.dd.api.ejb.Session;
import org.netbeans.modules.j2ee.refactoring.Utility;
import org.netbeans.modules.javacore.api.JavaModel;
import org.netbeans.modules.javacore.internalapi.ExternalChange;
import org.netbeans.modules.javacore.internalapi.JavaMetamodel;
import org.netbeans.modules.refactoring.api.AbstractRefactoring;
import org.netbeans.modules.refactoring.api.ChangeParametersRefactoring;
import org.netbeans.modules.refactoring.api.ChangeParametersRefactoring.ParameterInfo;
import org.netbeans.modules.refactoring.api.Problem;
import org.netbeans.modules.refactoring.api.RefactoringSession;
import org.netbeans.modules.refactoring.spi.RefactoringElementImplementation;
import org.netbeans.modules.refactoring.spi.RefactoringElementsBag;
import org.openide.ErrorManager;
import org.openide.filesystems.FileObject;
import org.openide.util.NbBundle;

/**
 *
 * @author Martin Grebac
 */
public final class EjbJarChangeParamRefactoring {
    
    private static final ErrorManager err = ErrorManager.getDefault().getInstance("org.netbeans.modules.j2ee.refactoring.changeparam");   // NOI18N

    /** Trying to change parameters of finder method in bean implementation class, and we should check also ejb-jar for references */
    private boolean isCmpImplFinder = false;

    /** Trying to change parameters of finder method in local home interface, and we should check also ejb-jar for references */
    private boolean isCmpLocalHomeFinder = false;

    /** Trying to change parameters of finder method in home interface, and we should check also ejb-jar for references */
    private boolean isCmpHomeFinder = false;

    /* stores collection of interface refactorings to be invoked */
    ArrayList ifaceRefactors = new ArrayList();

    /* stores collection of implementation class refactorings to be invoked */
    ArrayList implRefactors = new ArrayList();
    private static final String METHOD_EJBCREATE = "ejbCreate";

    public EjbJarChangeParamRefactoring() { }
    
    public Problem preCheck(RefObject refObject) {
        Problem[] problem = new Problem[] {null};
        if (refObject instanceof Method) {
            Method method = (Method) refObject;
            Collection emodules = Utility.getRelevantEjbModules(method);
            if ((emodules != null) && (emodules.size() > 0)) {
                // there are ejb modules that can be affected by the change
                ClassDefinition declaringClass = method.getDeclaringClass(); // get affected class
                err.log("classdefinition jc: " + declaringClass);
                if (declaringClass instanceof JavaClass) {
                    FileObject fo = JavaModel.getFileObject(method.getResource());
                    if (Utility.isSubTypeOf(declaringClass, Utility.SESSION_BEAN)) {
                        checkSessionBeanImpl(problem, method, fo, declaringClass);
                    } else if (Utility.isSubTypeOf(declaringClass, Utility.ENTITY_BEAN)) {
                        checkEntityBeanImpl(problem, method, fo, declaringClass);
                    } else if (Utility.isSubTypeOf(declaringClass, Utility.MESSAGE_DRIVEN_BEAN)) {
                        checkMessageDrivenBeanImpl(problem, method, declaringClass);
                    } else if (Utility.isSubTypeOf(declaringClass, Utility.EJB_HOME)) {
                        checkHomeInterface(problem, method, fo, declaringClass, false);
                    } else if (Utility.isSubTypeOf(declaringClass, Utility.EJB_LOCAL_HOME)) {
                        checkHomeInterface(problem, method, fo, declaringClass, true);
                    }
                }
            }
        }
        return problem[0];
    }

    private void checkSessionBeanImpl(Problem[] problem, Method method, FileObject fo, ClassDefinition jc) {
        // check if it's a method from javax.ejb.SesssionBean interface - do not allow such change
        if (Utility.isSignatureFromSessionBeanInterface(method)) {
            addFatalProblem(problem, "TXT_EjbJarMethodInJavaxInterfaceProblemChangeParam",
                    Utility.SESSION_BEAN);
        } else if (!checkCreateInSession(problem, fo, method)) {
            String className = jc.getName();
            // check if it's a home method - such change is allowed, but warnings are displayed that home/localhome methods are changed, too
            Method homeInHome =
                    Utility.getHomeHomeMethodForSessionImplHomeMethod(method, fo, className, false);
            checkMethodInInterface(problem, homeInHome, "home");
            Method homeInLocalHome =
                    Utility.getHomeHomeMethodForSessionImplHomeMethod(method, fo, className, true);
            checkMethodInInterface(problem, homeInLocalHome, "local home");
        }
    }

    private void checkEntityBeanImpl(Problem[] problem, Method method, FileObject fo, ClassDefinition jc) {
        // check if we are trying to rename a method from javax.ejb.EntityBean interface,
        // and do not allow this
        if (Utility.isSignatureFromEntityBeanInterface(method)) {
            addFatalProblem(problem, "TXT_EjbJarMethodInJavaxInterfaceProblemChangeParam",
                    Utility.ENTITY_BEAN);
        } else {
            String className = jc.getName();
            String methodName = method.getName();

            if (methodName.startsWith(METHOD_EJBCREATE)) { //NOI18N
                // check if it's a create method
                // such change is allowed, but warnings are displayed
                // that home/local home/postCreate methods are changed, too
                Method createInHome = Utility.getHomeCreateMethodForEntityImplCreateMethod(method, fo,
                        className, false);
                checkMethodInInterface(problem, createInHome, "home");
                Method createInLocalHome = Utility.getHomeCreateMethodForEntityImplCreateMethod(method, fo,
                        className, true);
                checkMethodInInterface(problem, createInLocalHome, "local home");
                Method postCreateInImpl = Utility.getImplPostCreateMethodForEntityImplCreateMethod(method);
                checkMethodInImpl(problem, postCreateInImpl,
                        "TXT_EjbJarPostCreateMethodWarningChangeParam");
            } else if (methodName.equals("ejbPostCreate")) { //NOI18N
                // check if it's a postcreate method
                // such change is allowed, but warnings are displayed
                // that home/local home methods are changed, too
                Method createInHome = Utility.getHomeCreateMethodForEntityImplPostCreateMethod(method, fo,
                        className, false);
                checkMethodInInterface(problem, createInHome, "home");
                Method createInLocalHome = Utility.getHomeCreateMethodForEntityImplPostCreateMethod(method,
                        fo, className, true);
                checkMethodInInterface(problem, createInLocalHome, "local home");
                Method createInImpl = Utility.getImplCreateMethodForEntityImplPostCreateMethod(method);
                checkMethodInImpl(problem, createInImpl, "TXT_EjbJarCreateMethodWarningChangeParam");
            } else if (methodName.startsWith("ejbFind")) {
                // check if it's a finder method
                // such change is allowed, but warnings are displayed
                // that home/localhome methods are changed, too
                isCmpImplFinder = true;
                Method finderInHome =
                        Utility.getHomeFinderMethodForImplFinderMethod(method, fo, className, false);
                checkMethodInInterface(problem, finderInHome, "home");
                Method finderInLocalHome =
                        Utility .getHomeFinderMethodForImplFinderMethod(method, fo, className, true);
                checkMethodInInterface(problem, finderInLocalHome, "local home");
            } else {
                // check if it's a home method
                // such change is allowed, but warnings are displayed
                // that home/localhome methods are changed, too
                Method homeInHome =
                        Utility.getHomeHomeMethodForEntityImplHomeMethod(method, fo, className, false);
                checkMethodInInterface(problem, homeInHome, "home");
                Method homeInLocalHome =
                        Utility.getHomeHomeMethodForEntityImplHomeMethod(method, fo, className, true);
                checkMethodInInterface(problem, homeInLocalHome, "local home");
            }
        }
    }

    private void checkMessageDrivenBeanImpl(Problem[] problem, Method method, ClassDefinition jc) {
        // check if we are trying to rename a method from javax.ejb.MessageDrivenBean interface,
        // and do not allow this
        if (Utility.isSignatureFromMessageDrivenBeanInterface(method)) {
            addFatalProblem(problem, "TXT_EjbJarMethodInJavaxInterfaceProblemChangeParam",
                    Utility.MESSAGE_DRIVEN_BEAN);
        }
        if (jc.isSubTypeOf(Utility.resolveRealClass(Utility.MESSAGE_LISTENER))) {
            if (Utility.isSignatureFromMessageListenerInterface(method)) {
                addFatalProblem(problem, "TXT_EjbJarMethodInJavaxInterfaceProblemChangeParam",
                        Utility.MESSAGE_LISTENER);

            }
        }
    }

    private void addFatalProblem(Problem[] problem, String resName, String param) {
        String msg = NbBundle.getMessage(EjbJarChangeParamRefactoring.class, resName, param);
        problem[0] = Utility.addProblemsToEnd(problem[0], new Problem(true, msg));
    }

    private void checkHomeInterface(Problem[] problem, Method method, FileObject fo, ClassDefinition classDefinition,
            boolean local) {
        String methodName = method.getName();
        String className = classDefinition.getName();
        if (methodName.startsWith(Utility.PREFIX_CREATE)) {
            // check renaming create method
            // such change is allowed, but warnings are displayed that home/local home methods are changed, too
            Method createInImpl = Utility.getImplCreateMethodForHomeCreateMethod(method, fo, className, local);
            if (checkCreateInSession(problem, fo, createInImpl)) {
                return;
            }
            checkMethodInImpl(problem, createInImpl);
            Method createInHome = Utility.getHomeCreateMethodForHomeCreateMethod(method, fo, className, !local);
            checkMethodInInterface(problem, createInHome);
            Method postCreateInImpl =
                    Utility.getImplPostCreateMethodForHomeCreateMethod(method, fo, className, local);
            checkMethodInImpl(problem, postCreateInImpl);
        } else if (methodName.startsWith("find")) {
            // check renaming some finder method
            // this is allowed with some restrictions and finders in implclass/home need to be renamed, too
            if (local) {
                isCmpLocalHomeFinder = true;
            } else {
                isCmpHomeFinder = true;
            }
            Method finderInImpl = Utility.getImplFinderMethodForHomeFinderMethod(method, fo, className, local);
            checkMethodInImpl(problem, finderInImpl);
            Method finderInHome = Utility.getHomeFinderMethodForHomeFinderMethod(method, fo, className, !local);
            checkMethodInInterface(problem, finderInHome);
        } else {
            // check renaming some home method
            // this is allowed with some restrictions and homes in implclass/home need to be renamed, too
            Method homeInImpl = Utility.getImplHomeMethodForHomeHomeMethod(method, fo, className, local);
            checkMethodInImpl(problem, homeInImpl);
            Method homeInHome = Utility.getHomeHomeMethodForHomeHomeMethod(method, fo, className, !local);
            checkMethodInInterface(problem, homeInHome);
        }
    }

    private boolean checkCreateInSession(Problem[] problem, FileObject fo, Method createInImpl) {
        if (createInImpl == null || !createInImpl.getName().equals(Utility.PREFIX_EJBCREATE)) {
            return false;
        }
        ClassDefinition implClass = createInImpl.getDeclaringClass();
        if (!Utility.isSubTypeOf(implClass, Utility.SESSION_BEAN)) {
            return false;
        }
        List parameters = createInImpl.getParameters();
        if (parameters != null && parameters.size() > 0) {
            return false;
        }
        EnterpriseBeans enterpriseBeans = Utility.getEnterpriseBeansFromDD(fo);
        if (enterpriseBeans == null) {
            return false;
        }
        Session[] sessions = enterpriseBeans.getSession();
        if (sessions == null) {
            return false;
        }
        for (int i = 0; i < sessions.length; i++) {
            Session session = sessions[i];
            String className = implClass.getName();
            if (session.getEjbClass().equals(className)) {
                if (Session.SESSION_TYPE_STATELESS.equals(session.getSessionType())) {
                    String msg = NbBundle.getMessage(EjbJarChangeParamRefactoring.class, "TXT_EjbJarCreateMethodErrorChangeParam");
                    problem[0] = Utility.addProblemsToEnd(problem[0], new Problem(true, msg));
                    return true;
                } else {
                    Method createInHome = Utility.getMethodInHomeClassForImpl(fo, Session.class, className,
                            Utility.PREFIX_CREATE, parameters, false);
                    checkMethodInInterface(problem, createInHome, "home");
                    Method createInLocalHome = Utility.getMethodInHomeClassForImpl(fo, Session.class, className,
                            Utility.PREFIX_CREATE, parameters, true);
                    checkMethodInInterface(problem, createInLocalHome, "local home");
                    return true;
                }
            }
        }
        return false;
    }

    private boolean checkMethod(List refactorings, Problem problem[], Method method, String resName, String param) {
        if (method != null) {
            ChangeParametersRefactoring refactoring = new ChangeParametersRefactoring(method);
            problem[0] = Utility.addProblemsToEnd(problem[0], refactoring.preCheck());
            refactorings.add(refactoring);
            if (resName != null) {
                String msg;
                if (param != null) {
                    msg = NbBundle.getMessage(EjbJarChangeParamRefactoring.class, resName, param);
                } else {
                    msg = NbBundle.getMessage(EjbJarChangeParamRefactoring.class, resName);
                }
                problem[0] = Utility.addProblemsToEnd(problem[0], new Problem(false, msg));
            }
            return true;
        } else {
            return false;
        }
    }


    private boolean checkMethodInImpl(Problem[] problem, Method method) {
        return checkMethod(implRefactors, problem, method, null, null);
    }

    private void checkMethodInImpl(Problem[] problem, Method postCreateInImpl, String resName) {
        checkMethod(implRefactors, problem, postCreateInImpl, resName, null);
    }

    private boolean checkMethodInInterface(Problem[] problem, Method method) {
        return checkMethod(ifaceRefactors, problem, method, null, null);
    }

    private boolean checkMethodInInterface(Problem[] problem, Method method, String interfaceName) {
        return checkMethod(ifaceRefactors, problem, method, "TXT_EjbJarIfaceMethodWarningChangeParam", interfaceName);
    }

    public Problem fastCheckParameters(RefObject refObject) {
//        if ((javaClass != null) && !(javaClass instanceof UnresolvedClass)) {
//            err.log("resolved class: " + javaClass);
//            
//       }
       return null;
    }

    public Problem checkParameters(RefObject refObject) {
        return fastCheckParameters(refObject);
    }

    public Problem prepare(AbstractRefactoring refactoring, RefObject refObject, ParameterInfo[] paramInfo,
            int modifier, RefactoringElementsBag refactoringElements) {
        RefactoringSession session = refactoringElements.getSession();
        Problem problem = null;
        if (refObject instanceof Method) {
            Method method = (Method) refObject;
            if (isChanged(method, paramInfo)) {
                processFinderMethods(refactoring, method, paramInfo, refactoringElements);
            }
            for (int i = 0; i < ifaceRefactors.size(); i++) {
                problem = prepareRefactoring((ChangeParametersRefactoring) ifaceRefactors.get(i), modifier, paramInfo,
                        session, problem);
            }
            for (int i = 0; i < implRefactors.size(); i++) {
                problem = prepareRefactoring((ChangeParametersRefactoring) implRefactors.get(i), modifier, paramInfo,
                        session, problem);
            }
        }
        return problem;
    }

    private void processFinderMethods(AbstractRefactoring refactoring, Method method, ParameterInfo[] paramInfo,
            RefactoringElementsBag refactoringElements) {
        if (isCmpImplFinder || isCmpHomeFinder || isCmpLocalHomeFinder) {
            String methodName = method.getName();
            if (isCmpImplFinder) {
                methodName = methodName.substring(9, 10).toLowerCase() + methodName.substring(10);
            }
            err.log("method name: " + methodName);

            FileObject fo = JavaModel.getFileObject(method.getResource());
            if (fo == null) {
                return;
            }
            Project prj = FileOwnerQuery.getOwner(fo);
            if (prj == null) {
                return;
            }
            org.netbeans.modules.j2ee.api.ejbjar.EjbJar emod =
                    org.netbeans.modules.j2ee.api.ejbjar.EjbJar.getEjbJar(prj.getProjectDirectory());
            ClassDefinition declaringClass = method.getDeclaringClass();
            err.log("classdefinition jc: " + declaringClass);
            if (declaringClass instanceof JavaClass) {
                Entity[] eBeans = Utility.getEntityBeans(emod);
                EjbJar ejbJarDD = Utility.getEjbJar(emod);
                if ((eBeans != null) && (ejbJarDD != null)) {
                    for (int i = 0; i < eBeans.length; i++) {
                        Entity entity = eBeans[i];
                        if (isCmpImplFinder) {
                            prepareQueries(declaringClass, entity.getEjbClass(), entity, methodName, ejbJarDD,
                                    paramInfo, emod, refactoringElements, refactoring);
                        } else {
                            if (isCmpHomeFinder) {
                                prepareQueries(declaringClass, entity.getHome(), entity, methodName, ejbJarDD,
                                        paramInfo, emod, refactoringElements, refactoring);
                            }
                            if (isCmpLocalHomeFinder) {
                                prepareQueries(declaringClass, entity.getLocalHome(), entity, methodName, ejbJarDD,
                                        paramInfo, emod, refactoringElements, refactoring);
                            }
                        }
                    }
                }
            }
        }
    }

    private Problem prepareRefactoring(ChangeParametersRefactoring chr, int modifier,
            ParameterInfo[] paramInfo, RefactoringSession session, Problem problem) {
        chr.setModifiers(modifier);
        chr.setParameterInfo(paramInfo);
        Problem p = chr.prepare(session);
        problem = Utility.addProblemsToEnd(problem, p);
        return problem;
    }

    private void prepareQueries(ClassDefinition jc, String ejbClass, Entity entity, String methodName, EjbJar ejbJarDD,
            ParameterInfo[] paramInfo, org.netbeans.modules.j2ee.api.ejbjar.EjbJar emod,
            RefactoringElementsBag refactoringElements, AbstractRefactoring refactoring) {
        if (jc.getName().equals(ejbClass)) {
            Query[] queries = entity.getQuery();
            for (int q=0; q<queries.length; q++) {
                QueryMethod qm = queries[q].getQueryMethod();
                err.log("query method: " + qm);
                if (methodName.equals(qm.getMethodName())) {
                    RefactoringElementImplementation elem =
                            new EjbJarFinderMethodChangeParamsRefactoringElement(ejbJarDD, methodName, paramInfo, entity, emod.getDeploymentDescriptor());
                    refactoringElements.add(refactoring, elem);
                }
            }
        }
    }

    public final class EjbJarFinderMethodChangeParamsRefactoringElement extends AbstractChangeParamsRefactoringElement
            implements ExternalChange {
        protected EjbJar ejbJarDD;
        private Entity entity;
        private String[] oldParams = new String[]{};

        /**
         * Creates a new instance of EjbJarFinderMethodChangeParamsRefactoringElement
         */
        public EjbJarFinderMethodChangeParamsRefactoringElement(EjbJar ejbJarDD, String methodName,
                ParameterInfo[] paramInfo, Entity entity, FileObject parentFile) {
            this.ejbJarDD = ejbJarDD;
            this.entity = entity;
            this.methodName = methodName;
            this.paramInfo = paramInfo;
            this.parentFile = parentFile;
        }

        /**
         * Returns text describing the refactoring formatted for display (using HTML tags).
         *
         * @return Formatted text.
         */
        public String getDisplayText() {
            Object[] args = new Object []{parentFile.getNameExt(), methodName};
            return MessageFormat.format(NbBundle.getMessage(EjbJarChangeParamRefactoring.class,
                    "TXT_EjbJarFinderMethodChangeParam"), args);
        }

        /**
         * Performs the change represented by this refactoring element.
         */
        public void performChange() {
            JavaMetamodel.getManager().registerExtChange(this);
        }

        public void performExternalChange() {
            Query[] queries = entity.getQuery();
            for (int q = 0; q < queries.length; q++) {
                QueryMethod qm = queries[q].getQueryMethod();
                if (methodName.equals(qm.getMethodName())) {
                    MethodParams mp = qm.getMethodParams();
                    if (mp != null) {
                        oldParams = mp.getMethodParam();
                        String[] newParams = getNewParameters(paramInfo, oldParams);
                        if (newParams != null) {
                            mp.setMethodParam(newParams);
                        }
                    }
                    try {
                        ejbJarDD.write(parentFile);
                    } catch (IOException ioe) {
                        //TODO
                    }
                }
            }
        }

        public void undoExternalChange() {
            Query[] queries = entity.getQuery();
            for (int q = 0; q < queries.length; q++) {
                QueryMethod qm = queries[q].getQueryMethod();
                if (methodName.equals(qm.getMethodName())) {
                    MethodParams mp = qm.getMethodParams();
                    if (mp != null) {
                        if (oldParams != null) {
                            mp.setMethodParam(oldParams);
                        }
                    }
                    try {
                        ejbJarDD.write(parentFile);
                    } catch (IOException ioe) {
                        //TODO
                    }
                }
            }
        }
    }

    /** Checks if the parameters changed (name of parameter is not checked, as it's not reflected */
    private static boolean isChanged(Method method, ParameterInfo[] paramInfo) {
        if ((paramInfo != null)) {
            if (paramInfo.length < method.getParameters().size()) { // parameter was removed
                return true;
            }
            if ((paramInfo.length > 0)) {
                for (int i=0; i<paramInfo.length; i++) {
                    ParameterInfo pi = paramInfo[i];
                    if ((pi.getDefaultValue() != null) ||
                        (pi.getOriginalIndex() != i) ||
                        (pi.getType() != null)
                       ) {
                        return true;
                    }
                }
            }
        }
        return false;
    }
    
    private static String[] getNewParameters(ParameterInfo[] paramInfo, String[] methodParams) {
        String[] newParams = new String[paramInfo.length];
        if ((paramInfo != null) && (paramInfo.length > 0)) {
            for (int i=0; i<paramInfo.length; i++) {
                ParameterInfo pi = paramInfo[i];
                Type newParamType = pi.getType();
                if (newParamType == null) {
                    newParams[i] = methodParams[pi.getOriginalIndex()];
                } else {
                    newParams[i] = newParamType.getName();
                }
            }
        }
        return newParams;
    }
}