/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */


package org.netbeans.modules.iep.editor.model;

import java.awt.Point;

import org.openide.loaders.DataObject;

import org.netbeans.modules.iep.editor.tcg.exception.ParseXmlException;
import org.netbeans.modules.iep.editor.tcg.exception.ModelSaveException;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponent;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponentType;
import org.netbeans.modules.iep.editor.tcg.model.TcgProperty;
import org.netbeans.modules.iep.editor.tcg.model.TcgModelManager;

import org.netbeans.modules.iep.editor.designer.EntityNode;
import org.netbeans.modules.iep.editor.designer.Link;
import org.netbeans.modules.iep.editor.designer.PdModel;
import org.netbeans.modules.iep.editor.share.SharedConstants;
import org.netbeans.modules.iep.editor.tcg.util.GenUtil;
import org.netbeans.modules.iep.editor.tcg.util.IOUtil;
import org.netbeans.modules.iep.editor.tcg.util.NameUtil;
import java.io.File;
import java.io.FileOutputStream;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.logging.Level;
import org.openide.util.NbBundle;

/**
 * Interface that extends Component
 *
 * @author Bing Lu
 *
 * @since July 8, 2004
 */
class PlanImpl extends ModelImpl implements Plan {
    /**
     *  Logger.
     */
    private static final java.util.logging.Logger mLog = java.util.logging.Logger.getLogger(PlanImpl.class.getName());
    
    private static final long serialVersionUID = -6393736384546452393L;
    
    private TcgComponent mMetadata;
    private TcgComponent mView;
    private TcgProperty  mOrthoFlow;
    private TcgComponent mOperators;
    private TcgComponent mLinks;
    private TcgComponent mSchemas;
    
    private static void cleanupDanglingReferences(TcgComponent component, Set set) {
        List list = component.getComponentList();
        // Because of the removal during the traversal, we have to work backword
        for (int i = list.size()-1; i >= 0; i--) {
            TcgComponent c = (TcgComponent)list.get(i);
            if (!set.contains(c)) {
                component.removeComponent(c.getName());
            }
        }
    }
    
    private static final int MAX_COUNT = 1000000000;
    private static String getNamePropValueForNewComponent(TcgComponent parent, TcgComponentType ct) {
        String tn = ct.getName();
        List list = parent.getComponentList();
        try {
            for (int i = 0; i < MAX_COUNT; i++) {
                String name = tn + i;
                boolean exist = false;
                for (int j = 0, J = list.size(); j < J; j++) {
                    TcgComponent c = (TcgComponent)list.get(j);
                    if (c.getProperty(NAME_KEY).getStringValue().equalsIgnoreCase(name)) {
                        exist = true;
                        break;
                    }
                }
                if (!exist) {
                    return name;
                }
            }
        } catch (Exception e) {
            mLog.log(Level.SEVERE, "PlanImpl.getNameForNewComponent failed: " + ct.getName(), e);
        }
        return null;
    }
    
    private static String getNameForNewComponent(TcgComponent parent, String prefix) {
        try {
            for (int i = 0; i < MAX_COUNT; i++) {
                String name = prefix + i;
                if (!parent.hasComponent(name)) {
                    return name;
                }
            }
        } catch (Exception e) {
            mLog.log(Level.SEVERE, "PlanImpl.getNameForNewComponent failed for prefix: "  + prefix, e);
        }
        return null;
    }
    
    private String getIdForNewOperator() {
        return getNameForNewComponent(mOperators, "o");
    }
    
    private String getNameForNewOperator(TcgComponentType ct) {
        return getNamePropValueForNewComponent(mOperators, ct);
    }
    
    private String getNameForNewLink() {
        return getNameForNewComponent(mLinks, "link");
    }
    
    protected void initialize(TcgComponent root) {
        super.initialize(root);
        try {
            mMetadata = root.getComponent(METADATA_KEY);
            mView = mMetadata.getComponent(VIEW_KEY);
            mOrthoFlow = mView.getProperty(ORTHO_FLOW_KEY);
            mOperators = root.getComponent(OPERATORS_KEY);
            mLinks = root.getComponent(LINKS_KEY);
            mSchemas = root.getComponent(SCHEMAS_KEY);
        } catch (Exception e) {
            e.printStackTrace();
            mLog.warning(e.getMessage());
        }
    }
    
    protected PlanImpl() {
        TcgComponentType planType =
                TcgModelManager.getTcgComponentType("/IEP/Model/Plan");
        
        initialize(planType.newTcgComponent("Plan", planType.getTitle()));
    }
    
    protected PlanImpl(TcgComponent root) {
        initialize(root);
    }
    
    protected PlanImpl(DataObject doc) throws ParseXmlException {
        super(doc);
    }
    
    protected PlanImpl(File doc) throws ParseXmlException {
        super(doc);
    }
    
    public File getWsdlFile() {
        String wsdlPath = getFullPath();
        wsdlPath = wsdlPath.substring(0, wsdlPath.length()-4) + ".wsdl"; //4: for '.iep'
        return new File(wsdlPath);
    }
    
    public void saveWsdl() throws ModelSaveException {
        // auto generate .wsdl
        // see org.netbeans.modules.iep.editor.jbiadapter.IEPSEDeployer
        String tns = NameUtil.makeJavaId(getFullName());
        FileOutputStream fos = null;
        try {
            // Generate .wsdl in the same dir as its .iep is
            String wsdl = getWSDL(tns);
            String wsdlPath = getFullPath();
            wsdlPath = wsdlPath.substring(0, wsdlPath.length()-4) + ".wsdl"; //4: for '.iep'
            GenUtil.createFile(new File(wsdlPath), false);
            fos = new FileOutputStream(wsdlPath);
            IOUtil.copy(wsdl.getBytes("UTF-8"), fos);
        } catch (Exception e) {
            e.printStackTrace();
            throw new ModelSaveException("ModelImpl.FAIL_SAVE_MODEL",
                    "org.netbeans.modules.iep.editor.model.Bundle",
                    new Object[]{getFullName()}, e);
        } finally {
            if (fos != null) {
                try {
                    fos.close();
                } catch (Exception e1) {
                    e1.printStackTrace();
                }
            }
        }
    }
    
    /**
     * This method is called by PdModel's changeValue(), which is in turn called by
     * the JGoDocumentChangedEdit.undo and redo methods.
     */
    public void addOperator(TcgComponent c) {
        if (mOperators.hasComponent(c.getName())) {
            mLog.log(Level.SEVERE, "PlanImpl.addOperator_failed_duplicate_component" + c.getName());
            return;
        }
        try {
            mOperators.addComponent(c);
            boolean isSchemaOwner = c.getProperty(IS_SCHEMA_OWNER_KEY).getBoolValue();
            if (isSchemaOwner) {
                String outputSchemaId = c.getProperty(OUTPUT_SCHEMA_ID_KEY).getStringValue();
                if (outputSchemaId != null && !outputSchemaId.trim().equals("")) {
                    addSchema((Schema)c.getMarker(SCHEMA_MARKER));
                }
            }
        } catch (Exception e) {
            mLog.log(Level.SEVERE, "PlanImpl.addOperator_failed " + c.getName(), e);
        }
        
    }
    
    public TcgComponent addNewOperator(TcgComponentType ct) {
        try {
            String ctPath = ct.getPath();
            String newName;
            TcgComponent newComp = null;
            if (ctPath.startsWith(OPERATOR_PATH) ||
                    ctPath.startsWith(INPUT_PATH) ||
                    ctPath.startsWith(OUTPUT_PATH)) {
                newName = getNameForNewOperator(ct);
                String newId = getIdForNewOperator();
                newComp = ct.newTcgComponent(newId, newId);
                newComp.getProperty(ID_KEY).setValue(newId);
                newComp.getProperty(NAME_KEY).setValue(newName);
                mOperators.addComponent(newComp);
            } else {
                mLog.warning("PlanImpl.PlanImpl_cannot_handle_TcgComponentType " + ctPath);
            }
            return newComp;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
    
    public TcgComponent copyAndAddOperator(TcgComponent c) {
        try {
            TcgComponentType ct = c.getType();
            String ctPath = c.getType().getPath();
            String newName;
            TcgComponent newComp = null;
            if (ctPath.startsWith(OPERATOR_PATH)
            || ctPath.startsWith(INPUT_PATH)
            || ctPath.startsWith(OUTPUT_PATH)) {
                newName = getNameForNewOperator(ct);
                String newId = getIdForNewOperator();
                newComp = ct.duplicate(newId, c);
                newComp.getProperty(ID_KEY).setValue(newId);
                newComp.getProperty(NAME_KEY).setValue(newName);
                mOperators.addComponent(newComp);
            }
            return newComp;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
    
    /**
     * This method is called by PdCanvas's deleteSelection when an EntityNode is
     * deleted. It is called by PdModel's changeValue, which is in turn called by
     * the JGoDocumentChangedEdit.undo and redo methods.
     */
    public void removeOperator(TcgComponent c) {
        try {
            if (c != null) {
                mOperators.removeComponent(c.getName());
                boolean isSchemaOwner = c.getProperty(IS_SCHEMA_OWNER_KEY).getBoolValue();
                if (isSchemaOwner) {
                    String outputSchemaId = c.getProperty(OUTPUT_SCHEMA_ID_KEY).getStringValue();
                    if (outputSchemaId != null && !outputSchemaId.trim().equals("")) {
                        c.setMarker(SCHEMA_MARKER, getSchema(outputSchemaId));
                        removeSchema(outputSchemaId);
                    }
                }
            }
        } catch (Exception e) {
            mLog.log(Level.SEVERE, "PlanImpl.removeOperator_failed: " + c.getName(), e);
        }
    }
    
    public TcgComponent getOperatorById(String id) {
        try {
            return mOperators.getComponent(id);
        } catch (Exception e) {
        }
        return null;
    }
    
    public boolean hasOperator(String name) {
        try {
            List list = mOperators.getComponentList();
            for (int i = 0, I = list.size(); i < I; i++) {
                TcgComponent c = (TcgComponent)list.get(i);
                if (name.equals(c.getProperty(NAME_KEY).getStringValue())) {
                    return true;
                }
            }
        } catch (Exception e) {
        }
        return false;
    }
    
    public void addLink(TcgComponent c) {
        if (mLinks.hasComponent(c.getName())) {
            mLog.log(Level.SEVERE, "PlanImpl.addLink_failed:_duplicate_component " + c.getName());
            return;
        }
        mLinks.addComponent(c);
    }
    
    public TcgComponent addNewLink() {
        try {
            String newName = getNameForNewLink();
            TcgComponent newComp = LINK_TYPE.newTcgComponent(newName, newName);
            newComp.getProperty(NAME_KEY).setValue(newName);
            mLinks.addComponent(newComp);
            return newComp;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
    
    public TcgComponent copyAndAddLink(TcgComponent c) {
        try {
            String newName = getNameForNewLink();
            TcgComponent newComp = LINK_TYPE.duplicate(newName, c);
            newComp.getProperty(NAME_KEY).setValue(newName);
            mLinks.addComponent(newComp);
            return newComp;
        } catch (Exception e) {
            e.printStackTrace();
        }
        return null;
    }
    
    public void removeLink(TcgComponent c) {
        if (c != null) {
            mLinks.removeComponent(c.getName());
        }
    }
    
    public void initializeModel(PdModel model) {
        try {
            model.startTransaction();
            // Add all relations, streams, operators by z order
            ArrayList list = new ArrayList();
            list.addAll(mOperators.getComponentList());
            Collections.sort(list, zOrderComparator);
            for (int i=0, I=list.size(); i < I; i++) {
                TcgComponent c = (TcgComponent)list.get(i);
                int x = ((Integer)c.getProperty(X_KEY).getValue()).intValue();
                int y = ((Integer)c.getProperty(Y_KEY).getValue()).intValue();
                EntityNode node = new EntityNode(this, c, new Point(x, y));
                model.addObjectAtTail(node);
            }
            list.clear();
            
            //Add all links
            list.addAll(mLinks.getComponentList());
            for (int i=0, I=list.size(); i < I; i++) {
                TcgComponent c = (TcgComponent)list.get(i);
                String from = c.getProperty(FROM_KEY).getStringValue();
                EntityNode fromNode = model.findNodeById(from);
                if (fromNode == null) {
                    continue;
                }
                String to = c.getProperty(TO_KEY).getStringValue();
                EntityNode toNode = model.findNodeById(to);
                if (toNode == null) {
                    continue;
                }
                TcgComponent comp = addNewLink();
                Link link = new Link(comp, fromNode.getOutputPort(), toNode.getInputPort());
                model.addObjectAtTail(link);
            }
            
            // Initialize properties after nodes and links are added to the model
            boolean orthoFlow = ((Boolean)mOrthoFlow.getValue()).booleanValue();
            model.setOrthogonalFlows(orthoFlow);
        } catch (Exception e) {
            e.printStackTrace();
            mLog.warning(e.getMessage());
        } finally {
            // Note that calling endTransaction(false) does not actually undo
            // any or all of the changes that have happened since the last call
            // to startTransaction.  Instead, only the record of those changes will
            // be thrown away. see JGoUndoManager endTransaction()
            //
            // This is the desired result.  Because the PdModel should be in no-change-yet state
            // right after it is initialized.
            model.endTransaction(false);
        }
    }
    
    public void setProperties(Map properties) {
        Boolean orthoFlow = (Boolean)properties.get(ORTHO_FLOW_KEY);
        if (orthoFlow != null) {
            mOrthoFlow.setValue(orthoFlow);
        }
    }
    
    public void cleanupDanglingReferences(Set set) {
        cleanupDanglingReferences(mOperators, set);
        cleanupDanglingReferences(mLinks, set);
    }
    
    public String getNameForNewSchema() {
        return getNameForNewComponent(mSchemas, "schema");
    }
    
    public void addSchema(Schema schema) {
        mSchemas.addComponent(schema.getComponent());
    }
    
    public Schema copyAndAddSchema(Schema schema) {
        String name = schema.getName();
        String newName = name;
        for (int i = 1; true; i++) {
            if (!hasSchema(newName)) {
                break;
            }
            newName = name + "_" + i;
        }
        Schema newSchema = schema.duplicateSchema(newName);
        addSchema(newSchema);
        return newSchema;
    }
    
    public boolean hasSchema(String name) {
        return mSchemas.hasComponent(name);
    }
    
    public Schema getSchema(String name) {
        TcgComponent c = mSchemas.getComponent(name);
        if (c == null) {
            return null;
        }
        return new SchemaImpl(c);
    }
    
    public void removeSchema(String name) {
        mSchemas.removeComponent(name);
    }
    
    private static final Comparator zOrderComparator = new ZOrderComparator();
    
    //==========================================================================
    private static Map SQL_2_XSD_MAP = new HashMap();
    static {
        SQL_2_XSD_MAP.put("BIT", "xsd:boolean");
        SQL_2_XSD_MAP.put("TINYINT", "xsd:byte");
        SQL_2_XSD_MAP.put("SMALLINT", "xsd:short");
        SQL_2_XSD_MAP.put("INTEGER", "xsd:int");
        SQL_2_XSD_MAP.put("BIGINT", "xsd:long");
        SQL_2_XSD_MAP.put("REAL", "xsd:float");
        SQL_2_XSD_MAP.put("FLOAT", "xsd:double");
        SQL_2_XSD_MAP.put("DOUBLE", "xsd:double");
        SQL_2_XSD_MAP.put("DECIMAL", "xsd:decimal");
        SQL_2_XSD_MAP.put("NUMERIC", "xsd:decimal");
        SQL_2_XSD_MAP.put("CHAR", "xsd:string");
        SQL_2_XSD_MAP.put("VARCHAR", "xsd:string");
        SQL_2_XSD_MAP.put("LONGVARCHAR", "xsd:string");
        SQL_2_XSD_MAP.put("DATE", "xsd:date");
        SQL_2_XSD_MAP.put("TIME", "xsd:time");
        SQL_2_XSD_MAP.put("TIMESTAMP", "xsd:dateTime");
        SQL_2_XSD_MAP.put("BINARY", "xsd:base64Binary");
        SQL_2_XSD_MAP.put("VARBINARY", "xsd:base64Binary");
        SQL_2_XSD_MAP.put("LONGVARBINARY", "xsd:base64Binary");
        SQL_2_XSD_MAP.put("BLOB", "xsd:base64Binary");
        SQL_2_XSD_MAP.put("CLOB", "xsd:string");
//        "ARRAY",
//        "REF",
//        "STRUCT",
    };
    
    private static final String INPUT_PORT_TYPE = "InputPt";
    private static final String INPUT_PARTNER_LINK_TYPE = "InputPlt";
    private static final String INPUT_ROLE_NAME = "InputRn";
    private static final String INPUT_PARTNER_LINK = "InputPl";
    
    private static String getOutputPortType(String opJName) {
        return "OutputPt_" + opJName;
    }
    private static String getOutputPartnerLinkType(String opJName) {
        return "OutputPlt_" + opJName;
    }
    private static String getOutputRoleName(String opJName) {
        return "OutputRn_" + opJName;
    }
    private static String getOutputPartnerLink(String opJName) {
        return "OutputPl_" + opJName;
    }
    
    public String getWSDL(String tns) throws Exception {
        List inList = getInputList();
        List outList = getOutputList();
        
        StringBuffer sb = new StringBuffer();
        sb.append("<definitions targetNamespace=\"" + tns + "\"\n");
        sb.append("             xmlns:tns=\"" + tns + "\"\n");
        sb.append("             xmlns:typens=\"" + tns + "\"\n");
        sb.append("             xmlns:defns=\"" + tns + "\"\n");
        sb.append("             xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\"\n");
        sb.append("             xmlns:soap=\"http://schemas.xmlsoap.org/wsdl/soap/\"\n");
        sb.append("             xmlns:file=\"http://schemas.sun.com/jbi/wsdl-extensions/file/\"\n");
        sb.append("             xmlns=\"http://schemas.xmlsoap.org/wsdl/\">\n");
        // types
        sb.append("<types>\n");
        sb.append("    <xsd:schema targetNamespace=\"" + tns + "\" xmlns:xsd=\"http://www.w3.org/2001/XMLSchema\">\n");
        // input types
        for (int i = 0, I = inList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)inList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            String schemaId = (String)op.getProperty(OUTPUT_SCHEMA_ID_KEY).getValue();
            if (schemaId == null) {
                throw new Exception(NbBundle.getMessage(PlanImpl.class,"PlanImpl.Input") + " " 
                        + name 
                        + " " + NbBundle.getMessage(PlanImpl.class,"PlanImpl_schema_is_not_defined"));
            }
            Schema schema = getSchema(schemaId);
            // input_MsgObj
            sb.append("        <xsd:element name=\"" + name + "_MsgObj\">\n");
            sb.append("            <xsd:complexType>\n");
            sb.append("                <xsd:sequence>\n");
            if (schema != null) {
                for (int j = 0, J = schema.getAttributeCount(); j < J; j++) {
                    AttributeMetadata cm = schema.getAttributeMetadata(j);
                    String cName = cm.getAttributeName();
                    String cType = cm.getAttributeType();
                    String xsdType = (String)SQL_2_XSD_MAP.get(cType);
                    sb.append("                    <xsd:element name=\"" + cName + "\" type=\"" + xsdType + "\"/>\n");
                }
            }
            sb.append("                </xsd:sequence>\n");
            sb.append("            </xsd:complexType>\n");
            sb.append("        </xsd:element>\n");
            
            // inputBatch_MsgObj
            sb.append("        <xsd:element name=\"" + name + "Batch_MsgObj\">\n");
            sb.append("            <xsd:complexType>\n");
            sb.append("                <xsd:sequence>\n");
            sb.append("                    <xsd:element name=\"" + name + "_MsgObj\" minOccurs=\"0\" maxOccurs=\"unbounded\">\n");
            sb.append("                        <xsd:complexType>\n");
            sb.append("                            <xsd:sequence>\n");
            if (schema != null) {
                for (int j = 0, J = schema.getAttributeCount(); j < J; j++) {
                    AttributeMetadata cm = schema.getAttributeMetadata(j);
                    String cName = cm.getAttributeName();
                    String cType = cm.getAttributeType();
                    String xsdType = (String)SQL_2_XSD_MAP.get(cType);
                    sb.append("                                <xsd:element name=\"" + cName + "\" type=\"" + xsdType + "\"/>\n");
                }
            }
            sb.append("                            </xsd:sequence>\n");
            sb.append("                        </xsd:complexType>\n");
            sb.append("                    </xsd:element>\n");
            sb.append("                </xsd:sequence>\n");
            sb.append("            </xsd:complexType>\n");
            sb.append("        </xsd:element>\n");
        }
        // output_MsgObj
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            boolean isRelation = op.getProperty(INPUT_TYPE_KEY).getStringValue().equals(IO_TYPE_RELATION);
            String schemaId = (String)op.getProperty(OUTPUT_SCHEMA_ID_KEY).getValue();
            if (schemaId == null) {
                throw new Exception(NbBundle.getMessage(PlanImpl.class,"PlanImpl.Output")  
                        + " " + name + " " +  NbBundle.getMessage(PlanImpl.class,"PlanImpl_schema_is_not_defined"));
            }
            Schema schema = getSchema(schemaId);
            boolean includeTimestamp = op.getProperty(INCLUDE_TIMESTAMP_KEY).getBoolValue();
            boolean batchMode = op.getProperty(BATCH_MODE_KEY).getBoolValue();
            String indent = batchMode? "            " : "";
            if (batchMode) {
                sb.append("        <xsd:element name=\"" + name + "Batch_MsgObj\">\n");
                sb.append("            <xsd:complexType>\n");
                sb.append("                <xsd:sequence>\n");
                sb.append("                    <xsd:element name=\"" + name + "_MsgObj\" minOccurs=\"0\" maxOccurs=\"unbounded\">\n");
            } else {
                sb.append("        <xsd:element name=\"" + name + "_MsgObj\">\n");
            }
            sb.append(indent + "            <xsd:complexType>\n");
            sb.append(indent + "                <xsd:sequence>\n");
            if (schema != null) {
                for (int j = 0, J = schema.getAttributeCount(); j < J; j++) {
                    AttributeMetadata cm = schema.getAttributeMetadata(j);
                    String cName = cm.getAttributeName();
                    String cType = cm.getAttributeType();
                    String xsdType = (String)SQL_2_XSD_MAP.get(cType);
                    sb.append(indent + "                   <xsd:element name=\"" + cName + "\" type=\"" + xsdType + "\"/>\n");
                }
            }
            if (isRelation) {
                sb.append(indent + "                   <xsd:element name=\"SeqId\" type=\"xsd:string\"/>\n");
            }
            if (includeTimestamp) {
                sb.append(indent + "                   <xsd:element name=\"Timestamp\" type=\"xsd:dateTime\"/>\n");
            }
            if (isRelation) {
                sb.append(indent + "                   <xsd:element name=\"Tag\" type=\"xsd:string\"/>\n");
            }
            sb.append(indent + "                </xsd:sequence>\n");
            sb.append(indent + "            </xsd:complexType>\n");
            sb.append(indent + "        </xsd:element>\n");
            if (batchMode) {
                sb.append("                </xsd:sequence>\n");
                sb.append("            </xsd:complexType>\n");
                sb.append("        </xsd:element>\n");
            }
        }
        
        sb.append("    </xsd:schema>\n");
        sb.append("</types>\n");
        sb.append("\n");
        // message
        for (int i = 0, I = inList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)inList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("<message name=\"" + name + "_Msg\">\n");
            sb.append("    <part name=\"input\" element=\"typens:" + name + "_MsgObj\"/>\n");
            sb.append("</message>\n");
            
            sb.append("<message name=\"" + name + "Batch_Msg\">\n");
            sb.append("    <part name=\"input\" element=\"typens:" + name + "Batch_MsgObj\"/>\n");
            sb.append("</message>\n");
        }
        
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            boolean batchMode = op.getProperty(BATCH_MODE_KEY).getBoolValue();
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            if (batchMode) {
                sb.append("<message name=\"" + name + "Batch_Msg\">\n");
                sb.append("    <part name=\"output\" element=\"typens:" + name + "Batch_MsgObj\"/>\n");
                sb.append("</message>\n");
            } else {
                sb.append("<message name=\"" + name + "_Msg\">\n");
                sb.append("    <part name=\"output\" element=\"typens:" + name + "_MsgObj\"/>\n");
                sb.append("</message>\n");
            }
        }
        
        sb.append("\n");
        //portType
        sb.append("<portType name=\"" + INPUT_PORT_TYPE + "\">\n");
        for (int i = 0, I = inList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)inList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("    <operation name=\"" + name + "\">\n");
            sb.append("        <input message=\"tns:" + name + "_Msg\"/>\n");
            sb.append("    </operation>\n");
            sb.append("    <operation name=\"" + name + "Batch\">\n");
            sb.append("        <input message=\"tns:" + name + "Batch_Msg\"/>\n");
            sb.append("    </operation>\n");
        }
        sb.append("</portType>\n");
        
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            boolean batchMode = op.getProperty(BATCH_MODE_KEY).getBoolValue();
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("<portType name=\"" + getOutputPortType(name) + "\">\n");
            sb.append("    <operation name=\"" + name + "\">\n");
            if (batchMode) {
                sb.append("        <input message=\"tns:" + name + "Batch_Msg\"/>\n");
            } else {
                sb.append("        <input message=\"tns:" + name + "_Msg\"/>\n");
            }
            sb.append("    </operation>\n");
            sb.append("</portType>\n");
        }
        
        sb.append("\n");
        
        // partnerLinkType
        sb.append("<plnk:partnerLinkType name=\"" + INPUT_PARTNER_LINK_TYPE + "\" xmlns:plnk=\"http://docs.oasis-open.org/wsbpel/2.0/plnktype\">\n");
        sb.append("    <plnk:role name = \"" + INPUT_ROLE_NAME + "\" portType=\"tns:" + INPUT_PORT_TYPE + "\"/>\n");
        sb.append("</plnk:partnerLinkType>\n");
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("<plnk:partnerLinkType name=\"" + getOutputPartnerLinkType(name) + "\" xmlns:plnk=\"http://docs.oasis-open.org/wsbpel/2.0/plnktype\">\n");
            sb.append("    <plnk:role name = \"" + getOutputRoleName(name) + "\" portType=\"tns:" + getOutputPortType(name) + "\"/>\n");
            sb.append("</plnk:partnerLinkType>\n");
        }
        // input binding
        sb.append("<!-- input binding -->\n");
        sb.append("<binding name=\"InputBinding\" type=\"defns:InputPt\">\n");
        sb.append("    <soap:binding style=\"document\" transport=\"http://schemas.xmlsoap.org/soap/http\"/>\n");
        for (int i = 0, I = inList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)inList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("    <operation name=\"" + name + "\">\n");
            sb.append("        <soap:operation soapAction=\"" + name + "\"/>\n");
            sb.append("        <input>\n");
            sb.append("            <soap:body use=\"literal\"/>\n");
            sb.append("        </input>\n");
            sb.append("    </operation>\n");
            
            sb.append("    <operation name=\"" + name + "Batch\">\n");
            sb.append("        <soap:operation soapAction=\"" + name + "Batch\"/>\n");
            sb.append("        <input>\n");
            sb.append("            <soap:body use=\"literal\"/>\n");
            sb.append("        </input>\n");
            sb.append("    </operation>\n");
        }
        sb.append("</binding>\n");
        
        // input service
        sb.append("<!-- input service -->\n");
        sb.append("<service name=\"InputService\">\n");
        sb.append("    <port name=\"InputPort\" binding=\"tns:InputBinding\">\n");
        sb.append("        <soap:address location=\"http://localhost:12100/service/" + tns + "\"/>\n");
        sb.append("    </port>\n");
        sb.append("</service>\n");

        // http output binding and service
        if (outList.size() > 0) {
            sb.append("\n<!-- http output binding and service\n");
        }
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("<binding name=\"OutputBinding_" + name + "\" type=\"defns:OutputPt_" + name + "\">\n");
            sb.append("    <operation name=\"" + name + "\">\n");
            sb.append("        <soap:operation soapAction=\"" + name + "\"/>\n");
            sb.append("        <input>\n");
            sb.append("            <soap:body use=\"literal\"/>\n");
            sb.append("        </input>\n");
            sb.append("    </operation>\n");
            sb.append("</binding>\n");
        }
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            String servceName = tns + "_" + name + "_callee";
            sb.append("<service name=\"OutputService_" + name + "\">\n");
            sb.append("    <port name=\"OutputPort_" + name + "\" binding=\"tns:OutputBinding_" + name + "\">\n");
            sb.append("        <soap:address location=\"http://localhost:12100/service/" + servceName + "\"/>\n");
            sb.append("    </port>\n");
            sb.append("</service>\n");
        }
        if (outList.size() > 0) {
            sb.append(" end of http output binding and service -->\n\n");
        }

        // file output binding and service
        if (outList.size() > 0) {
            sb.append("<!-- file output binding and service -->\n");
        }
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            sb.append("<binding name=\"OutputBinding_" + name + "\" type=\"defns:OutputPt_" + name + "\">\n");
            sb.append("    <file:binding/>\n");
            sb.append("    <operation name=\"" + name + "\">\n");
            sb.append("        <file:operation/>\n");
            sb.append("        <input>\n");
            sb.append("            <file:message fileName=\"" + name + ".txt\"\n");
            sb.append("                    fileNameIsPattern=\"false\"\n");
            sb.append("                    addEOL=\"false\"\n");
            sb.append("                    multipleRecordsPerFile=\"true\"\n");
            sb.append("                    use=\"literal\"/>\n");
            sb.append("        </input>\n");
            sb.append("    </operation>\n");
            sb.append("</binding>\n");
        }
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            String servceName = tns + "_" + name + "_callee";
            sb.append("<service name=\"OutputService_" + name + "\">\n");
            sb.append("    <port name=\"OutputPort_" + name + "\" binding=\"tns:OutputBinding_" + name + "\">\n");
            sb.append("        <file:address fileDirectory=\"C:/temp/" + tns + "\"/>\n");
            sb.append("    </port>\n");
            sb.append("</service>\n");
        }
        if (outList.size() > 0) {
            sb.append("<!-- end of file output binding and service -->\n");
        }
        sb.append("</definitions>\n");
        return sb.toString();
    }
    
    public List getPortMapEntryList(String tns) throws Exception {
        List pmeList = new ArrayList();
        
        List inList = getInputList();
        List outList = getOutputList();
        if (inList.size() > 0) {
            String partnerLink = tns + ":" + INPUT_PARTNER_LINK;
            String portType = tns + ":" + INPUT_PORT_TYPE;
            PortMapEntry pme = new PortMapEntry(partnerLink, portType, PortMapEntry.MY_ROLE, INPUT_ROLE_NAME);
            pmeList.add(pme);
        }
        for (int i = 0, I = outList.size(); i < I; i++) {
            TcgComponent op = (TcgComponent)outList.get(i);
            String name = op.getProperty(NAME_KEY).getStringValue();
            name = NameUtil.makeJavaId(name);
            String partnerLink = tns + ":" + getOutputPartnerLink(name);
            String portType = tns + ":" + getOutputPortType(name);
            String roleName = getOutputRoleName(name);
            PortMapEntry pme = new PortMapEntry(partnerLink, portType, PortMapEntry.PARTNER_ROLE, roleName);
            pmeList.add(pme);
        }
        return pmeList;
    }
    
    private List getInputList() {
        List list = new ArrayList();
        try {
            List opList = mOperators.getComponentList();
            for (int i = 0, I = opList.size(); i < I; i++) {
                TcgComponent c = (TcgComponent)opList.get(i);
                boolean wsInput = c.getProperty(WS_INPUT_KEY).getBoolValue();
                if (wsInput) {
                    list.add(c);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }
    
    private List getOutputList() {
        List list = new ArrayList();
        try {
            List opList = mOperators.getComponentList();
            for (int i = 0, I = opList.size(); i < I; i++) {
                TcgComponent c = (TcgComponent)opList.get(i);
                boolean wsOutput = c.getProperty(WS_OUTPUT_KEY).getBoolValue();
                if (wsOutput) {
                    list.add(c);
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        return list;
    }
}

class ZOrderComparator implements Comparator, Serializable, SharedConstants {
    private static final long serialVersionUID = -6393736384445462797L;
    
    public ZOrderComparator() {
    }
    
    public int compare(Object o1, Object o2) {
        try {
            TcgComponent c1 = (TcgComponent)o1;
            TcgComponent c2 = (TcgComponent)o2;
            Integer z1 = (Integer)c1.getProperty(Z_KEY).getValue();
            Integer z2 = (Integer)c2.getProperty(Z_KEY).getValue();
            return z1.compareTo(z2);
        } catch (Exception e) {
            e.printStackTrace();
        }
        return 0;
    }
    public boolean equals(Object obj) {
        return this == obj;
    }
    public int hashCode() {
        assert false : "hashCode not designed";
        return 42; // any arbitrary constant will do
    }
}

