/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.iep.editor.designer;

import com.nwoods.jgo.JGoCopyEnvironment;
import java.awt.Point;
import java.awt.Toolkit;
import java.awt.event.KeyEvent;
import java.awt.dnd.DropTargetDragEvent;
import java.awt.dnd.DropTargetDropEvent;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.awt.datatransfer.Clipboard;
import java.util.logging.Level;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import javax.swing.JPopupMenu;

import org.openide.windows.IOProvider;
import org.openide.windows.InputOutput;
import org.openide.windows.OutputWriter;


import com.nwoods.jgo.JGoDocument;
import com.nwoods.jgo.JGoDocumentEvent;
import com.nwoods.jgo.JGoView;
import com.nwoods.jgo.JGoListPosition;
import com.nwoods.jgo.JGoObject;
import com.nwoods.jgo.JGoObjectSimpleCollection;
import com.nwoods.jgo.JGoPen;
import com.nwoods.jgo.JGoPort;
import com.nwoods.jgo.JGoSelection;
import com.nwoods.jgo.JGoViewEvent;
import com.nwoods.jgo.JGoViewListener;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponentType;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponent;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponentValidationMsg;
import org.netbeans.modules.iep.editor.tcg.model.TcgComponentValidationReport;
import org.netbeans.modules.iep.editor.model.Plan;
import org.netbeans.modules.iep.editor.tcg.palette.TcgActiveEditorDrop;
import org.netbeans.modules.iep.editor.tcg.model.TcgModelManager;
import org.netbeans.modules.iep.editor.tcg.ps.TcgComponentNode;
import org.netbeans.modules.iep.editor.tcg.ps.TcgComponentNodeProperty;
import org.netbeans.modules.iep.editor.tcg.ps.TcgComponentNodePropertyCustomizerDialogManager;
import org.netbeans.modules.iep.editor.tcg.ps.TcgComponentNodeView;
import org.netbeans.spi.palette.PaletteController;
import org.openide.text.ActiveEditorDrop;
import org.openide.util.Lookup;
import org.openide.util.NbBundle;

// Provide a view of a PdModel
// Implement various command handlers
public class PdCanvas extends JGoView implements JGoViewListener, GuiConstants, TcgComponentNodeView {
    private static final java.util.logging.Logger mLog = java.util.logging.Logger.getLogger(PdCanvas.class.getName());
    
    // there's just a single JPopupMenu, shared by all instances of this class,
    // that gets initialized in the app initialization
    private JPopupMenu mPopupMenu = new JPopupMenu();
    
    // State
    protected PlanDesigner mDesigner = null;
    
    protected Map mMsgListenerTable = new HashMap();
    
    /**
     * The sizing of the JGoText instance, if not explicitly set by calling setBoundingRect()
     * or setSize() or setHeight() or the like, is continually determined by the text string
     * and the font size and other attributes, if isAutoResize() is true. However, the text
     * string cannot be measured unless a Graphics[2D] instance is available. Since
     * Component.getGraphics() returns null if it has not yet become visible, JGoText cannot
     * be sized automatically until after the view becomes visible. Thus if you care about
     * the size of a JGoText but want it to be smart and not have to set it explicitly,
     * you should wait to add the JGoText object to the document until after calling
     * JGoView.setVisible(true). For example, it's common to initialize JGo documents
     * in the init() method of applets, rather than in the constructor.
     */
    void setDoc(PdModel doc) {
        setDocument(doc);
        doc.setDesigner(mDesigner);
        mDesigner.showPropertyPane(mDesigner.getPlan(), this);
    }
    
    // convenience method--the return value is a PdModel instead
    // of a JGoDocument
    PdModel getDoc() {
        return (PdModel)getDocument();
    }
    
    PlanDesigner getDesigner() {
        return mDesigner;
    }
    
    void help() {
        try {
            boolean valid = true;
            final StringBuffer helpMsg = new StringBuffer();
            JGoSelection sel = getSelection();
            if (sel.isEmpty()) {
                return;
            }
            
            Plan plan = getDoc().getPlan();
            InputOutput io = IOProvider.getDefault().getIO(plan.getFullName(), false);
            io.select();
            OutputWriter out = io.getOut();
            
            JGoListPosition pos = sel.getFirstObjectPos();
            while (pos != null) {
                JGoObject obj = sel.getObjectAtPos(pos);
                pos = sel.getNextObjectPos(pos);
                
                if (!obj.isTopLevel()) {
                    continue;
                }
                if (obj instanceof EntityNode) {
                    EntityNode node = (EntityNode)obj;
                    TcgComponent component = node.getComponent();
                    TcgComponentValidationReport r = component.validate();
                    if (!r.isOK()) {
                        valid = false;
                        out.println(node.getLabelString() + ": ");
                        List msgList = r.getMessageList();
                        for (int i = 0, I = msgList.size(); i < I; i++) {
                            TcgComponentValidationMsg msg = (TcgComponentValidationMsg)msgList.get(i);
                            String type = msg.getType().equals(VALIDATION_ERROR_KEY)? "error" : "warning";
                            out.println("\t" + type + ": " + msg.getText());
                        }
                    }
                }
            }
            if (valid) {
                helpMsg.append(NbBundle.getMessage(PdCanvas.class,"PdCanvas.No_error_found"));
            }
            out.println(helpMsg.toString());
            out.flush();
        } catch (Exception e) {
            //e.printStackTrace();
            mLog.warning("Exception :" + e.getMessage());
        }
        
    }
    
    void validatePlan() {
        Plan plan = getDoc().getPlan();
        InputOutput io = IOProvider.getDefault().getIO(plan.getFullName(), false);
        io.select();
        OutputWriter out = io.getOut();
        try {
            out.reset();
        } catch (Exception e) {
            e.printStackTrace();
        }
        PdModel doc = getDoc();
        int errorCnt = 0;
        int warningCnt = 0;
        // To include plan level valiation error reporting.
        TcgComponentValidationReport planValidationReport = plan.validate();
        if(!planValidationReport.isOK()) {
            try {
                out.println(plan.getFullName() + ": ", null);
                List msgList = planValidationReport.getMessageList();
                for (int i = 0, I = msgList.size(); i < I; i++) {
                    TcgComponentValidationMsg msg = (TcgComponentValidationMsg)msgList.get(i);
                    String type = "";
                    if (msg.getType().equals(VALIDATION_ERROR_KEY)) {
                        type = "error";
                        errorCnt++;
                    } else {
                        type = "warning";
                        warningCnt++;
                    }
                    out.println("\t" + type + ": " + msg.getText(), null);
                }
            } catch(Exception e) {
                mLog.log(Level.WARNING,e.getMessage(),e);
            }
        }
        
        JGoListPosition pos = doc.getFirstObjectPos();
        try {
            while (pos != null) {
                JGoObject obj = doc.getObjectAtPos(pos);
                pos = doc.getNextObjectPos(pos);
                if (obj instanceof EntityNode) {
                    EntityNode node = (EntityNode)obj;
                    TcgComponent component = node.getComponent();
                    TcgComponentValidationReport r = component.validate();
                    if (!r.isOK()) {
                        PdMessageListener msgListener = (PdMessageListener)mMsgListenerTable.get(node);
                        if (msgListener == null) {
                            msgListener = new PdMessageListener(this, node);
                            mMsgListenerTable.put(node, msgListener);
                        }
                        out.println(node.getLabelString() + ": ");
                        List msgList = r.getMessageList();
                        for (int i = 0, I = msgList.size(); i < I; i++) {
                            TcgComponentValidationMsg msg = (TcgComponentValidationMsg)msgList.get(i);
                            String type = "";
                            if (msg.getType().equals(VALIDATION_ERROR_KEY)) {
                                type = "error";
                                errorCnt++;
                            } else {
                                type = "warning";
                                warningCnt++;
                            }
                            out.println("\t" + type + ": " + msg.getText(), msgListener);
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
        if (errorCnt == 0 && warningCnt == 0) {
            out.println(NbBundle.getMessage(PdCanvas.class,"PdCanvas.event_process_validates_succesfully"));
        }
        out.flush();
    }
    
    // toggle the grid appearance
    void showGrid() {
        int style = getGridStyle();
        if (style == JGoView.GridInvisible) {
            style = JGoView.GridDot;
            setGridPen(JGoPen.black);
            setSnapMove(JGoView.SnapJump);
        } else {
            style = JGoView.GridInvisible;
            setSnapMove(JGoView.NoSnap);
        }
        setGridStyle(style);
    }
    
    
    void zoomIn() {
        double newscale = Math.rint(getScale() / 0.9f * 100f) / 100f;
        setScale(newscale);
    }
    
    void zoomOut() {
        double newscale = Math.rint(getScale() * 0.9f * 100f) / 100f;
        setScale(newscale);
    }
    
    void zoomNormal() {
        setScale(1.0d);
    }
    
    void zoomToFit() {
        double newscale = 1;
        if (!getDocument().isEmpty()){
            double extentWidth = getExtentSize().width;
            double printWidth = getPrintDocumentSize().width;
            double extentHeight = getExtentSize().height;
            double printHeight = getPrintDocumentSize().height;
            newscale = Math.min((extentWidth / printWidth),(extentHeight / printHeight));
        }
        if (newscale > 2) {
            newscale = 1;
        }
        newscale *= getScale();
        setScale(newscale);
        setViewPosition(0, 0);
    }
    
    // Behavior
    
    public PdCanvas() {
        super();
    }
    
    public void initialize(PlanDesigner designer) {
        mDesigner = designer;
        addViewListener(this);
        setGridWidth(10);
        setGridHeight(10);
    }
    
    // handle DELETE, HOME, and arrow keys as well as the page up/down keys
    public void onKeyEvent(KeyEvent evt) {
        PdModel doc = getDoc();
        int t = evt.getKeyCode();
        if (t == KeyEvent.VK_DELETE) {
            if (doc.isModifiable()) {
                doc.startTransaction();
                deleteSelection();
                doc.endTransaction("deleteByKey");
            }
        } else if (t == KeyEvent.VK_HOME) {
            setViewPosition(0, 0);
        } else if (t == KeyEvent.VK_RIGHT) {
            if (doc.isModifiable()) {
                doMoveSelection(0, getGridWidth(), 0, EventMouseUp);
            }
        } else if (t == KeyEvent.VK_LEFT) {
            if (doc.isModifiable()) {
                doMoveSelection(0, -getGridWidth(), 0, EventMouseUp);
            }
        } else if (t == KeyEvent.VK_DOWN) {
            if (doc.isModifiable()) {
                doMoveSelection(0, 0, getGridHeight(), EventMouseUp);
            }
        } else if (t == KeyEvent.VK_UP) {
            if (doc.isModifiable()) {
                doMoveSelection(0, 0, -getGridHeight(), EventMouseUp);
            }
        } else if (t == KeyEvent.VK_ESCAPE) {
            getSelection().clearSelection();
        } else if (Character.isLetter(evt.getKeyChar())) {
            if (!selectNextNode(evt.getKeyChar())) {
                Toolkit.getDefaultToolkit().beep();
            }
        } else {
            super.onKeyEvent(evt);
        }
    }
    
    private EntityNode lastSelectedNode = null;
    public void mouseSelect(EntityNode node) {
        selectObject(node);
        lastSelectedNode = node;
        node.mouseSelect(this);
    }
    
    public boolean doMouseDblClick(int modifiers, java.awt.Point dc, java.awt.Point vc) {
        boolean selectableOnly = true;
        JGoObject obj = pickDocObject(dc, selectableOnly);
        if (obj instanceof EntityNode &&
                getCurrentMouseEvent() != null) {
            mouseSelect((EntityNode)obj);
            doPopupCutomizer((EntityNode)obj);
            return true;
        }
        return super.doMouseDblClick(modifiers, dc, vc);
    }
    
    private void doPopupCutomizer(EntityNode node) {
        TcgComponent comp = node.getComponent();
        if (!comp.hasProperty(PROPERTY_EDITOR_KEY)) {
            return;
        }
        try {
            TcgComponentNode n = new TcgComponentNode(comp, mDesigner.getPlan(), node);
            TcgComponentNodeProperty p = TcgComponentNodeProperty.newInstance(PROPERTY_EDITOR_KEY, n);
            TcgComponentNodePropertyCustomizerDialogManager.showDialog(p);
        } catch (Exception e) {
        }
    }
    
    public boolean doMouseDown(int modifiers, Point dc, Point vc) {
        boolean selectableOnly = true;
        JGoObject obj = pickDocObject(dc, selectableOnly);
        if (obj == null) {
            // Canvas is selected instead of things on the canvas
            // Show the properties of Plan
            mDesigner.showPropertyPane(mDesigner.getPlan(), this);
            return super.doMouseDown(modifiers, dc, vc);
        }
        // otherwise implement the default behavior
        return super.doMouseDown(modifiers, dc, vc);
    }
    
    public void documentChanged(JGoDocumentEvent evt)    {
        switch (evt.getHint()) {
        case JGoDocumentEvent.CHANGED:
            Object o = evt.getObject(); // Source Object
            if (o != null && o instanceof SimpleNodeLabel) {
                // When the SimpleNode is dragged around, there is a repainting problem with
                // its SimpleNodeLabel;
                // SimpleNodeLabel label = (SimpleNodeLabel) o;
                repaint();
            }
            break;
        case JGoDocumentEvent.REMOVED:
            repaint();
            break;
        }
        mDesigner.setDirty();
        super.documentChanged(evt);
    }
    
    // implement JGoViewListener
    // just need to keep the actions enabled appropriately
    // depending on the selection
    public void viewChanged(JGoViewEvent e) {
        // if the selection changed, maybe some commands need to
        // be disabled or re-enabled
        switch(e.getHint()) {
        case JGoViewEvent.UPDATE_ALL:
        case JGoViewEvent.SELECTION_GAINED:
        case JGoViewEvent.SELECTION_LOST:
        case JGoViewEvent.SCALE_CHANGED:
            PdAction.updateAllActions(getDesigner());
            break;
        }
    }
    
    // Override JGoView's newLink to force the creation of a Link
    // instead of JGoLink
    // let PdModel do the work
    public void newLink(JGoPort from, JGoPort to) {
        getDoc().startTransaction();
        Link l = getDoc().newLink(from, to);
        fireUpdate(JGoViewEvent.LINK_CREATED, 0, l);
        getDoc().endTransaction("newLink");
        
        // In case the toNode was selected just before the new link is created,
        // refresh property pane to show the new property values
        mDesigner.refreshPropertyPane();
    }
    
    public boolean matchesNodeLabel(EntityNode node, char c) {
        if (node == null) {
            return false;
        }
        String name = node.getText();
        return (name.length() > 0 && Character.toUpperCase(name.charAt(0)) == c);
    }
    
    public boolean selectNextNode(char c) {
        c = Character.toUpperCase(c);
        
        JGoDocument doc = getDocument();
        
        EntityNode startnode = null;
        JGoObject obj = getSelection().getPrimarySelection();
        if (obj != null && obj instanceof EntityNode) {
            startnode = (EntityNode)obj;
        }
        
        JGoListPosition startpos = null;
        if (startnode != null) {
            startpos = doc.findObject(startnode);
        }
        
        JGoListPosition pos = startpos;
        if (pos != null) {
            pos = doc.getNextObjectPosAtTop(pos);
        }
        
        while (pos != null) {
            obj = doc.getObjectAtPos(pos);
            pos = doc.getNextObjectPosAtTop(pos);
            
            if (obj instanceof EntityNode) {
                EntityNode pn = (EntityNode)obj;
                if (matchesNodeLabel(pn, c)) {
                    mouseSelect(pn);
                    scrollRectToVisible(pn.getBoundingRect());
                    return true;
                }
            }
        }
        pos = doc.getFirstObjectPos();
        while (pos != null && pos != startpos) {
            obj = doc.getObjectAtPos(pos);
            pos = doc.getNextObjectPosAtTop(pos);
            
            if (obj instanceof EntityNode) {
                EntityNode pn = (EntityNode)obj;
                if (matchesNodeLabel(pn, c)) {
                    mouseSelect(pn);
                    scrollRectToVisible(pn.getBoundingRect());
                    return true;
                }
            }
        }
        return false;
    }
    
    public boolean validLink(JGoPort from, JGoPort to) {
        JGoObject fromObj = from.getParentNode();
        JGoObject toObj = to.getParentNode();
        if (!(fromObj instanceof EntityNode) || !(toObj instanceof EntityNode)) {
            return false;
        }
        EntityNode fromNode = (EntityNode)fromObj;
        EntityNode toNode = (EntityNode)toObj;
        
        if (fromNode.getOutputType().equals(IO_TYPE_TABLE)) {
            if (toNode.getStaticInputCount() >= toNode.getStaticInputMaxCount()) {
                return false;
            }
        } else {
            // Check input count
            if (toNode.getInputCount() >= toNode.getInputMaxCount()) {
                return false;
            }
            // Check type compatibility
            if (!toNode.getInputType().equals(fromNode.getOutputType())) {
                return false;
            }
        }
        // Set a flag in the node being linked from
        boolean bReturn = true;
        int nFlags = fromNode.getFlags();
        fromNode.setFlags(fromNode.getFlags() | 0x10000);
        
        // Recursively traverse nodes starting from node being linked to
        // looking for the flag.  If found, we have a circular path.
        if (toObj instanceof EntityNode) {
            bReturn =  !toNode.downstreamNodeContainsFlag(0x10000);
        }
        fromNode.setFlags(fromNode.getFlags() & ~(0x10000));
        return bReturn && super.validLink(from, to);
    }
    
    public void destroy() {
        getDoc().releasePlan();
        for (Iterator itr = mMsgListenerTable.values().iterator(); itr.hasNext();) {
            PdMessageListener listener = (PdMessageListener)itr.next();
            listener.destroy();
        }
        mMsgListenerTable.clear();
    }
    
    //==================DnD Begin==================================
    private static boolean isDragAcceptable(DropTargetDragEvent e) {
        DataFlavor[] dfs = e.getCurrentDataFlavors();
        if ((dfs.length == 0) || e.isDataFlavorSupported(PaletteController.ITEM_DATA_FLAVOR)) {
            return true;
        }
        return false;
    }
    
    public void dragEnter(DropTargetDragEvent e) {
        if (!isDragAcceptable(e)) {
            super.dragEnter(e);
            return;
        }
        return;
    }
    
    public void dragOver(DropTargetDragEvent e) {
        if (!isDragAcceptable(e)) {
            super.dragOver(e);
            return;
        }
        return;
    }
    
    public void drop(DropTargetDropEvent e) {
        Transferable transferable = e.getTransferable();
        if (transferable.isDataFlavorSupported(PaletteController.ITEM_DATA_FLAVOR)) {
            try {
                Lookup itemLookup = (Lookup)transferable.getTransferData(PaletteController.ITEM_DATA_FLAVOR);
                TcgActiveEditorDrop iaed = (TcgActiveEditorDrop)itemLookup.lookup(ActiveEditorDrop.class);
                String ctPath = iaed.getPath();
                mLog.info("drop ctPath: " + ctPath);
                TcgComponentType type = TcgModelManager.getTcgComponentType(ctPath);
                Plan plan = mDesigner.getPlan();
                TcgComponent comp = plan.addNewOperator(type);
                Point vc = e.getLocation();
                Point dc = viewToDocCoords(vc);
                EntityNode node = new EntityNode(plan, comp, dc);
                PdModel model = getDoc();
                model.startTransaction();
                model.addObjectAtTail(node);
                fireUpdate(JGoViewEvent.EXTERNAL_OBJECTS_DROPPED, 0, null);
                model.endTransaction("newEntityNode");
            } catch (Exception ex) {
                mLog.log(Level.SEVERE, "drop failed.", ex);
            }
            return;
        }
        
        super.drop(e);
    }
    
    public void dropActionChanged(DropTargetDragEvent e) {
        if (!isDragAcceptable(e)) {
            super.dropActionChanged(e);
            return;
        }
        return;
    }
    
    //==================DnD End ===================================
    
    //==================Cut,Copy,and Paste Begin===========================
    /**
     * Copy JGoObjects from the clipboard into this document.
     *
     * @param clipboard the clipboard supporting the standard JGoDocument
     *        DataFlavor containing objects to be copied to this document
     *        using the default copying method.
     */
    public JGoCopyEnvironment pasteFromClipboard(Clipboard clipboard) {
        Transferable contents = clipboard.getContents(this);
        DataFlavor jgoflavor = JGoDocument.getStandardDataFlavor();
        if ((contents != null) && contents.isDataFlavorSupported(jgoflavor)) {
            try {
                PdModel doc = getDoc();
                if (doc != null) {
                    // copy objects from selection into this document
                    JGoObjectSimpleCollection sel =
                            (JGoObjectSimpleCollection)contents.getTransferData(jgoflavor);
                    if (doc.getDefaultLayer().isModifiable())
                        return doc.islandCopyFromCollection(sel, new Point(5, 5));
                    else
                        return null;
                }
            } catch (Exception e) {
                mLog.log(Level.SEVERE,"pasteFromClipbaord failed.", e);
            }
        }
        return null;
    }
    //==================Cut,Copy,and Paste End===========================
    
    public void deleteSelection() {
        List nodeList = new ArrayList();
        List linkList = new ArrayList();
        JGoSelection sel = getSelection();
        JGoListPosition pos = sel.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = sel.getObjectAtPos(pos);
            pos = sel.getNextObjectPos(pos);
            if (obj.getLayer() != null && !obj.getLayer().isModifiable()) {
                // skip this object because it's in an unmodifiable layer
                continue;
            }
            obj = obj.getDraggingObject();
            if (obj instanceof EntityNode) {
                nodeList.add(((EntityNode)obj).getComponent());
                continue;
            }
            if (obj instanceof Link) {
                linkList.add(((Link)obj).getComponent());
                continue;
            }
        }
        super.deleteSelection();
        Plan plan = mDesigner.getPlan();
        for (int i = 0, I = nodeList.size(); i < I; i++) {
            TcgComponent c = (TcgComponent)nodeList.get(i);
            plan.removeOperator(c);
        }
        for (int i = 0, I = linkList.size(); i < I; i++) {
            TcgComponent c = (TcgComponent)linkList.get(i);
            plan.removeLink(c);
        }
    }
    
    // ================= Support for Testtools Begin============================
    public EntityNode findNodeByLabel(String label) {
        PdModel doc = getDoc();
        JGoListPosition pos = doc.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = doc.getObjectAtPos(pos);
            pos = doc.getNextObjectPos(pos);
            if (obj instanceof EntityNode) {
                EntityNode node = (EntityNode)obj;
                String name = node.getText();
                if (name.equals(label)) {
                    return node;
                }
            }
        }
        return null;
    }
    
    public Link findLink(String fromNodeLabel, String toNodeLabel) {
        PdModel doc = getDoc();
        JGoListPosition pos = doc.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = doc.getObjectAtPos(pos);
            pos = doc.getNextObjectPos(pos);
            if (obj instanceof Link) {
                Link link = (Link)obj;
                EntityNode fromNode = link.getFromNode();
                EntityNode toNode = link.getToNode();
                if (fromNode.getText().equals(fromNodeLabel) && toNode.getText().equals(toNodeLabel)) {
                    return link;
                }
            }
        }
        return null;
    }
    
    public boolean isNodeSelected(String label) {
        JGoSelection sel = getSelection();
        if (sel.isEmpty()) {
            return false;
        }
        JGoListPosition pos = sel.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = sel.getObjectAtPos(pos);
            pos = sel.getNextObjectPos(pos);
            if (obj instanceof EntityNode) {
                EntityNode node = (EntityNode)obj;
                if (node.getText().equals(label)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    public boolean isLinkSelected(String fromNodeLabel, String toNodeLabel) {
        JGoSelection sel = getSelection();
        if (sel.isEmpty()) {
            return false;
        }
        JGoListPosition pos = sel.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = sel.getObjectAtPos(pos);
            pos = sel.getNextObjectPos(pos);
            if (obj instanceof Link) {
                Link link = (Link)obj;
                EntityNode fromNode = link.getFromNode();
                EntityNode toNode = link.getToNode();
                if (fromNode.getText().equals(fromNodeLabel) && toNode.getText().equals(toNodeLabel)) {
                    return true;
                }
            }
        }
        return false;
    }
    
    public int getContainerCount(Point p) {
        int cnt = 0;
        PdModel doc = getDoc();
        JGoListPosition pos = doc.getFirstObjectPos();
        while (pos != null) {
            JGoObject obj = doc.getObjectAtPos(pos);
            pos = doc.getNextObjectPos(pos);
            if (obj instanceof Link || obj instanceof EntityNode) {
                if (obj.isPointInObj(p)) {
                    cnt++;
                }
            }
        }
        return cnt;
    }
    
    public boolean isOrthogonalFlows() {
        return getDoc().isOrthogonalFlows();
    }
    
    // TcgCoomponentNodeView
    public void updateTcgComponentNodeView() {
    }
    
}
