/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2007 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.sun.manager.jbi.management;

import java.io.IOException;
import java.io.Serializable;
import java.io.StringReader;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Date;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.Set;

import javax.management.Attribute;
import javax.management.AttributeList;
import javax.management.AttributeNotFoundException;
import javax.management.InstanceNotFoundException;
import javax.management.InvalidAttributeValueException;
import javax.management.MBeanAttributeInfo;
import javax.management.MBeanException;
import javax.management.MBeanInfo;
import javax.management.MBeanOperationInfo;
import javax.management.MBeanServerConnection;
import javax.management.MalformedObjectNameException;
import javax.management.ObjectName;
import javax.management.QueryExp;
import javax.management.ReflectionException;
import javax.management.remote.JMXConnector;
import javax.management.remote.JMXConnectorFactory;
import javax.management.remote.JMXServiceURL;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import org.netbeans.modules.sun.manager.jbi.GenericConstants;
import org.netbeans.modules.sun.manager.jbi.util.MyMBeanAttributeInfo;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.InputSource;


/**
 * Allows configuration of JBI Service engines, and JBI Binding components
 * at runtime
 *
 * @author Graj
 * @author jqian
 */
public class JBIComponentConfigurator implements Serializable {
    
    private static final String OLD_OBJECT_NAME_PREFIX = "com.sun.ebi:ServiceType=Configuration,InstallationType="; // NOI18N          
    private static final String OLD_OBJECT_NAME_SUFFIX = ",IdentificationName="; // NOI18N
    
    private static final String NEW_OBJECT_NAME_PREFIX = "com.sun.jbi:JbiName=server,CustomControlName=Configuration,ComponentName="; // NOI18N          
    private static final String NEW_OBJECT_NAME_SUFFIX = ",ControlType=Custom,*"; // NOI18N
        
    private String oldObjectNameString;
    private String newObjectNameString;
    
    private List<String> operationNames = new ArrayList<String>();
    
    private MBeanServerConnection connection;
    
    private ObjectName objectName = null;
    private Properties typeProperties = new Properties();
    
    private static final String SCHEMA_RETRIEVAL_OPERATION = "retrieveConfigurationDisplaySchema";   // NOI18N
    private static final String DATA_RETRIEVAL_OPERATION = "retrieveConfigurationDisplayData";       // NOI18N
    
    private static final String DISPLAY_NAME = "displayName"; // NOI18N
    private static final String DISPLAY_DESCRIPTION = "displayDescription"; // NOI18N
    private static final String IS_PASSWORD_FIELD = "isPasswordField"; // NOI18N
    private static final String NEEDS_APPLICATION_RESTART = "isApplicationRestartRequired"; // NOI18N
    private static final String NEEDS_COMPONENT_RESTART = "isComponentRestartRequired"; // NOI18N
    private static final String NEEDS_SERVER_RESTART = "isServerRestartRequired"; // NOI18N
                    
    
    /**
     *
     * @param type
     * @param name
     * @param serverConnection
     */
    public JBIComponentConfigurator(String type, String name,
            MBeanServerConnection serverConnection)
            throws MalformedObjectNameException,
            AttributeNotFoundException, InstanceNotFoundException,
            MBeanException, ReflectionException, IOException{
        super();
        this.connection = serverConnection;
        
        this.oldObjectNameString = 
                OLD_OBJECT_NAME_PREFIX + type + OLD_OBJECT_NAME_SUFFIX + name;
        this.newObjectNameString = 
                NEW_OBJECT_NAME_PREFIX + name + NEW_OBJECT_NAME_SUFFIX;
        this.initialize();
        
        if(this.objectName != null) {
            this.populateTypeProperties();
            this.retrieveOperationNames();
        }        
    }
    
    
    /**
     *
     * @throws MalformedObjectNameException
     * @throws AttributeNotFoundException
     * @throws InstanceNotFoundException
     * @throws MBeanException
     * @throws ReflectionException
     * @throws IOException
     */
    private void initialize() throws MalformedObjectNameException,
            AttributeNotFoundException, InstanceNotFoundException,
            MBeanException, ReflectionException, IOException {
        QueryExp queryExpression = null;
        
        if (this.connection != null) {
            ObjectName objName = new ObjectName(this.newObjectNameString);
            Set set = this.connection.queryNames(objName, queryExpression);
            Iterator iterator = set.iterator();
            if (iterator != null && iterator.hasNext()) {
                objName = (ObjectName) iterator.next();
                if (objName != null) {
                    this.objectName = objName;
                }
            }
            
            if (this.objectName == null) {
                objName = new ObjectName(this.oldObjectNameString);
                set = this.connection.queryNames(objName, queryExpression);
                iterator = set.iterator();
                if (iterator != null && iterator.hasNext()) {
                    objName = (ObjectName) iterator.next();
                    if (objName != null) {
                        this.objectName = objName;
                    }
                }
            }
        } else {
            System.out.println("Connection Failed"); // NOI18N
        }
    }
    
    /**
     *
     */
    private void populateTypeProperties() {
        Properties properties = this.getProperties();
        Enumeration enumeration = properties.keys();
        String key = null;
        Object value = null;
        String signature = null;
        while(enumeration.hasMoreElements() == true) {
            key = (String)enumeration.nextElement();
            if(key != null) {
                value = properties.get(key);
                signature = this.getSignature(value);
                if(signature != null) {
                    typeProperties.put(key, signature);
                }
            }
        }
    }
    
    /**
     * get parameter signatures based on the object type class
     *
     * @param params
     * @return
     */
    private String[] getSignatures(Object[] params) {
        if (params == null || params.length == 0) {
            return null;
        }
        String[] signatures = new String[params.length];
        for (int index = 0; index < params.length; index++) {
            if (params[index] == null) {
                signatures[index] = "java.lang.Object"; // NOI18N
            } else {
                signatures[index] = params[index].getClass().getName();
            }
        }
        return signatures;
    }
    
    /**
     *
     * @param param
     * @return
     */
    private String getSignature(Object param) {
        String result = "java.lang.Object"; // NOI18N
        if(param != null) {
            result = param.getClass().getName();
        }
        return result;
    }
    
    /**
     *
     * @param objectName
     * @param operationName
     * @param parameters
     * @return
     */
    private Object invoke(ObjectName objectName, String operationName, Object[] parameters) {
        Object result = null;
        
        String[] signature = this.getSignatures(parameters);
        try {
            result = this.connection.invoke(objectName,
                    operationName,
                    parameters,
                    signature);
        } catch (ReflectionException e) {
            // Undefined operation like " retrieveConfigurationDisplayData" is OK
            System.err.println("WARNING: " + e.getMessage());
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        return result;
    }
    
    /**
     * Gets the list of MBean attributes and their values
     *
     */
    public Properties getProperties() {
        Properties properties = new Properties();
        Attribute attribute = null;
        String key; Object value = null;
        MBeanInfo mbeanInfo = null;
        String[] allAttributeNames = null;
        AttributeList attributeList = null;
        
        if((this.connection != null) && (this.objectName != null)) {
            try {
                mbeanInfo = connection.getMBeanInfo(this.objectName);
                MBeanAttributeInfo[] attrs = mbeanInfo.getAttributes();
                allAttributeNames = new String[attrs.length];
                for (int attrCount = 0; attrCount < attrs.length; attrCount++) {
                    allAttributeNames[attrCount] = attrs[attrCount].getName();
                }
                attributeList = connection.getAttributes(this.objectName, allAttributeNames);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        if(attributeList != null) {
            Iterator iterator = attributeList.iterator();
            while(iterator.hasNext() == true) {
                attribute = (Attribute) iterator.next();
                if(attribute != null) {
                    key = attribute.getName();
                    value = attribute.getValue();
                    if((key != null) && (value != null)) {
                        properties.put(key, value);
                    }
                }
            }
        }
        return properties;
    }
        
    /**
     * Gets the property map mapping from Attribute to MBeanAttributeInfo.
     * @return
     */
    public Map<Attribute, ? extends MBeanAttributeInfo> getPropertyMap() {
        
        Attribute attribute = null;
        MBeanInfo mbeanInfo = null;
        MBeanAttributeInfo[] attrInfoes = null;
        String[] allAttributeNames = null;
        AttributeList attributeList = null;
        Map<Attribute, MBeanAttributeInfo> map = 
                new HashMap<Attribute, MBeanAttributeInfo>();
        
        if ((this.connection != null) && (this.objectName != null)) {
            try {
                mbeanInfo = connection.getMBeanInfo(this.objectName);
                attrInfoes = mbeanInfo.getAttributes();
                allAttributeNames = new String[attrInfoes.length];
                for (int attrCount = 0; attrCount < attrInfoes.length; attrCount++) {
                    allAttributeNames[attrCount] = attrInfoes[attrCount].getName();
                }
                attributeList = connection.getAttributes(
                        this.objectName, allAttributeNames);
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
       
        String configXmlData = getConfigurationXmlData();
        
        if (configXmlData == null) {
            // Fallback on regular attributes if the component does not have 
            // configuration schema defined yet.
            if (attributeList != null) {
                Iterator iterator = attributeList.iterator();
                int index = 0;
                while (iterator.hasNext()) {
                    attribute = (Attribute) iterator.next();
                    if (attribute != null) {
                        map.put(attribute, attrInfoes[index]);
                    }
                    index++;
                }
            }
        } else {
            Map<String, String> attrDisplayNameMap = new HashMap<String, String>();
            Map<String, String> attrDescriptionMap = new HashMap<String, String>();
            List<String> passwordAttrList = new ArrayList<String>();
            List<String> needsServerRestartAttrList = new ArrayList<String>();
            List<String> needsComponentRestartAttrList = new ArrayList<String>();
            List<String> needsApplicationRestartAttrList = new ArrayList<String>();

            try {
                DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
                factory.setNamespaceAware(true);
                DocumentBuilder builder = factory.newDocumentBuilder();
                Document document = builder.parse(
                        new InputSource(new StringReader(configXmlData)));

                Element root = document.getDocumentElement();
                NodeList children = root.getChildNodes();
                for (int i = 0; i < children.getLength(); i++) {
                    if (children.item(i) instanceof Element) {
                        Element child = (Element) children.item(i);
                        String name = child.getLocalName();
                        String displayName = child.getAttribute(DISPLAY_NAME);
                        String description = child.getAttribute(DISPLAY_DESCRIPTION);
                        String isPassword = child.getAttribute(IS_PASSWORD_FIELD);
                        String needsApplicationRestart = child.getAttribute(NEEDS_APPLICATION_RESTART);
                        String needsComponentRestart = child.getAttribute(NEEDS_COMPONENT_RESTART);
                        String needsServerRestart = child.getAttribute(NEEDS_SERVER_RESTART);
                        
                        attrDisplayNameMap.put(name, displayName);
                        attrDescriptionMap.put(name, description);
                        if (Boolean.valueOf(isPassword)) {
                            passwordAttrList.add(name);
                        }
                        if (needsApplicationRestart != null &&
                                Boolean.valueOf(needsApplicationRestart)) {
                            needsApplicationRestartAttrList.add(name);
                        }
                        if (needsComponentRestart != null &&
                                Boolean.valueOf(needsComponentRestart)) {
                            needsComponentRestartAttrList.add(name);
                        }
                        if (needsServerRestart != null && 
                                Boolean.valueOf(needsServerRestart)) {
                            needsServerRestartAttrList.add(name);
                        }
                    }
                }

                if (attributeList != null) {
                    Iterator iterator = attributeList.iterator();
                    int index = 0;
                    while (iterator.hasNext()) {
                        attribute = (Attribute) iterator.next();
                        if (attribute != null) {
                            String attrName = attrInfoes[index].getName();
                            String displayName = attrDisplayNameMap.get(attrName);
                            String description = attrDescriptionMap.get(attrName);
                            if (displayName == null || description == null) {
                                // e.x., EnvironmentVariables
                                map.put(attribute, attrInfoes[index]);
                            } else {
                                map.put(new Attribute(attrName, attribute.getValue()),
                                    new MyMBeanAttributeInfo(
                                        displayName, 
                                        attrInfoes[index].getType(), 
                                        description,
                                        attrInfoes[index].isReadable(), 
                                        attrInfoes[index].isWritable(), 
                                        attrInfoes[index].isIs(),
                                        passwordAttrList.contains(attrName),
                                        needsApplicationRestartAttrList.contains(attrName),
                                        needsComponentRestartAttrList.contains(attrName),
                                        needsServerRestartAttrList.contains(attrName)
                                        ));
                            }
                        }
                        index++;
                    }
                }
            } catch (Exception e) {
                e.printStackTrace();
            }
        }
        return map;
    }
        
    public Object getPropertyValue(String attrName) {
        if (attrName != null && connection != null && objectName != null) {
            try {
                return connection.getAttribute(objectName, attrName);
            } catch (Exception ex) {
                ex.printStackTrace();
            }
        }
        return null;
    }
    
    /**
     * Sets the MBean attribute
     *
     * @param key
     * @param value
     */
    public void setPropertyValue(String key, Object value)
    throws InvalidAttributeValueException,
            AttributeNotFoundException, InstanceNotFoundException,
            MBeanException, ReflectionException, IOException {
        if (key != null && value != null) {
            Attribute attribute = new Attribute(key, value);
            if (connection != null && objectName != null) {
                connection.setAttribute(this.objectName, attribute);
            }
        }
    }
    
    /**
     * Sets the MBean attribute
     *
     * @param key
     * @param value
     */
    public void setPropertyValue(String key, String value)
    throws InvalidAttributeValueException,
            AttributeNotFoundException, InstanceNotFoundException,
            MBeanException, ReflectionException, IOException {
       
        Object object = null;
        if ((key != null) && (this.typeProperties != null)){
            String type = (String)this.typeProperties.get(key);
            if (type != null) {
                if (type.equals(GenericConstants.OPEN_TYPE_CLASS_BIGDECIMAL)) {
                    object = new BigDecimal(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_BIGINTEGER)) {
                    object = new BigInteger(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_BOOLEAN)) {
                    object = new Boolean(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_BYTE)) {
                    object = new Byte(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_CHARACTER)) {
                    object = new Character(value.charAt(0));
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_DATE)) {
                    long longValue = (new Long(value)).longValue();
                    object = new Date(longValue);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_DOUBLE)) {
                    object = new Double(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_FLOAT)) {
                    object = new Float(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_INTEGER)) {
                    object = new Integer(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_LONG)) {
                    object = new Long(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_OBJECTNAME)) {
                    try {
                        object = new ObjectName(value);
                    } catch (MalformedObjectNameException e) {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                    } catch (NullPointerException e) {
                        // TODO Auto-generated catch block
                        e.printStackTrace();
                    }
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_SHORT)) {
                    object = new Short(value);
                } else if (type.equals(GenericConstants.OPEN_TYPE_CLASS_STRING)) {
                    object = value;
                }
            }
            if (object != null) {
                this.setPropertyValue(key, object);
            }
        }
    }
    
    public String getConfigurationSchema() {
        return (String) invoke(objectName, SCHEMA_RETRIEVAL_OPERATION, null);
    }
    
    public String getConfigurationXmlData() {
        return (String)invoke(objectName, DATA_RETRIEVAL_OPERATION, null);
    }
    
    /**
     * Returns a string array of available processes
     * @return a String array of available process names
     */
    private String[] listAvailableProcessNames() {
        String[] result = null;
        try {
            result = (String[])this.connection.invoke(objectName,
                    "listAvailableProcessNames", // NOI18N
                    null, null);
        } catch (Exception e) {
            e.printStackTrace();
        }
        
        return result;
    }
    
    /**
     * When the truncateAll method is called the Service engine
     * will delete all entries in all tables in all Service Engine
     * processes that are deployed
     * @return result
     */
    private String truncateAll() {
        String result = null;
        Object[] params = null;
        result = (String)this.invoke(objectName, "truncateAll", params); // NOI18N
        return result;
    }
    
    /**
     * When this method is called it will delete all entries in
     * all tables in the named process.
     * @param processName
     * @return result
     */
    private String truncate(String processName) {
        String result = null;
        Object[] params = new String[1];
        if(processName != null) {
            params[0] = processName;
            result = (String)this.invoke(objectName, "truncate", params); // NOI18N
        }
        return result;
    }
    
    /**
     * Determines if the operation is implemented by this MBean
     * @param operationName
     * @return true if valid operationName, false if not
     */
    private  boolean isValidOperation(String operationName) {
        if (operationName == null) {
            return false;
        }
        return this.operationNames.contains(operationName);
    }
    
    /**
     * Retrieves the list of operations supported by this MBean
     * @return
     */
    private void retrieveOperationNames() {
        try {
            if (this.objectName != null) {
                MBeanInfo beanInfo = this.connection.getMBeanInfo(this.objectName);
                if (beanInfo != null) {
                    MBeanOperationInfo[] info = beanInfo.getOperations();
                    for (int index = 0; index < info.length; index++) {
                        if (info[index] != null) {
                            String operationName = info[index].getName();
                            if (operationName != null) {
                                operationNames.add(operationName);
                            }
                        }
                    }
                }
            }
        } catch (Exception e) {
            e.printStackTrace();
        }
    }
    
    private static void test(MBeanServerConnection connection)
    throws MalformedObjectNameException, InvalidAttributeValueException,
            AttributeNotFoundException, InstanceNotFoundException,
            MBeanException, ReflectionException, IOException{
        String type = GenericConstants.BINDING_COMPONENTS_FOLDER_NAME;
        String name = "sun.http.binding"; // NOI18N
        JBIComponentConfigurator configurator = new JBIComponentConfigurator(type, name, connection);
        Properties properties = configurator.getProperties();
        Properties typeProperties = new Properties();
        Enumeration enumeration = properties.keys();
        String key = null;
        Object value = null;
        String signature = null;
        while(enumeration.hasMoreElements() == true) {
            key = (String)enumeration.nextElement();
            if(key != null) {
                value = properties.get(key);
                signature = configurator.getSignature(value);
                if(signature != null) {
                    typeProperties.put(key, signature);
                    if(signature.equals(GenericConstants.OPEN_TYPE_CLASS_STRING) == false) {
                        System.out.println(key+" = "+value); // NOI18N
                        configurator.setPropertyValue(key, value);
                    }
                }
            }
        }
    }
    
    /**
     *
     * @param args
     */
    public static void main(String[] args) {
        String PROTOCOL_CLASS = "com.sun.enterprise.admin.jmx.remote.protocol"; // NOI18N
        String HTTP_AUTH_PROPERTY_NAME = "com.sun.enterprise.as.http.auth"; // NOI18N
        String DEFAULT_HTTP_AUTH_SCHEME = "BASIC"; // NOI18N
        String ADMIN_USER_ENV_PROPERTY_NAME = "USER"; // NOI18N
        String ADMIN_PASSWORD_ENV_PROPERTY_NAME = "PASSWORD"; // NOI18N
        String RTS_HTTP_CONNECTOR = "s1ashttp"; // NOI18N
        
        String hostName = "127.0.0.1"; // NOI18N
        String port = "4848"; // NOI18N
        String userName = "admin"; // NOI18N
        String password = "adminadmin"; // NOI18N
        
        final Map<String, String> environment = new HashMap<String, String>();
        environment.put(JMXConnectorFactory.PROTOCOL_PROVIDER_PACKAGES, PROTOCOL_CLASS);
        environment.put(HTTP_AUTH_PROPERTY_NAME, DEFAULT_HTTP_AUTH_SCHEME);
        environment.put(ADMIN_USER_ENV_PROPERTY_NAME, userName);
        environment.put(ADMIN_PASSWORD_ENV_PROPERTY_NAME, password);
        
        try {
            int portValue = new Integer(port).intValue();
            MBeanServerConnection connection = null;
            
            JMXServiceURL serviceURL = new JMXServiceURL(RTS_HTTP_CONNECTOR, hostName, portValue);
            JMXConnector connector = JMXConnectorFactory.connect(serviceURL, environment);
            connection = connector.getMBeanServerConnection();
            test(connection);
            
        } catch (Exception exception) {
            exception.printStackTrace();
        }
        
    }
}
