 /*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * The Original Software is NetBeans. The Initial Developer of the Original
 * Software is Sun Microsystems, Inc. Portions Copyright 1997-2006 Sun
 * Microsystems, Inc. All Rights Reserved.
 *
 * If you wish your version of this file to be governed by only the CDDL
 * or only the GPL Version 2, indicate your decision by adding
 * "[Contributor] elects to include this software in this distribution
 * under the [CDDL or GPL Version 2] license." If you do not indicate a
 * single choice of license, a recipient has the option to distribute
 * your version of this file under either the CDDL, the GPL Version 2 or
 * to extend the choice of license to its licensees as provided above.
 * However, if you add GPL Version 2 code and therefore, elected the GPL
 * Version 2 license, then the option applies only if the new code is
 * made subject to such option by the copyright holder.
 */

package org.netbeans.modules.editor.settings.storage;

import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeSupport;
import java.lang.ref.WeakReference;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.WeakHashMap;
import java.util.logging.Level;
import java.util.logging.Logger;
import javax.swing.KeyStroke;
import org.netbeans.api.editor.mimelookup.MimePath;
import org.netbeans.api.editor.settings.KeyBindingSettings;
import org.netbeans.api.editor.settings.MultiKeyBinding;
import org.netbeans.modules.editor.settings.storage.api.EditorSettings;
import org.netbeans.modules.editor.settings.storage.api.KeyBindingSettingsFactory;
import org.openide.util.Utilities;

/**
 * KeyBindings settings are represented by List of keybindings.
 * The List contains the instances of {@link MultiKeyBinding}.
 * <br>
 * Instances of this class should be retrieved from the {@link org.netbeans.api.editor.mimelookup.MimeLookup}
 * for a given mime-type.
 * <br>
 * <font color="red">This class must NOT be extended by any API clients</font>
 *
 * @author Jan Jancura
 */
public final class KeyBindingSettingsImpl extends KeyBindingSettingsFactory {

    private static final Logger LOG = Logger.getLogger(KeyBindingSettingsImpl.class.getName());
    
    private static final Map<MimePath, WeakReference<KeyBindingSettingsImpl>> INSTANCES =
        new WeakHashMap<MimePath, WeakReference<KeyBindingSettingsImpl>>();
    
    public static synchronized KeyBindingSettingsImpl get(MimePath mimePath) {
        WeakReference<KeyBindingSettingsImpl> reference = INSTANCES.get(mimePath);
        KeyBindingSettingsImpl result = reference == null ? null : reference.get();
        
        if (result == null) {
            result = new KeyBindingSettingsImpl(mimePath);
            INSTANCES.put(mimePath, new WeakReference<KeyBindingSettingsImpl>(result));
        }
        
        return result;
    }
    
    private MimePath mimePath;
    private PropertyChangeSupport   pcs;
    private Map<String, Map<Collection<KeyStroke>, MultiKeyBinding>> keyMaps = new HashMap<String, Map<Collection<KeyStroke>, MultiKeyBinding>>();
    private KeyBindingSettingsImpl baseKBS;
    private Listener                listener;
    
    private String logActionName = null;
    
    /**
     * Construction prohibited for API clients.
     */
    private KeyBindingSettingsImpl (MimePath mimePath) {
        this.mimePath = mimePath;
        pcs = new PropertyChangeSupport (this);
        
        // init logging
        String myClassName = KeyBindingSettingsImpl.class.getName ();
        String value = System.getProperty(myClassName);
        if (value != null) {
            if (!value.equals("true")) {
                logActionName = System.getProperty(myClassName);
            }
        } else if (mimePath.size() == 1) {
            logActionName = System.getProperty(myClassName + '.' + mimePath.getMimeType(0));
        }
    }
    
    private boolean init = false;
    private void init () {
        if (init) return;
        init = true;
        if (mimePath.size() > 0) {
            baseKBS = get(MimePath.EMPTY);
        }
        listener = new Listener(this, baseKBS);
    }
    
    /**
     * Gets the keybindings list, where items are instances of {@link MultiKeyBinding}
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List<MultiKeyBinding> getKeyBindings() {
        return getKeyBindings(EditorSettingsImpl.getInstance().getCurrentKeyMapProfile());
    }
    
    /**
     * Gets the keybindings list, where items are instances of {@link MultiKeyBinding}
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List<MultiKeyBinding> getKeyBindings(String profile) {
        init ();
        
        // 1) get real profile
	profile = EditorSettingsImpl.getInstance().getInternalKeymapProfile(profile);
        
        Map<Collection<KeyStroke>, MultiKeyBinding> allShortcuts = new HashMap<Collection<KeyStroke>, MultiKeyBinding>();

        // Add base shortcuts
        if (baseKBS != null) {
            Map<Collection<KeyStroke>, MultiKeyBinding> baseShortcuts = baseKBS.getShortcuts(profile);
            allShortcuts.putAll(baseShortcuts);
        }

        // Add local shortcuts
        Map<Collection<KeyStroke>, MultiKeyBinding> localShortcuts = getShortcuts(profile);
        allShortcuts.putAll(localShortcuts);
        
        // Prepare the result
        List<MultiKeyBinding> result = new ArrayList<MultiKeyBinding>(allShortcuts.values());
        log ("getKeyBindings", result);
        
	return Collections.unmodifiableList(result);
    }

    private Map<Collection<KeyStroke>, MultiKeyBinding> getShortcuts(String profile) {
        synchronized (this) {
            Map<Collection<KeyStroke>, MultiKeyBinding> shortcuts = keyMaps.get(profile);
            
            if (shortcuts == null) {
                shortcuts = KeyMapsStorage.loadKeyMaps(mimePath, profile, false);
                keyMaps.put(profile, shortcuts);
            }
            
            return shortcuts;
        }
    }
    
    /**
     * Returns default keybindings list for given keymap name, where items 
     * are instances of {@link MultiKeyBinding}.
     *
     * @return List of {@link MultiKeyBinding}
     */
    public List<MultiKeyBinding> getKeyBindingDefaults(String profile) {
        // 1) get real profile
	profile = EditorSettingsImpl.getInstance().getInternalKeymapProfile(profile);
        return Collections.unmodifiableList(new ArrayList<MultiKeyBinding>(getDefaults(profile).values()));
    }
    
    /**
     * Gets the keybindings list, where items are instances of 
     * {@link MultiKeyBinding}.
     *
     * @return List of {@link MultiKeyBinding}
     */
    public void setKeyBindings (
        String profile, 
        List<MultiKeyBinding> keyBindings
    ) {
        synchronized (this) {
            log ("setKeyBindings", keyBindings);

            // 1) get real profile
            profile = EditorSettingsImpl.getInstance().getInternalKeymapProfile(profile);

            init ();
            if (keyBindings == null) {
                // 1) delete user changes / user profile
                keyMaps.remove (profile);
                KeyMapsStorage.deleteProfile(mimePath, profile, false);
                return;
            }

            // 1) convert keyBindings: List (MultiKeyBinding) to 
            //            m: Map (List (KeyStroke) > MultiKeyBinding).
            Map<Collection<KeyStroke>, MultiKeyBinding> shortcuts = new HashMap<Collection<KeyStroke>, MultiKeyBinding>();
            for(MultiKeyBinding mkb : keyBindings) {
                shortcuts.put(mkb.getKeyStrokeList(), mkb);
            }

            Map<Collection<KeyStroke>, MultiKeyBinding> shortcutsForCache = new HashMap<Collection<KeyStroke>, MultiKeyBinding>(shortcuts);
            keyMaps.put(profile, shortcutsForCache);

            // 2) compute removed shortcuts & remove unchanged maappings from m
            Map<Collection<KeyStroke>, MultiKeyBinding> defaults = getDefaults(profile);
            Set<Collection<KeyStroke>> removed = new HashSet<Collection<KeyStroke>>();

            for(Collection<KeyStroke> shortcut : defaults.keySet()) {
                MultiKeyBinding mkb2 = defaults.get(shortcut);
                if (!shortcuts.containsKey (shortcut)) {
                    removed.add (shortcut);
                } else {
                    MultiKeyBinding mkb1 = (MultiKeyBinding) shortcuts.get (shortcut);
                    if (mkb1.getActionName ().equals (mkb2.getActionName ()))
                        shortcuts.remove (shortcut);
                }
            }

            log ("  changed:", shortcuts.values ());
            log ("  removed:", removed);
            log ("", Collections.EMPTY_LIST);

            // 3) save diff & removed
            listener.removeListeners ();
            KeyMapsStorage.saveKeyMaps(mimePath, profile, false, shortcuts.values(), removed);

            listener.addListeners ();
        }
        
        pcs.firePropertyChange (null, null, null);
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be registerred
     */
    public void addPropertyChangeListener (PropertyChangeListener l) {
        pcs.addPropertyChangeListener (l);
    }
    
    /**
     * PropertyChangeListener registration.
     *
     * @param l a PropertyChangeListener to be unregisterred
     */
    public void removePropertyChangeListener (PropertyChangeListener l) {
        pcs.removePropertyChangeListener (l);
    }    
    
    // other methods ...........................................................
    
    // Map (String (profile) > Map (String (shortcut) > MultiKeyBinding)).
    private Map<String, Map<Collection<KeyStroke>, MultiKeyBinding>> defaults = 
        new HashMap<String, Map<Collection<KeyStroke>, MultiKeyBinding>>();
    
    /**
     * Returns default shortcut set for given profile. Returns empty map for 
     * custom (user defined) profiles.
     *
     * @return Map (List (KeyStroke) > MultiKeyBinding)
     */
    private Map<Collection<KeyStroke>, MultiKeyBinding> getDefaults(String profile) {
        if (!defaults.containsKey (profile)) {
            Map<Collection<KeyStroke>, MultiKeyBinding> keyMap = KeyMapsStorage.loadKeyMaps(mimePath, profile, true);
            defaults.put(profile, keyMap);
        }
        
        return defaults.get (profile);
    }

    /**
     * External change.
     */
    private void refresh () {
        synchronized (this) {
            keyMaps.clear();
            log ("refresh", Collections.EMPTY_SET);
        }
        
        pcs.firePropertyChange (null, null, null);
    }
    
    private void log (String text, Collection keymap) {
        if (!LOG.isLoggable(Level.FINE)) {
            return;
        }
        if (text.length() != 0) {
            if (mimePath.size() == 1) {
                text += " " + mimePath.getMimeType(0);
            }
            text += " " + EditorSettingsImpl.getInstance().getCurrentKeyMapProfile();
        }
        if (keymap == null) {
            LOG.fine(text + " : null");
            return;
        }
        LOG.fine(text);
        Iterator it = keymap.iterator ();
        while (it.hasNext ()) {
            Object mkb = it.next ();
            if (logActionName == null || !(mkb instanceof MultiKeyBinding)) {
                LOG.fine("  " + mkb);
            } else if (mkb instanceof MultiKeyBinding &&
                logActionName.equals(((MultiKeyBinding) mkb).getActionName ()))
            {
                LOG.fine("  " + mkb);
            }
        }
    }

    public Object createInstanceForLookup() {
        List<MultiKeyBinding> keyB = getKeyBindings();
        return new Immutable(new ArrayList<MultiKeyBinding>(keyB));
    }

    
    private static class Listener extends WeakReference<KeyBindingSettingsImpl> 
    implements PropertyChangeListener, Runnable {
        private KeyBindingSettingsFactory      baseKBS;
        
        Listener (
            KeyBindingSettingsImpl      kb,
            KeyBindingSettingsFactory          baseKBS
        ) {
            super(kb, Utilities.activeReferenceQueue());
            this.baseKBS = baseKBS;
            addListeners ();
        }
        
        private KeyBindingSettingsImpl getSettings () {
            KeyBindingSettingsImpl r = get ();
            if (r != null) return r;
            removeListeners ();
            return null;
        }
        
        private void addListeners () {
            EditorSettingsImpl.getInstance().addPropertyChangeListener(
                EditorSettings.PROP_CURRENT_KEY_MAP_PROFILE,
                this
            );
            if (baseKBS != null)
                baseKBS.addPropertyChangeListener (this);
        }
        
        private void removeListeners () {
            if (baseKBS != null)
                baseKBS.removePropertyChangeListener (this);
            EditorSettingsImpl.getInstance().removePropertyChangeListener(
                EditorSettings.PROP_CURRENT_KEY_MAP_PROFILE,
                this
            );
        }
        
        public void propertyChange (PropertyChangeEvent evt) {
            KeyBindingSettingsImpl r = getSettings ();
            if (r == null) return;
            r.log ("refresh2", Collections.EMPTY_SET);
            r.pcs.firePropertyChange (null, null, null);
        }

        public void run() {
            removeListeners();
        }
    }
    
    /* package */ static final class Immutable extends KeyBindingSettings {
        private List<MultiKeyBinding> keyBindings;
        
        public Immutable(List<MultiKeyBinding> keyBindings) {
            this.keyBindings = keyBindings;
        }
        
        public List<MultiKeyBinding> getKeyBindings() {
            return Collections.unmodifiableList(keyBindings);
        }
    }
    
}
