/*
 * DO NOT ALTER OR REMOVE COPYRIGHT NOTICES OR THIS HEADER.
 *
 * Copyright 1997-2007 Sun Microsystems, Inc. All rights reserved.
 *
 * The contents of this file are subject to the terms of either the GNU
 * General Public License Version 2 only ("GPL") or the Common
 * Development and Distribution License("CDDL") (collectively, the
 * "License"). You may not use this file except in compliance with the
 * License. You can obtain a copy of the License at
 * http://www.netbeans.org/cddl-gplv2.html
 * or nbbuild/licenses/CDDL-GPL-2-CP. See the License for the
 * specific language governing permissions and limitations under the
 * License.  When distributing the software, include this License Header
 * Notice in each file and include the License file at
 * nbbuild/licenses/CDDL-GPL-2-CP.  Sun designates this
 * particular file as subject to the "Classpath" exception as provided
 * by Sun in the GPL Version 2 section of the License file that
 * accompanied this code. If applicable, add the following below the
 * License Header, with the fields enclosed by brackets [] replaced by
 * your own identifying information:
 * "Portions Copyrighted [year] [name of copyright owner]"
 *
 * Contributor(s):
 *
 * Portions Copyrighted 2007 Sun Microsystems, Inc.
 */
package org.netbeans.modules.db.explorer.dlg;

import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.Vector;
import org.netbeans.api.db.explorer.DatabaseException;
import org.netbeans.lib.ddl.impl.AddColumn;
import org.netbeans.lib.ddl.impl.CreateIndex;
import org.netbeans.lib.ddl.impl.DriverSpecification;
import org.netbeans.lib.ddl.impl.DropIndex;
import org.netbeans.lib.ddl.impl.Specification;
import org.netbeans.lib.ddl.impl.TableColumn;
import org.netbeans.lib.ddl.util.CommandBuffer;

/**
 * This class factors out the logic of actually adding a column to
 * the database.  It is responsible for interacting with the DDL package.
 * 
 * This refactoring is done to both separate the view from the underlying
 * logic, and also to make it more possible to unit test this logic
 */
public class AddTableColumnDDL {
    private Specification       spec;
    private DriverSpecification drvSpec;
    private String              schema;
    private String              tablename;
    private Map                 indexMap;
    private Map                 uniqueIndexMap;
    

    public AddTableColumnDDL(
            Specification spec, 
            DriverSpecification drvspec,
            String schema,
            String tablename) {
        this.spec       = spec;
        this.drvSpec    = drvspec;
        this.schema     = schema;
        this.tablename  = tablename;
    }
        
    public Map getIndexMap() throws DatabaseException {
        if ( indexMap == null ) {
            buildIndexMaps();
        }
        
        return indexMap;
    }
    
    public Map getUniqueIndexMap() throws DatabaseException {
        if ( uniqueIndexMap == null ) {
            buildIndexMaps();            
        }
        
        return uniqueIndexMap;
    }
    
    private void buildIndexMaps() throws DatabaseException {
        try {
            drvSpec.getIndexInfo(tablename, false, true);
            ResultSet rs = drvSpec.getResultSet();
            HashMap rset = new HashMap();

            indexMap = new HashMap();
            uniqueIndexMap = new HashMap();
            String ixname;
            while (rs.next()) {
                rset = drvSpec.getRow();
                ixname = (String) rset.get(new Integer(6));
                if (ixname != null) {
                    Vector ixcols = (Vector)indexMap.get(ixname);
                    if (ixcols == null) {
                        ixcols = new Vector();
                        indexMap.put(ixname,ixcols);
                        boolean uq = !Boolean.valueOf( 
                                (String)rset.get( new Integer(4) ) ).booleanValue();
                        if(uq)
                            uniqueIndexMap.put( ixname, ColumnItem.UNIQUE );
                    }

                    ixcols.add((String) rset.get(new Integer(9)));
                }
                rset.clear();
            }
            rs.close();
        } catch (SQLException sqle) {
            DatabaseException dbe = new DatabaseException(sqle.getMessage());
            dbe.initCause(sqle);
            throw dbe;
        }
    }
    
    private boolean useIndex(ColumnItem citem) {
        return citem.isIndexed() && !citem.isUnique() && !citem.isPrimaryKey(); 
    }
        
    public boolean execute(String colname, ColumnItem citem, 
            String indexName) throws Exception {
        assert citem != null;
        assert colname != null;
        assert (useIndex(citem) && indexName != null) ||
                !useIndex(citem);
        
        CommandBuffer cbuff = new CommandBuffer();

        AddColumn cmd = spec.createCommandAddColumn(tablename);
        cmd.setObjectOwner(schema);
        org.netbeans.lib.ddl.impl.TableColumn col = null;
        if (citem.isPrimaryKey()) {
          col = cmd.createPrimaryKeyColumn(colname);
        } else if (citem.isUnique()) {
          col = cmd.createUniqueColumn(colname);
        } else col = (TableColumn)cmd.createColumn(colname);
        col.setColumnType(Specification.getType(citem.getType().getType()));
        col.setColumnSize(citem.getSize());
        col.setDecimalSize(citem.getScale());
        col.setNullAllowed(citem.allowsNull());
        if (citem.hasDefaultValue()) col.setDefaultValue(citem.getDefaultValue());

        if (citem.hasCheckConstraint()) {
            // add COLUMN constraint (without constraint name)
            col.setCheckCondition(citem.getCheckConstraint());
        }

        cbuff.add(cmd);

        if (useIndex(citem) ) {
            assert indexName != null;
            addIndex(cbuff, citem, indexName);
        }

        cbuff.execute();

        return cbuff.wasException();
    }
    
    private void addIndex(CommandBuffer cbuff, ColumnItem citem, 
            String indexName) throws Exception {
          buildIndexMaps();

          String isUQ = new String();
          if (indexMap.containsKey(indexName)) {
              if(uniqueIndexMap.containsKey(indexName))
                                isUQ = ColumnItem.UNIQUE;
              DropIndex dropIndexCmd = spec.createCommandDropIndex(indexName);
              dropIndexCmd.setTableName(tablename);
              dropIndexCmd.setObjectOwner(schema);
              cbuff.add(dropIndexCmd);
          }

          CreateIndex xcmd = spec.createCommandCreateIndex(tablename);
          xcmd.setIndexName(indexName);
          xcmd.setIndexType(isUQ);
          xcmd.setObjectOwner(schema);
          Enumeration enu = ((Vector)indexMap.get(indexName)).elements();
          while (enu.hasMoreElements()) {
              xcmd.specifyColumn((String)enu.nextElement());
          }
          xcmd.specifyNewColumn(citem.getName());
          cbuff.add(xcmd);

    }

}
