/*
Copyright (c) 2004-2005, Dennis M. Sosnoski
All rights reserved.

Redistribution and use in source and binary forms, with or without modification,
are permitted provided that the following conditions are met:

 * Redistributions of source code must retain the above copyright notice, this
   list of conditions and the following disclaimer.
 * Redistributions in binary form must reproduce the above copyright notice,
   this list of conditions and the following disclaimer in the documentation
   and/or other materials provided with the distribution.
 * Neither the name of JiBX nor the names of its contributors may be used
   to endorse or promote products derived from this software without specific
   prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS" AND
ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE FOR
ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
(INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

package org.jibx.binding.model;

import java.util.ArrayList;

import org.jibx.binding.util.StringArray;
import org.jibx.runtime.IUnmarshallingContext;
import org.jibx.runtime.JiBXException;

/**
 * Model component for <b>structure</b> element of binding definition.
 *
 * @author Dennis M. Sosnoski
 * @version 1.0
 */
 
public class StructureElement extends StructureElementBase
{
    /** Enumeration of allowed attribute names */
    public static final StringArray s_allowedAttributes =
        new StringArray(new String[] { "map-as" },
        StructureElementBase.s_allowedAttributes);
    
    /** Mapped type name to use for this object. */
    private String m_mapAsName;
    
    /** Flag for structure has a concrete mapping, possibly indeterminant. */
    private boolean m_hasMappingName;
    
    /** Binding to use for this object. */
    private TemplateElementBase m_mapAsMapping;
    
    /**
	 * Default constructor.
	 */
	public StructureElement() {
        super(STRUCTURE_ELEMENT);
    }
    
    /**
     * Get name of specified type.
     * 
     * @return specified type name (or <code>null</code> if none)
     */
    public String getMapAsName() {
        return m_mapAsName;
    }
    
    /**
     * Set name of specified type.
     * 
     * @param name specified type name (or <code>null</code> if none)
     */
    public void setMapAsName(String name) {
        m_mapAsName = name;
    }
    
    /**
     * Get specified type mapping. This call is only meaningful after
     * validation.
     * 
     * @return specified type mapping (or <code>null</code> if none)
     */
    public TemplateElementBase getMapAsMapping() {
        return m_mapAsMapping;
    }
    
    //
    // Overrides of base class methods
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#hasName()
     */
    public boolean hasName() {
        if (m_mapAsMapping instanceof MappingElement) {
            if (((MappingElement)m_mapAsMapping).getName() != null) {
                return true;
            }
        } else if (m_hasMappingName) {
            return true;
        }
        return super.hasName();
    }
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#getName()
     */
    public String getName() {
        if (m_mapAsMapping instanceof MappingElement) {
            String name = ((MappingElement)m_mapAsMapping).getName();
            if (name != null) {
                return name;
            }
        } else if (m_hasMappingName) {
            return "#" + getType().getName();
        }
        return super.getName();
    }
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#getUri()
     */
    public String getUri() {
        if (m_mapAsMapping instanceof MappingElement) {
            String uri = ((MappingElement)m_mapAsMapping).getUri();
            if (uri != null) {
                return uri;
            }
        }
        return super.getUri();
    }
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#hasAttribute()
     */
    public boolean hasAttribute() {
        if (hasName()) {
            return false;
        } else {
            return super.hasAttribute();
        }
    }
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#hasContent()
     */
    public boolean hasContent() {
        if (hasName()) {
            return true;
        } else {
            return super.hasContent();
        }
    }
    
    /* (non-Javadoc)
     * @see org.jibx.binding.model.IComponent#getType()
     */
    public IClass getType() {
        if (m_mapAsMapping == null) {
            return super.getType();
        } else {
            return m_mapAsMapping.getHandledClass();
        }
    }
        
    //
    // Validation methods
    
    /**
     * Make sure all attributes are defined.
     *
     * @param uctx unmarshalling context
     * @exception JiBXException on unmarshalling error
     */
    private void preSet(IUnmarshallingContext uctx) throws JiBXException {
        validateAttributes(uctx, s_allowedAttributes);
    }

    /**
     * Check that child components are of types compatible with the structure
     * type. This method may call itself recursively to process the children of
     * child components which do not themselves set a type.
     * 
     * @param vctx validation context
     * @param type structure object type
     * @param children list of child components to be checked
     */
    private void checkCompatibleChildren(ValidationContext vctx, IClass type,
        ArrayList children) {
        for (int i = 0; i < children.size(); i++) {
            ElementBase child = (ElementBase)children.get(i);
            boolean expand = true;
            if (child instanceof IComponent) {
                IComponent comp = (IComponent)child;
                IClass ctype = comp.getType();
                if (comp instanceof ContainerElementBase) {
                    ContainerElementBase contain = (ContainerElementBase)comp;
                    expand = !contain.hasObject();
                }
                if (comp.isImplicit()) {
                    if (!type.isAssignable(ctype)) {
                        vctx.addFatal
                            ("References to structure object must have " +
                            "compatible types: " + type.getName() +
                            " cannot be used as " + ctype.getName(), child);
                    }
                }
            }
            if (expand && child instanceof NestingElementBase) {
                checkCompatibleChildren(vctx, type,
                    ((NestingElementBase)child).children());
            }
        }
    }

    /* (non-Javadoc)
     * @see org.jibx.binding.model.ElementBase#validate(org.jibx.binding.model.ValidationContext)
     */
    public void validate(ValidationContext vctx) {
        
        // check if there's a mapping if used without children
        DefinitionContext dctx = vctx.getDefinitions();
        if (children().size() == 0) {
            if (m_mapAsName == null) {
                
                // see if this is using implicit marshaller/unmarshaller
                if ((vctx.isInBinding() && getUnmarshallerName() == null) ||
                    (vctx.isOutBinding() && getMarshallerName() == null)) {
                    if (getUsing() == null) {
                        IClass type = getType();
                        String tname = type.getName();
                        if (!"java.lang.Object".equals(tname)) {
                            m_mapAsMapping = dctx.getNamedTemplate(tname);
                            m_hasMappingName = true;
                            if (m_mapAsMapping == null) {
                                if (dctx.isCompatibleTemplateType(type)) {
                                } else {
                                    vctx.addFatal("No compatible mapping " +
                                        "defined for type " + tname);
                                }
                            }
                        }
                    }
                }
                
            } else {
                
                // find mapping by type name or class name
                TemplateElementBase base = dctx.getNamedTemplate(m_mapAsName);
                if (base == null) {
                    base = dctx.getSpecificTemplate(m_mapAsName);
                    if (base == null) {
                        vctx.addFatal("No mapping with type name " +
                            m_mapAsName);
                    }
                }
                if (base != null) {
                    
                    // make sure type is compatible
                    IClass type = getType();
                    if (!type.isAssignable(base.getHandledClass()) &&
                        !base.getHandledClass().isAssignable(type)) {
                        vctx.addError("Object type " + type.getName() +
                            " is incompatible with binding for class " +
                            base.getClassName());
                    }
                    m_mapAsMapping = base;
                    
                    // set flag for mapping with name
                    m_hasMappingName = base instanceof MappingElement &&
                        !((MappingElement)base).isAbstract();
                }
                
            }
        } else if (m_mapAsName != null) {
            vctx.addError("map-as attribute cannot be used with children");
        }
        IClass type = getType();
        if (type != null) {
            
            // check each child component for compatible type
            ArrayList children = children();
            if (hasProperty()) {
                checkCompatibleChildren(vctx, type, children);
            }
            
            // check for only set-method supplied
            if (!vctx.isOutBinding() && getField() == null &&
                getGet() == null && getSet() != null) {
                
                // no way to handle both elements and attributes
                if (hasAttribute() && hasContent()) {
                    vctx.addError("Need way to load existing object instance " +
                        "to support combined attribute and element values");
                } else {
                    vctx.addWarning("No way to load prior value - " +
                        "new instance will be created on each unmarshalling");
                }
            }
        }
        super.validate(vctx);
    }
}