/* ===================================================
 * JCommon : a free general purpose Java class library
 * ===================================================
 *
 * Project Info:  http://www.jfree.org/jcommon/index.html
 * Project Lead:  David Gilbert (david.gilbert@object-refinery.com);
 *
 * (C) Copyright 2000-2003, by Object Refinery Limited and Contributors.
 *
 * This library is free software; you can redistribute it and/or modify it under the terms
 * of the GNU Lesser General Public License as published by the Free Software Foundation;
 * either version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License along with this
 * library; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * --------------------------
 * SystemPropertiesFrame.java
 * --------------------------
 * (C) Copyright 2000-2003, by Object Refinery Limited.
 *
 * Original Author:  David Gilbert (for Object Refinery Limited);
 * Contributor(s):   -;
 *
 * $Id: SystemPropertiesFrame.java,v 1.4 2003/06/12 16:54:46 mungady Exp $
 *
 * Changes (from 26-Oct-2001)
 * --------------------------
 * 26-Oct-2001 : Changed package to com.jrefinery.ui (DG);
 * 26-Nov-2001 : Made a separate class SystemPropertiesPanel.java (DG);
 * 28-Feb-2002 : Moved to package com.jrefinery.ui.about (DG);
 * 15-Mar-2002 : Modified to use a ResourceBundle for elements that require localisation (DG);
 * 08-Oct-2002 : Fixed errors reported by Checkstyle (DG);
 *
 */

package org.jfree.ui.about;

import java.awt.BorderLayout;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.util.ResourceBundle;

import javax.swing.BorderFactory;
import javax.swing.JButton;
import javax.swing.JFrame;
import javax.swing.JMenu;
import javax.swing.JMenuBar;
import javax.swing.JMenuItem;
import javax.swing.JPanel;

/**
 * A frame containing a table that displays the system properties for the current Java Virtual
 * Machine (JVM).  It is useful to incorporate this frame into an application for diagnostic
 * purposes, since it provides a convenient means for the user to return configuration and
 * version information when reporting problems.
 *
 * @author David Gilbert
 */
public class SystemPropertiesFrame extends JFrame implements ActionListener {

    /** Copy action command. */
    private static final String COPY_COMMAND = "COPY";

    /** Close action command. */
    private static final String CLOSE_COMMAND = "CLOSE";

    /** A system properties panel. */
    private SystemPropertiesPanel panel;

    /**
     * Constructs a standard frame that displays system properties.
     * <P>
     * If a menu is requested, it provides a menu item that allows the user to copy the contents of
     * the table to the clipboard in tab-delimited format.
     *
     * @param menu  flag indicating whether or not the frame should display a menu to allow
     *              the user to copy properties to the clipboard.
     */
    public SystemPropertiesFrame(boolean menu) {

        String baseName = "org.jfree.ui.about.resources.AboutResources";
        ResourceBundle resources = ResourceBundle.getBundle(baseName);

        String title = resources.getString("system-frame.title");
        setTitle(title);

        setDefaultCloseOperation(JFrame.DISPOSE_ON_CLOSE);

        if (menu) {
            setJMenuBar(createMenuBar(resources));
        }

        JPanel content = new JPanel(new BorderLayout());
        panel = new SystemPropertiesPanel();
        panel.setBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5));

        content.add(panel, BorderLayout.CENTER);

        JPanel buttonPanel = new JPanel(new BorderLayout());
        buttonPanel.setBorder(BorderFactory.createEmptyBorder(5, 0, 0, 0));

        String label = resources.getString("system-frame.button.close");
        Character mnemonic = (Character) resources.getObject("system-frame.button.close.mnemonic");
        JButton closeButton = new JButton(label);
        closeButton.setMnemonic(mnemonic.charValue());

        closeButton.setActionCommand(CLOSE_COMMAND);
        closeButton.addActionListener(this);

        buttonPanel.add(closeButton, BorderLayout.EAST);
        content.add(buttonPanel, BorderLayout.SOUTH);

        setContentPane(content);

    }

    /**
     * Handles action events generated by the user.
     *
     * @param e  the event.
     */
    public void actionPerformed(ActionEvent e) {

        String command = e.getActionCommand();
        if (command.equals(CLOSE_COMMAND)) {
            dispose();
        }
        else if (command.equals(COPY_COMMAND)) {
            panel.copySystemPropertiesToClipboard();
        }

    }


    /**
     * Creates and returns a menu-bar for the frame.
     *
     * @param resources  localised resources.
     *
     * @return a menu bar.
     */
    private JMenuBar createMenuBar(ResourceBundle resources) {

        JMenuBar menuBar = new JMenuBar();

        String label = resources.getString("system-frame.menu.file");
        Character mnemonic = (Character) resources.getObject("system-frame.menu.file.mnemonic");
        JMenu fileMenu = new JMenu(label, true);
        fileMenu.setMnemonic(mnemonic.charValue());

        label = resources.getString("system-frame.menu.file.close");
        mnemonic = (Character) resources.getObject("system-frame.menu.file.close.mnemonic");
        JMenuItem closeItem = new JMenuItem(label, mnemonic.charValue());
        closeItem.setActionCommand(CLOSE_COMMAND);

        closeItem.addActionListener(this);
        fileMenu.add(closeItem);

        label = resources.getString("system-frame.menu.edit");
        mnemonic = (Character) resources.getObject("system-frame.menu.edit.mnemonic");
        JMenu editMenu = new JMenu(label);
        editMenu.setMnemonic(mnemonic.charValue());

        label = resources.getString("system-frame.menu.edit.copy");
        mnemonic = (Character) resources.getObject("system-frame.menu.edit.copy.mnemonic");
        JMenuItem copyItem = new JMenuItem(label, mnemonic.charValue());
        copyItem.setActionCommand(COPY_COMMAND);
        copyItem.addActionListener(this);
        editMenu.add(copyItem);

        menuBar.add(fileMenu);
        menuBar.add(editMenu);
        return menuBar;

    }

}
