package com.clarkware.junitperf;

import junit.framework.Test;
import junit.framework.TestResult;

/**
 * The <code>ThreadedTest</code> is a test decorator that
 * runs a test in a separate thread.
 *
 * @author <a href="mailto:mike@clarkware.com">Mike Clark</a>
 * @author <a href="http://www.clarkware.com">Clarkware Consulting, Inc.</a>
 *
 * @see junit.framework.Test
 */

public class ThreadedTest implements Test {

	private final Test _test;
	private final ThreadGroup _group;
	private final ThreadBarrier _barrier;
	
	/**
	 * Constructs a <code>ThreadedTest</code> to decorate the
	 * specified test using the same thread group as the
	 * current thread.
	 *
	 * @param test Test to decorate.
	 */
	public ThreadedTest(Test test) {
		this(test, null, new ThreadBarrier(1));
	}

	/**
	 * Constructs a <code>ThreadedTest</code> to decorate the
	 * specified test using the specified thread group and
	 * thread barrier.
	 *
	 * @param test Test to decorate.
	 * @param group Thread group.
	 * @param barrier Thread barrier.
	 */
	public ThreadedTest(Test test, ThreadGroup group, ThreadBarrier barrier) {
		_test = test;
		_group = group;
		_barrier = barrier;
	}

	/**
	 * Returns the number of test cases in this test.
	 *
	 * @return Number of test cases.
	 */
	public int countTestCases() {
		return _test.countTestCases();
	}

	/**
	 * Runs this test.
	 *
	 * @param result Test result.
	 */
	public void run(TestResult result) {
		Thread t = new Thread(_group, new TestRunner(result));
		t.start();
	}
	
	
	class TestRunner implements Runnable  {
		
		private TestResult _result;
		
		public TestRunner(TestResult result)  {
			_result = result;
		}
		
		public void run() {
			_test.run(_result);
			_barrier.onCompletion(Thread.currentThread());
		}
	}

	/**
	 * Returns the test description.
	 *
	 * @return Description.
	 */
	public String toString() {
		return "ThreadedTest: " + _test.toString();
	}
}
