package com.clarkware.junitperf;

/**
 * The <code>ThreadBarrier</code> class provides a callback
 * method for threads to signal their completion. 
 *
 * @author <a href="mailto:mike@clarkware.com">Mike Clark</a>
 * @author <a href="http://www.clarkware.com">Clarkware Consulting, Inc.</a>
 */

public class ThreadBarrier {

	public int _returnedCount;
	public final int _dispatchedCount;

	/**
	 * Constructs a <code>ThreadBarrier</code> with the
	 * specified number of threads to wait for.
	 *
	 * @param numDispatched Number of threads dispatched.
	 */
	public ThreadBarrier(int numDispatched) {
		_returnedCount = 0;
		_dispatchedCount = numDispatched;
	}

	/**
	 * Called when the specified thread is complete.
	 *
	 * @param t Completed thread.
	 */
	public synchronized void onCompletion(Thread t) {
		_returnedCount++;
	}

	/**
	 * Determines whether the thread barrier has been reached -
	 * when all dispatched threads have returned.
	 *
	 * @return <code>true</code> if the barrier has been reached;
	 *         <code>false</code> otherwise.
	 */
	public boolean isReached() {
		return (_returnedCount >= _dispatchedCount);
	}
	
	/**
	 * Cancels the specified number of threads.
	 *
	 * @param threadCount Number of threads to cancel.
	 */
	public synchronized void cancelThreads(int threadCount) {
		_returnedCount += threadCount;
	}
}
