/*  GtkTiEmu - a TI emulator
 *  GtkSpecific.c: Linux display specific part
 *  Originally written by Jonas Minsberg
 *  Copyright (C) 2000, Thomas Corvazier, Romain Lievin
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details. *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* 
   This source come from the xspecific.c file of (x)tiger. I modified
   and adapted so that it can work with GTK. Moreover, I corrected some
   bugs with the management of the display in 24 colors mode.
*/

/* SDL */
#include <SDL/SDL.h>
#include <signal.h>

/* GTK */
#include <gtk/gtk.h>
#include "gtk_tiemu_cb.h"
#include "gtk_refresh.h"
#include "support.h"
#include "img_fmt.h"

/* Program dependencies */
#include "platform.h"
#include "specific.h"
#include "skn_loader.h"
#include "struct.h"


#define DEFAULT_BPP   8
#define DEFAULT_FLAGS (SDL_HWPALETTE | SDL_HWSURFACE) // RESIZABLE requires SDL 1.1.2 mini

/*
#define DEFAULT_BPP   16
#define DEFAULT_FLAGS 0
*/

/*****************************************/
/* Various variables and short functions */
/*****************************************/

char *sTitle = "GtkTiEmu";
unsigned long White, Black;
int iBpp, iDepth; // bytes per pixel
int iScrLineSize;
SDL_Surface *pScreen;
int iXWin, iYWin, iXLcd, iYLcd;
int iXMax = 240, iYMax = 128; // TI89 screen is a clipped TI92 screen
int iLineLength;
UBYTE *pLcd;
ULONG *pScreenBuf = NULL; // TI screen (bytemapped)
UWORD *pConvTable = 0;
ULONG convtab[512]; /* Planar->Chunky conversion table */
UBYTE sdl2ti[512];
volatile int iKeyWasPressed;
int bSdlOkay = 0;
int lastKey = -1;
int suspended = 0;
GtkWidget *window;
int bFullscreen=0;
int bFullscreenFailed=0;
int bWindowedFailed=0;
int iLcdForeground;
int iLcdBackground;
int bGtkOkay = 0;
int iContrast = 16;
int iLastContrast = 0;
int iNewContrast = 16;
int iGrayPlanes = -1;
int iCurrPlane = 0;
int iEmuState = 0;
int iAlpha = 0;

unsigned long lcdPal[16];  // contains the greyscale palette
unsigned long winPal[256]; // contains skin palette
SDL_Color sdlPal[256];    // global palette: greyscale + skin

void switch_without_skin();
void switch_with_skin();
void switch_fullscreen();
void switch_windowed();
void compute_convtable(void);
int sdl_to_ti(int key);
int gtk_main_window(void);
void set_colors(void);
UWORD rgb24_to_rgb16(UBYTE r, UBYTE g, UBYTE b) 
{
  return (((r>>3)<<11) | ((g>>2)<<5) | (b>>3));
}


/*************/
/* Init/Exit */
/*************/

static int init_specific(void) 
{
  int i, j;

  gtk_main_window();
  
  /* Init SDL */
  DISPLAY("Initializing SDL... ");
  if(SDL_Init(SDL_INIT_VIDEO) < 0) 
    {
      fprintf(stderr,"Couldn't initialize SDL: %s\n", SDL_GetError());
      return -1;
    }
  //atexit(SDL_Quit);
  //atexit(gtk_main_quit);
  DISPLAY("Done.\n");

  SDL_WM_SetCaption(sTitle, "gtktiemu.xpm");

  /* Load skin */
  if (ti68k_getCalcType() & TI92) 
    {
      iXLcd = 240; iYLcd = 128; iLineLength = 240;
      if (!loadSkin("ti92.skn")) return -1;
    }
  else 
    {
      iXLcd = 160; iYLcd = 100; iLineLength = 240;
      if (!loadSkin("ti89.skn")) return -1;
    }
  
  if(glob_inf.background) 
    {
      iYWin = ySkin;
      iXWin = xSkin;
    }
  else 
    {
      iYWin = iYLcd;
      iXWin = iXLcd;
    }

  /* Init the planar/chunky conversion table */  
  compute_convtable();

  /* Allocate the TI screen buffer */
  pScreenBuf = malloc(iXMax * iYMax);

  /* Init the SDL key -> TI key conversion table */
  for(i=0; i<512; i++)
    sdl2ti[i] = 0;
  for(i=0; i<256; i++)
    sdl2ti[i] = sdl_to_ti(i);
  for(i=0; i<256; i++)
    sdl2ti[i+256] = sdl_to_ti(i+65288);

  /* Set VIDEO mode */
  if (!(pScreen = SDL_SetVideoMode(iXWin, iYWin, DEFAULT_BPP, DEFAULT_FLAGS)))
    {
      DISPLAY("Could not set video mode: %s\n", SDL_GetError());
      return -1;
    }
  iDepth = pScreen->format->BitsPerPixel;
  iBpp = pScreen->format->BytesPerPixel;
  iScrLineSize = (pScreen->pitch) >> (iBpp >> 1);
  DISPLAY("SDL: iDepth = %i, iBpp = %i\n", iDepth, iBpp);
  bSdlOkay = 1;

  White = SDL_MapRGB(pScreen->format, 255, 255, 255);
  Black = SDL_MapRGB(pScreen->format, 0, 0, 0);

  /* Compute greyscale colormap */
   for(j=0; j<16; j++)
    {
      lcdPal[j] = 0;
    }
  set_colors();  
  
  redraw_skin();

  return 0;
}

static int exit_specific(void) 
{
  bSdlOkay=0;
  SDL_FreeSurface(pScreen); pScreen = 0;
  SDL_Quit();

  return 0;
}


/***********************/
/* Keyboard management */
/***********************/

/* Converts a keyboard key (an SDL event) into a TI key */
int sdl_to_ti(int key) 
{
  if(ti68k_getCalcType() & TI92)
    {
      switch(key) 
	{
	case SDLK_a : return TIKEY_A;
	case SDLK_b : return TIKEY_B;
	case SDLK_c : return TIKEY_C;
	case SDLK_d : return TIKEY_D;
	case SDLK_e : return TIKEY_E;
	case SDLK_f : return TIKEY_F;
	case SDLK_g : return TIKEY_G;
	case SDLK_h : return TIKEY_H;
	case SDLK_i : return TIKEY_I;
	case SDLK_j : return TIKEY_J;
	case SDLK_k : return TIKEY_K;
	case SDLK_l : return TIKEY_L;
	case SDLK_m : return TIKEY_M;
	case SDLK_n : return TIKEY_N;
	case SDLK_o : return TIKEY_O;
	case SDLK_p : return TIKEY_P;
	case SDLK_q : return TIKEY_Q;
	case SDLK_r : return TIKEY_R;
	case SDLK_s : return TIKEY_S;
	case SDLK_t : return TIKEY_T;
	case SDLK_u : return TIKEY_U;
	case SDLK_v : return TIKEY_V;
	case SDLK_w : return TIKEY_W;
	case SDLK_x : return TIKEY_X;
	case SDLK_y : return TIKEY_Y;
	case SDLK_z : return TIKEY_Z;
	case SDLK_KP0 : return TIKEY_0;
	case SDLK_KP1 : return TIKEY_1;
	case SDLK_KP2 : return TIKEY_2;
	case SDLK_KP3 : return TIKEY_3;
	case SDLK_KP4 : return TIKEY_4;
	case SDLK_KP5 : return TIKEY_5;
	case SDLK_KP6 : return TIKEY_6;
	case SDLK_KP7 : return TIKEY_7;
	case SDLK_KP8 : return TIKEY_8;
	case SDLK_KP9 : return TIKEY_9;
	case SDLK_0 : return TIKEY_0;
	case SDLK_1 : return TIKEY_1;
	case SDLK_2 : return TIKEY_2;
	case SDLK_3 : return TIKEY_3;
	case SDLK_4 : return TIKEY_4;
	case SDLK_5 : return TIKEY_5;
	case SDLK_6 : return TIKEY_6;
	case SDLK_7 : return TIKEY_7;
	case SDLK_8 : return TIKEY_8;
	case SDLK_9 : return TIKEY_9;
	case SDLK_UP : return TIKEY_UP;
	case SDLK_LEFT : return TIKEY_LEFT;
	case SDLK_RIGHT : return TIKEY_RIGHT;
	case SDLK_DOWN : return TIKEY_DOWN;
	case SDLK_F1 : return TIKEY_F1;
	case SDLK_F2 : return TIKEY_F2;
	case SDLK_F3 : return TIKEY_F3;
	case SDLK_F4 : return TIKEY_F4;
	case SDLK_F5 : return TIKEY_F5;
	case SDLK_F6 : return TIKEY_F6;
	case SDLK_F7 : return TIKEY_F7;
	case SDLK_F8 : return TIKEY_F8;
	case SDLK_RETURN : return TIKEY_ENTER1;
	case SDLK_KP_ENTER : return TIKEY_ENTER2;
	case SDLK_LSHIFT : return TIKEY_SHIFT;
	case SDLK_RSHIFT : return TIKEY_SHIFT;
	case SDLK_RCTRL : return TIKEY_DIAMOND;
	case SDLK_LCTRL : return TIKEY_DIAMOND;
	case SDLK_LALT : return TIKEY_2ND;
	case SDLK_RALT : return TIKEY_2ND;
	case SDLK_CAPSLOCK : return TIKEY_HAND;
	case SDLK_TAB : return TIKEY_STORE;
	case SDLK_SPACE : return TIKEY_SPACE;
	case SDLK_ESCAPE : return TIKEY_ESCAPE;
	case SDLK_BACKSPACE : return TIKEY_BACKSPACE;
	case SDLK_LEFTPAREN : return TIKEY_PALEFT;
	case SDLK_RIGHTPAREN : return TIKEY_PARIGHT;
	case SDLK_PERIOD : return TIKEY_PERIOD;
	case SDLK_COMMA : return TIKEY_COMMA;
	case SDLK_KP_PLUS : return TIKEY_PLUS;
	case SDLK_KP_MULTIPLY : return TIKEY_MULTIPLY;
	case SDLK_KP_DIVIDE : return TIKEY_DIVIDE;    
	case SDLK_KP_MINUS : return TIKEY_MINUS;
	case SDLK_MINUS : return TIKEY_NEGATE;
	case SDLK_BACKSLASH : return TIKEY_ON;
	  //case SDLK_quoteright : return TIKEY_MULTIPLY;
	case SDLK_SLASH : return TIKEY_DIVIDE;
	case SDLK_SEMICOLON : return TIKEY_THETA;
	case SDLK_EQUALS : return TIKEY_EQUALS;
	  //case SDLK_quoteleft : return TIKEY_POWER;
	case SDLK_LESS : return TIKEY_NEGATE;
	  //case SDLK_KP_Decimal : return TIKEY_PERIOD;
	case SDLK_INSERT : return TIKEY_SIN;
	case SDLK_DELETE : return TIKEY_CLEAR;
	case SDLK_HOME : return TIKEY_COS;
	case SDLK_END : return TIKEY_LN;
	case SDLK_PAGEUP : return TIKEY_TAN;
	case SDLK_PAGEDOWN : return TIKEY_MODE;
	case SDLK_SCROLLOCK : return TIKEY_ON;
	case SDLK_F9 : return OPT_DEBUGGER;
	case SDLK_F10 : return OPT_SCREENCAPTURE;
	  // F11 & F12 are used by WindowMaker
	case SDLK_PRINT : return TIKEY_APPS;
	  /*  
	      case SDLK_BREAK_ALTERNATIVE :
	      case SDLK_BREAK : return OPT_QUITNOSAVE;
	  */
	default : return TIKEY_NU;
	}
    }
  else
    {
      iAlpha = 0;
      switch(key) 
	{
	case SDLK_a : iAlpha = 1; return TIKEY_EQUALS;
	case SDLK_b : iAlpha = 1; return TIKEY_PALEFT;
	case SDLK_c : iAlpha = 1; return TIKEY_PARIGHT;
	case SDLK_d : iAlpha = 1; return TIKEY_COMMA;
	case SDLK_e : iAlpha = 1; return TIKEY_DIVIDE;
	case SDLK_f : iAlpha = 1; return TIKEY_F;
	case SDLK_g : iAlpha = 1; return TIKEY_7;
	case SDLK_h : iAlpha = 1; return TIKEY_8;
	case SDLK_i : iAlpha = 1; return TIKEY_9;
	case SDLK_j : iAlpha = 1; return TIKEY_MULTIPLY;
	case SDLK_k : iAlpha = 1; return TIKEY_EE;
	case SDLK_l : iAlpha = 1; return TIKEY_4;
	case SDLK_m : iAlpha = 1; return TIKEY_5;
	case SDLK_n : iAlpha = 1; return TIKEY_6;
	case SDLK_o : iAlpha = 1; return TIKEY_MINUS;
	case SDLK_p : iAlpha = 1; return TIKEY_STORE;
	case SDLK_q : iAlpha = 1; return TIKEY_1;
	case SDLK_r : iAlpha = 1; return TIKEY_2;
	case SDLK_s : iAlpha = 1; return TIKEY_3;
	case SDLK_t : return TIKEY_T;
	case SDLK_u : iAlpha = 1; return TIKEY_PLUS;
	case SDLK_v : iAlpha = 1; return TIKEY_0;
	case SDLK_w : iAlpha = 1; return TIKEY_PERIOD;
	case SDLK_x : return TIKEY_X;
	case SDLK_y : return TIKEY_Y;
	case SDLK_z : return TIKEY_Z;
	case SDLK_KP0 : return TIKEY_0;
	case SDLK_KP1 : return TIKEY_1;
	case SDLK_KP2 : return TIKEY_2;
	case SDLK_KP3 : return TIKEY_3;
	case SDLK_KP4 : return TIKEY_4;
	case SDLK_KP5 : return TIKEY_5;
	case SDLK_KP6 : return TIKEY_6;
	case SDLK_KP7 : return TIKEY_7;
	case SDLK_KP8 : return TIKEY_8;
	case SDLK_KP9 : return TIKEY_9;
	case SDLK_0 : return TIKEY_0;
	case SDLK_1 : return TIKEY_1;
	case SDLK_2 : return TIKEY_2;
	case SDLK_3 : return TIKEY_3;
	case SDLK_4 : return TIKEY_4;
	case SDLK_5 : return TIKEY_5;
	case SDLK_6 : return TIKEY_6;
	case SDLK_7 : return TIKEY_7;
	case SDLK_8 : return TIKEY_8;
	case SDLK_9 : return TIKEY_9;
	case SDLK_UP : return TIKEY_UP;
	case SDLK_LEFT : return TIKEY_LEFT;
	case SDLK_RIGHT : return TIKEY_RIGHT;
	case SDLK_DOWN : return TIKEY_DOWN;
	case SDLK_F1 : return TIKEY_F1;
	case SDLK_F2 : return TIKEY_F2;
	case SDLK_F3 : return TIKEY_F3;
	case SDLK_F4 : return TIKEY_F4;
	case SDLK_F5 : return TIKEY_F5;
	case SDLK_F6 : return TIKEY_CATALOG;
	case SDLK_F7 : return TIKEY_F7;
	case SDLK_F8 : return TIKEY_F8;
	case SDLK_RETURN : return TIKEY_ENTER1;
	case SDLK_KP_ENTER : return TIKEY_ENTER2;
	case SDLK_LSHIFT : return TIKEY_SHIFT;
	case SDLK_RSHIFT : return TIKEY_SHIFT;
	case SDLK_RCTRL : return TIKEY_DIAMOND;
	case SDLK_LCTRL : return TIKEY_DIAMOND;
	case SDLK_LALT : return TIKEY_2ND;
	case SDLK_RALT : return TIKEY_2ND;
	case SDLK_CAPSLOCK : return TIKEY_ALPHA;
	case SDLK_TAB : return TIKEY_STORE;
	case SDLK_SPACE : return TIKEY_SPACE;
	case SDLK_ESCAPE : return TIKEY_ESCAPE;
	case SDLK_BACKSPACE : return TIKEY_BACKSPACE;
	case SDLK_LEFTPAREN : return TIKEY_PALEFT;
	case SDLK_RIGHTPAREN : return TIKEY_PARIGHT;
	case SDLK_PERIOD : return TIKEY_PERIOD;
	case SDLK_COMMA : return TIKEY_COMMA;
	case SDLK_KP_PLUS : return TIKEY_PLUS;
	case SDLK_KP_MULTIPLY : return TIKEY_MULTIPLY;
	case SDLK_KP_DIVIDE : return TIKEY_DIVIDE;    
	case SDLK_KP_MINUS : return TIKEY_MINUS;
	case SDLK_MINUS : return TIKEY_NEGATE;
	case SDLK_BACKSLASH : return TIKEY_ON;
	  //case SDLK_quoteright : return TIKEY_MULTIPLY;
	case SDLK_SLASH : return TIKEY_DIVIDE;
	  //case SDLK_semicolon : return TIKEY_THETA;
	case SDLK_EQUALS : return TIKEY_EQUALS;
	  //case SDLK_quoteleft : return TIKEY_POWER;
	case SDLK_LESS : return TIKEY_NEGATE;
	  //case SDLK_KP_Decimal : return TIKEY_PERIOD;
	case SDLK_INSERT : return TIKEY_SIN;
	case SDLK_DELETE : return TIKEY_CLEAR;
	case SDLK_HOME : return TIKEY_COS;
	case SDLK_END : return TIKEY_LN;
	case SDLK_PAGEUP : return TIKEY_EE;	
	case SDLK_PAGEDOWN : return TIKEY_POWER;
	case SDLK_SCROLLOCK : return TIKEY_ON;
	case SDLK_F9 : return OPT_DEBUGGER;
	case SDLK_F10 : return OPT_SCREENCAPTURE;
	  // F11 & F12 are used by WindowMaker
	case SDLK_PRINT : return TIKEY_APPS;
	  /*  
	      case SDLK_BREAK_ALTERNATIVE :
	      case SDLK_BREAK : return OPT_QUITNOSAVE;
	  */
	default : return TIKEY_NU;
	}
    }
}


/*static*/ int update_keys(void) 
{
  SDL_Event event;
  int i;

  iKeyWasPressed = 0;
  
  while(SDL_PollEvent(&event)) 
    {
      if(event.type==SDL_MOUSEBUTTONDOWN) 
	{
	  if(event.button.button==1) 
	    {
	      i = pos_to_key(event.button.x,event.button.y);
	      if(i>=0) 
		{
		  if(event.button.button == 1) 
		    {
		      lastKey = i;
		      ti68k_setActiveKey(i,1);
		      iKeyWasPressed = 1;
		    }
		}
	    }
#ifndef EXT_WIN
	  else if(event.button.button == 3)
	    {
	      if(!bFullscreen) 
		{
		  SDL_WaitEvent(&event); // flush event !!!
		  gtk_menu_popup(GTK_MENU(display_popup_menu()),
				 NULL, NULL, NULL, NULL,
				 event.button.button,
				 -897199374); // cannot pass time
		  suspend();
		}
	      else
		switch_windowed();
	    }
#endif
	}
      else if(event.button.button==3) 
	{
	  
	}
      else if(event.type==SDL_MOUSEBUTTONUP) 
	{
	  if(event.button.button==1) 
	    {
	      if(lastKey!=-1) 
		{
		  ti68k_setActiveKey(lastKey,0);
		  lastKey = -1;
		}
	    }
	}
      else if(event.type==SDL_KEYDOWN) 
	{
	  if((event.key.keysym.mod & KMOD_RALT || 
	      event.key.keysym.mod & KMOD_LALT) && 
	     (event.key.keysym.sym==SDLK_RETURN)) 
	    {
	      if (bFullscreen)
		switch_windowed();
	      else
		switch_fullscreen();
	    }
	  else if(event.key.keysym.sym == SDLK_F9)
	    {
	      ti68k_setActiveKey(OPT_DEBUGGER, 0);
	      ti68k_launchDebugger();
	    }
	  else if(event.key.keysym.sym == SDLK_F10)
	    {
	      ti68k_setActiveKey(OPT_SCREENCAPTURE, 0);
	      do_screenshot(options.img_format, options.img_type, options.img_size, 
			    NULL);
	      unsuspend();
	    }
	  else
	    {
	      iKeyWasPressed = 1;
	      if(iAlpha)
		ti68k_setActiveKey(TIKEY_ALPHA, 1);
	      ti68k_setActiveKey(sdl_to_ti(event.key.keysym.sym), 1);
	    }
	}
      else if(event.type==SDL_KEYUP) 
	{
	  if(iAlpha)
	    {
	      ti68k_setActiveKey(TIKEY_ALPHA, 0);
	      iAlpha = 0;
	    }
	  ti68k_setActiveKey(sdl_to_ti(event.key.keysym.sym), 0);
	}
      else if(event.type==SDL_QUIT) 
	{
	  SDL_Quit();
	  exit(0);
	}
    }

  return iKeyWasPressed;
}


/* 
   Converts the mouse position into a TIKEY_xx code
   Checks if the mouse cursor is within a rectangle defined in the keyDefs
   array
*/
int pos_to_key(int x, int y) 
{
  int i;
  
  for(i = 0; i<80 ;i++)
    {
      if ( (x >= rcRectKeys[i].left) && (x < rcRectKeys[i].right) 
	  && (y >= rcRectKeys[i].top) && (y < rcRectKeys[i].bottom) ) 
	return sknKey[i];
    }

  return -1;
}


/*********************/
/* Screen management */
/*********************/

/* The value v is between l and h and can not be outside */
#define filter(v, l, h) (v<l ? l : (v>h ? h : v))

/* Computes the 16 grays level colors and allocates a colormap */
void set_colors(void)
{
  int i;
  int sr, sg, sb;
  int er, eg, eb;
  int r,g,b;
  UWORD cr, cg, cb;

  /*
  sr = 0xA8<<8; sg = 0xB4<<8; sb = 0xA8<<8;
  er = 0x00<<8; eg = 0x00<<8; eb = 0x34<<8;
  */
  sr = (*options.light_color & 0xff0000) >> 8;
  sg = (*options.light_color & 0x00ff00);
  sb = (*options.light_color & 0x0000ff) << 8;
  er = (*options.dark_color & 0xff0000) >> 8;
  eg = (*options.dark_color & 0x00ff00);
  eb = (*options.dark_color & 0x0000ff) << 8;

  if(iContrast < 16) 
    {
      sr = sr - (sr-er)*(16 - iContrast)/13;
      sg = sg - (sg-eg)*(16 - iContrast)/13;
      sb = sb - (sb-eb)*(16 - iContrast)/13;
    }
  else 
    {
      er = er - (er-sr)*(iContrast - 16)/13;
      eg = eg - (eg-sg)*(iContrast - 16)/13;
      eb = eb - (eb-sb)*(iContrast - 16)/13;
    }
  
  r = sr, g = sg, b = sb;

  if(iEmuState & ESTAT_SCREENON) 
    {
      for(i=0; i<=(iGrayPlanes+1); i++) 
	{
	  cr = filter(r, 0x0000, 0xA800);
	  cg = filter(g, 0x0000, 0xB400);
	  cb = filter(b, 0x3400, 0xA800);
       
	  switch(iDepth)
	    {
	    case 8:
	      sdlPal[i].r = r;   sdlPal[i].g = g;   sdlPal[i].b = b;
	      if(!SDL_SetColors(pScreen, sdlPal, 0, i+16))
		{
		  DISPLAY("Unable to allocate a color in the colormap.\n");
		  DISPLAY("Program aborted.\n");
		  exit(-1);
		}
	    case 16:
	    case 24:
	    case 32:
	      lcdPal[i] = SDL_MapRGB(pScreen->format, cr>>8, cg>>8, cb>>8);
	      break;
	    }

	  r-=((sr-er)/(iGrayPlanes+1));
	  g-=((sg-eg)/(iGrayPlanes+1));
	  b-=((sb-eb)/(iGrayPlanes+1));
	}
    }
}


int set_contrast(int c)
{
  if(ti68k_getCalcType() == TI89)
    c = 31-c;

  iNewContrast = (c+iLastContrast)/2;
  iLastContrast = c;
  return 0;
}

static int screen_on(void) 
{
  iEmuState |= ESTAT_SCREENON;
  return 0;
}

static int screen_off(void) 
{
  iEmuState &= ~ESTAT_SCREENON;
  redraw_skin(); //for clearing LCD
  return 0;
}

void set_screen_ptr(UBYTE* ptr) 
{
  pLcd = ptr;
}

void update_window(void) 
{
  if (ti68k_getCalcType() & TI92) 
    {
      iXLcd = 240; iYLcd = 128;
      if (!loadSkin("ti92.skn"))
	return;
    }
  else {
    iXLcd = 160; iYLcd = 100;
    if (!loadSkin("ti89.skn"))
      return;
  }
  
  if(glob_inf.background) 
    {
      iYWin = ySkin;
      iXWin = xSkin;
    }
  else 
    {
      iYWin = iYLcd;
      iXWin = iXLcd;
    }
  
  if (bFullscreen)
    switch_fullscreen();
  else
    switch_windowed(); 
}

static int update_screen(void) 
{
  UBYTE *ptr = ti68k_getLcdPtr(); //pLcd;
  UBYTE *ptr8;
  UWORD *ptr16;
  ULONG *ptr32;
  int i, j, k;

  if(iGrayPlanes != glob_inf.grayPlanes) 
    {
      iGrayPlanes = glob_inf.grayPlanes;
      set_colors();
    }

  if(!(iEmuState & ESTAT_SCREENON)) 
    return 0;

  if(iContrast != iNewContrast) 
    {
      iContrast = iNewContrast;
      set_colors();
    }

  // Convert the bitmap screen to a bytemap screen */
  if(!iGrayPlanes || !iCurrPlane) 
    { // no gray scale or init gray plane
      for(j=0, k=0; k<iYLcd; k++)
	for(i=0; i<iXMax>>3; i++, ptr++) 
	  {
	    pScreenBuf[j++] = convtab[((*ptr)<<1)  ];
	    pScreenBuf[j++] = convtab[((*ptr)<<1)+1];
	  }
    }
  else 
    { // compute gray scale
      for(j=0, k=0; k<iYLcd; k++)
	for(i=0; i<iXMax>>3; i++, ptr++) 
	  {
	    pScreenBuf[j++] += convtab[((*ptr)<<1)  ];
	    pScreenBuf[j++] += convtab[((*ptr)<<1)+1];
	  }
    }
  
  // Display computed gray scale if required
  if(iCurrPlane++ >= iGrayPlanes) 
  {
      int add2line = iLineLength - iXLcd;
      
      if ( SDL_MUSTLOCK(pScreen) ) 
	{
	  if ( SDL_LockSurface(pScreen) < 0 )
	    return -1;
	}

	  // For updating variable (with/without skin)
	iScrLineSize = (pScreen->pitch) >> (iBpp >> 1);

      switch(iDepth) 
	{
	case 8:
	  ptr = (UBYTE *)pScreenBuf;
	  ptr8 = (UBYTE *)pScreen->pixels;
	  if(glob_inf.background)
	    ptr8 += rcLcd.top*iScrLineSize + rcLcd.left;
	  for(j=0; j<iYLcd; j++) 
	    {
	      for (i=0;i<iXLcd;i++) 
		{
		  ptr8[j*iScrLineSize+i] = lcdPal[*ptr++];
		}
	      ptr += add2line;
	    }
	  break;
	case 16:
	  ptr = (UBYTE *)pScreenBuf;
	  ptr16 = (UWORD*)pScreen->pixels;
	  if(glob_inf.background)
	    ptr16 += rcLcd.top*iScrLineSize + rcLcd.left;
	  for(j=0; j<iYLcd; j++) 
	    {
	      for (i=0;i<iXLcd;i++) 
		{
		  ptr16[j*iScrLineSize+i] = lcdPal[*ptr++];
		}
	      ptr += add2line;
	    }
	  break;
	case 24:
	case 32:
	  ptr = (UBYTE *)pScreenBuf;
	  ptr32 = (ULONG *)pScreen->pixels;
	  if(glob_inf.background)
	    ptr32 += rcLcd.top*iScrLineSize + rcLcd.left;
	  for(j=0; j<iYLcd; j++) 
	    {
	      for (i=0;i<iXLcd;i++) 
		{
		  ptr32[j*iScrLineSize+i] = lcdPal[*ptr++];
		}
	      ptr += add2line;
	    }
	}
      
      iCurrPlane = 0;
      
      if ( SDL_MUSTLOCK(pScreen) ) 
	{
	  SDL_UnlockSurface(pScreen);
	}
      if(glob_inf.background)
	SDL_UpdateRect(pScreen, rcLcd.left, rcLcd.top, iXLcd, iYLcd);
      else
	SDL_UpdateRect(pScreen, 0, 0, iXLcd, iYLcd); 
  }

  return 0;
}


/* Redraw the skin */
void redraw_skin(void) 
{
  int i, j;
  UBYTE r, g, b;
  UBYTE *ptr8;
  UWORD *ptr16;
  ULONG *ptr32;
  
  if(!bSdlOkay) return;
  if(!glob_inf.background) return;
  
  if(SDL_MUSTLOCK(pScreen)) 
    {
      if( SDL_LockSurface(pScreen) < 0 )
	return;
    }
  iScrLineSize = pScreen->pitch>>(pScreen->format->BytesPerPixel>>1);

  // Load the skin colormap and allocate colors
  if(iBpp == 1)
    {
      for(i=0; i<skinNumberOfColors; i++)
	{
	  sdlPal[i+16].r = skinColormap[0][i];
	  sdlPal[i+16].g = skinColormap[1][i];
	  sdlPal[i+16].b = skinColormap[2][i];
	}
      if(!SDL_SetColors(pScreen, sdlPal, 0, i+16))
	{
	  DISPLAY("Unable to allocate a color in the colormap.\n");
	  DISPLAY("Program aborted.\n");
	  exit(-1);
	}
    }
  
  if(!glob_inf.background) return;

  // Fill the buffer with the pixmap
  switch(iDepth) 
    {
    case 8:
      ptr8 = (UBYTE *)pScreen->pixels;
      for (j=0;j<iYWin;j++)
	{
	  for (i=0;i<iXWin;i++) 
	    {
	      r = skinColormap[0][pSkin[j*iXWin+i]];
	      g = skinColormap[1][pSkin[j*iXWin+i]];
	      b = skinColormap[2][pSkin[j*iXWin+i]];
	      //ptr8[j*iScrLineSize+i] = 16 + ((UBYTE *)pSkin)[j*iXWin+i];
	      ptr8[j*iScrLineSize+i] = SDL_MapRGB(pScreen->format, r, g, b);
	    }
	}
      
      break;
    case 16:
      ptr16 = (UWORD *)pScreen->pixels;
      for (j=0;j<iYWin;j++)
	{
	  for (i=0;i<iXWin;i++) 
	    {
	      r = skinColormap[0][pSkin[j*iXWin+i]];
	      g = skinColormap[1][pSkin[j*iXWin+i]];
	      b = skinColormap[2][pSkin[j*iXWin+i]];
	      //ptr16[j*iScrLineSize+i] = ((r>>3)<<11) | ((g>>2)<<5) | (b>>3);
	      ptr16[j*iScrLineSize+i] = SDL_MapRGB(pScreen->format, r, g, b);
	    }
	}
      break;
    case 24:
    case 32:
      ptr32 = (ULONG *)pScreen->pixels;
      for (j=0;j<iYWin;j++)
	{
	  for (i=0;i<iXWin;i++) 
	    {
	      r = skinColormap[0][pSkin[j*iXWin+i]];
	      g = skinColormap[1][pSkin[j*iXWin+i]];
	      b = skinColormap[2][pSkin[j*iXWin+i]];
	      //ptr32[j*iScrLineSize+i] = (r << 16) | (g << 8) | b;
	      ptr32[j*iScrLineSize+i] = SDL_MapRGB(pScreen->format, r, g, b);
	    }
	}
      break;
    }
  
  if ( SDL_MUSTLOCK(pScreen) ) 
    {
      SDL_UnlockSurface(pScreen);
    }
  
  SDL_UpdateRect(pScreen, 0, 0, 0, 0);
}	


/****************/
/* GTK routines */
/****************/

/* A GTK callback: displays a popup menu */
gboolean
button_press_event        (GtkWidget       *widget,
                           GdkEventButton  *event,
                           gpointer         user_data)
{
  GtkWidget *menu;

  /* Displays the popup menu */
  if(event->button == 3)
    {
      menu = display_popup_menu();
      gtk_widget_grab_focus(menu);
      gdk_event_get();
      gtk_menu_popup(GTK_MENU(menu), 
		     NULL, NULL, NULL, NULL, 
		     event->button, event->time);
      suspend();
    }
  
  return FALSE;
}

/* The GTK auxiliary window */
int gtk_main_window(void)
{
  GtkWidget *eventbox;
  GtkWidget *label;
  
  if (bGtkOkay) return 0;
  gtk_init (NULL, NULL);
    
  /* The main window */
  window = gtk_window_new (GTK_WINDOW_TOPLEVEL);
  gtk_window_set_title (GTK_WINDOW (window), "GtkTiEmu");
  gtk_widget_set_usize(window, 250, 50);

  /* The event box */
  eventbox = gtk_event_box_new ();
  gtk_container_add (GTK_CONTAINER (window), eventbox);
  GTK_WIDGET_SET_FLAGS (eventbox, GTK_CAN_FOCUS);
  GTK_WIDGET_SET_FLAGS (eventbox, GTK_CAN_DEFAULT);
  gtk_widget_set_events (eventbox, GDK_BUTTON_PRESS_MASK | GDK_KEY_PRESS_MASK);
  gtk_widget_show (eventbox);

  label = gtk_label_new ("Click here to display a menu");
  gtk_container_add (GTK_CONTAINER (eventbox), label);
  gtk_widget_show (label);
  
  gtk_signal_connect (GTK_OBJECT (window), "destroy",
                      GTK_SIGNAL_FUNC (gtk_main_quit),
                      NULL);
  gtk_signal_connect (GTK_OBJECT (eventbox), "button_press_event",
                      GTK_SIGNAL_FUNC (button_press_event),
                      NULL);
  //gtk_widget_grab_focus (eventbox);
  gtk_widget_grab_default (eventbox);

#ifdef EXT_WIN  
  gtk_widget_show_all(window);
#endif

  bGtkOkay = 1;
  return 0;
}


/* Bitmap to bytemap conversion table */
void compute_convtable(void) 
{
  int i,j;
  UBYTE k;
  UBYTE *tmptab = (UBYTE *)convtab;

  for(i=0, j=0; i<256; i++) 
    {
      for(k = 1<<7; k; k>>=1)
	{
	  if(i & k)
	    tmptab[j++] = 1;
	  else
	    tmptab[j++] = 0;
	}
    }
}


int update_gui(void) 
{
  gtk_main_iteration_do(FALSE);
  return suspended;
}

int is_suspended() 
{
  return suspended;
}

void suspend(void) 
{
  suspended = 1;
}

void unsuspend(void) 
{
  suspended = 0;
}


int set_gui_callbacks(void)
{
  return ti68k_setGuiCallbacks(init_specific,
			       exit_specific,
			       update_screen,
			       update_keys,
			       screen_on,
			       screen_off,
			       NULL,
			       set_contrast);
  //gt_init_refresh_functions();
}


int do_screenshot(int format, int type, int size, char *filename)
{
  Image img;
  char outfile[MAXCHARS];
  char *ext = "???";
  FILE *fp = NULL;
  int row, col;
  UBYTE *ptr8;
  UWORD *ptr16;
  ULONG *ptr32;
  unsigned long l; // a pixel
  int i, j, k;
  SDL_Surface *pLcd = NULL;
  int add2line = iLineLength - iXLcd;

  if(filename == NULL)
    {
      switch(format)
	{
	case IMG_XPM: ext = "xpm"; break;
	case IMG_PCX: ext = "pcx"; break;
	case IMG_JPG: ext = "jpg"; break;
	case IMG_BMP: ext = "bmp"; break;
	default: break;
	}
      sprintf(outfile, "%s%03i.%s", options.screen_file, 
	      options.screen_counter, ext);
    }
  else
    strcpy(outfile, filename);
  
  if(format != IMG_BMP)
  {
	fp = fopen(outfile, "wb");
	if(fp == NULL) exit(-1);
  }
  
  DISPLAY("Screenshot to %s... ", outfile);
  //  
  // Write the TI screen in black & white
  //  
  if((type == IMG_BW) && (size == IMG_LCD))
    {
      img.height = iYLcd;
      img.width = iXLcd;
      img.inverted = !0;
      img.depth = 2;
      alloc_bitmap(&img);

      if(ti68k_getCalcType() & TI92)
	{
	  memcpy(img.bitmap, ti68k_getLcdPtr(), 
		 ((img.width * img.height) >> 3) * sizeof(byte));
	}
      else
	{
	  UBYTE *pLcd = ti68k_getLcdPtr();
	  for(j=0; j<iYLcd; j++) 
	    {
	      for (i=0;i<(iXLcd>>3);i++) 
		{
		  (img.bitmap)[j*(iXLcd>>3)+i] = *pLcd++;
		}
	      pLcd += add2line>>3;
	    }
	}
    }
  //
  // Write only the TI screen in color
  //
  if((type == IMG_COL) && (size == IMG_LCD))
    {
      img.height = iYLcd;
      img.width = iXLcd;
      img.depth = 255;

      if(format == IMG_BMP) // required in BMP format for SDL
	{
	  int iLcdLineSize;
	  UBYTE *ptr;
	  int i, j;

	  pLcd = SDL_CreateRGBSurface(SDL_SWSURFACE, iXLcd, iYLcd, iDepth, 0, 0, 0, 0);
	  SDL_SetColors(pLcd, sdlPal, 0, 256);
	  iLcdLineSize = (pLcd->pitch) >> (iBpp >> 1);
	  
	  ptr = (UBYTE *)pLcd->pixels;
	  ptr8 = (UBYTE *)pScreen->pixels;
	  ptr8 += rcLcd.top*iScrLineSize + rcLcd.left;
	  for(j=0; j<iYLcd; j++) 
	    {
	      for (i=0;i<iXLcd;i++) 
		{
		  ptr[j*iLcdLineSize+i] = ptr8[j*iScrLineSize+i];
		}
	      //ptr += add2line;
	    }
	}

      switch(iDepth) 
	{
	case 8: // rrggbb is 8 bits + palette
	  alloc_colormap(&img);
	  for(k=0; k<256; k++)
	    {
	      (img.colormap)[3*k+0] = sdlPal[k].r;
	      (img.colormap)[3*k+1] = sdlPal[k].g;
	      (img.colormap)[3*k+2] = sdlPal[k].b;
	    }  
	  ptr8 = (UBYTE *)pScreen->pixels;
	  ptr8 += rcLcd.top*iScrLineSize + rcLcd.left;
	  alloc_bytemap(&img);
	  for(row=0; row<iYLcd; row++)
	    {
	      for(col=0; col<iXLcd; col++)
		{
		  (img.bytemap)[row*iXLcd+col] = ptr8[row*iScrLineSize+col];
		}
	      //ptr+=add2line;
	    }
	  break;
	case 16: // rrggbb is 5/6/5 bits
	  ptr16 = (UWORD *)pScreen->pixels;
	  ptr16 += rcLcd.top*iScrLineSize + rcLcd.left;
	  alloc_rgbmap(&img);
	  for(row=0; row<iYLcd; row++)
	    {
	      for(col=0; col<iXLcd; col++)
		{	
		  // 16 planes only      
		  l = ptr16[row*iXLcd+col];
		  (img.rgbmap)[3*(row*iXLcd+col)+0] = ((l & 0xf800) >> 11) << 3;
		  (img.rgbmap)[3*(row*iXLcd+col)+1] = ((l & 0x7e0) >> 5) << 2;
		  (img.rgbmap)[3*(row*iXLcd+col)+2] = ((l & 0x1f)) << 3;
		}
	      //ptr+=add2line;
	    }
	  convert_rgbmap_to_bytemap(&img);
	  break;
	case 24:
	case 32: // rrggbb is 8/8/8 bits
	  ptr32 = (ULONG *)(pScreen->pixels);
	  ptr32 += rcLcd.top*iScrLineSize + rcLcd.left;
	  alloc_rgbmap(&img);
	  for(row=0; row<iYLcd; row++)
	    {
	      for(col=0; col<iXLcd; col++)
		{
		  l = ptr32[row*iXLcd+col];
		  (img.rgbmap)[3*(row*iXLcd+col)+0] = ((l & 0xff0000) >> 16);
		  (img.rgbmap)[3*(row*iXLcd+col)+1] = ((l & 0x00ff00) >>  8);
		  (img.rgbmap)[3*(row*iXLcd+col)+2] = ((l & 0x0000ff) >>  0);
		}
	      //ptr+=add2line;
	    }
	  convert_rgbmap_to_bytemap(&img);
	  break;
	}
    }
  //
  // Write the entire screen in color
  //  
  if(size == IMG_SKIN)
    {
      img.height = iYWin;
      img.width = iXWin;
      img.depth = 255;

      DISPLAY("iDepth: %i\n", iDepth);
      switch(iDepth) 
	{
	case 8: // rrggbb is 8 bits + palette
	  alloc_colormap(&img);
	  for(k=0; k<256; k++)
	    {
	      (img.colormap)[3*k+0] = sdlPal[k].r;
	      (img.colormap)[3*k+1] = sdlPal[k].g;
	      (img.colormap)[3*k+2] = sdlPal[k].b;
	    }  
	  ptr8 = (UBYTE *)pScreen->pixels;
	  alloc_bytemap(&img);
	  for(row=0; row<iYWin; row++)
	    {
	      for(col=0; col<iXWin; col++)
		{
		  (img.bytemap)[row*iXWin+col] = ptr8[row*iScrLineSize+col];
		}
	    }
	  break;
	case 16: // rrggbb is 5/6/5 bits
	  ptr16 = (UWORD *)pScreen->pixels;
	  alloc_rgbmap(&img);
	  for(row=0; row<iYWin; row++)
	    {
	      for(col=0; col<iXWin; col++)
		{
		  l = ptr16[row*iScrLineSize+col];
		  (img.rgbmap)[3*(row*iXWin+col)+0] = ((l & 0xf800) >> 11) << 3;
		  (img.rgbmap)[3*(row*iXWin+col)+1] = ((l & 0x7e0) >> 5) << 2;
		  (img.rgbmap)[3*(row*iXWin+col)+2] = ((l & 0x1f)) << 3;
		}
	    }
	  convert_rgbmap_to_bytemap(&img);
	  break;
	case 24:
	case 32: // rrggbb is 8/8/8 bits
	  ptr32 = (ULONG *)pScreen->pixels;
	  alloc_rgbmap(&img);
	  for(row=0; row<iYWin; row++)
	    {
	      for(col=0; col<iXWin; col++)
		{
		  l = ptr32[row*iXWin+col];
		  (img.rgbmap)[3*(row*iXWin+col)+0] = ((l & 0xff0000) >> 16);
		  (img.rgbmap)[3*(row*iXWin+col)+1] = ((l & 0x00ff00) >>  8);
		  (img.rgbmap)[3*(row*iXWin+col)+2] = ((l & 0x0000ff) >>  0);
		}
	    }
	  convert_rgbmap_to_bytemap(&img);
	  break;
	}
    }

  switch(format)
    { 
    case IMG_XPM: write_xpm_format(fp, &img); break;
    case IMG_PCX: write_pcx_format(fp, &img); break;
    case IMG_JPG: write_jpg_format(fp, &img); break;
    case IMG_BMP: 
      if(size == IMG_LCD)
	{
	  SDL_SaveBMP(pLcd, outfile);
	  SDL_FreeSurface(pLcd);
	}
      else
	SDL_SaveBMP(pScreen, outfile); 
      break;
    default: DISPLAY("Invalid image format\n"); break;
    }

  //delete_image(&img); // ??
    if(format != IMG_BMP)
	{
		fclose(fp);
	}
  DISPLAY("Done !\n");
  options.screen_counter++;

  return 0;
}


/*******************************/
/* SDLspecific public routines */
/*******************************/

void switch_with_skin(void)
{
  if (!bSdlOkay)
    return;	
  bSdlOkay = 0;
  SDL_FreeSurface(pScreen);
  
  if (!(pScreen = SDL_SetVideoMode(iXWin, iYWin, DEFAULT_BPP, 
				   DEFAULT_FLAGS))) 
    {
      fprintf(stderr,
	      "Couldn't switch to fullscreen\n");
    }
  iScrLineSize = (pScreen->pitch) >> (iBpp >> 1);
  
  glob_inf.background = 1;
  bSdlOkay = 1;
  redraw_skin();
  set_colors();
}

// Dangerous -> SDL crashes -> Linux crashes, shit !
void switch_without_skin(void)
{
  if (!bSdlOkay)
    return;	
  bSdlOkay = 0;
  SDL_FreeSurface(pScreen);
  
  if (!(pScreen = SDL_SetVideoMode(iXLcd, iYLcd, DEFAULT_BPP, 
				   DEFAULT_FLAGS))) 
    {
      fprintf(stderr, "Couldn't switch without skin\n");
    }
  iScrLineSize = (pScreen->pitch) >> (iBpp >> 1);
  
  bSdlOkay = 1;
  glob_inf.background = 0;
  set_colors();
}

void switch_fullscreen(void) 
{
  if (!bSdlOkay)
    return;	
  bSdlOkay = 0;
  SDL_FreeSurface(pScreen);
  
  if (!(pScreen = SDL_SetVideoMode(iXWin, iYWin, DEFAULT_BPP, 
				   DEFAULT_FLAGS | SDL_FULLSCREEN))) 
    {
      fprintf(stderr, "Couldn't switch with skin\n");
      if (!bWindowedFailed)
	switch_windowed();
      else
	exit(1);

      bFullscreenFailed = 1;
  }
  
  bFullscreen = 1;
  bFullscreenFailed = 0;
  bSdlOkay=1;
  redraw_skin();
  DISPLAY("Done.\n");
}

void switch_windowed(void) 
{
  if (!bSdlOkay)
    return;
  bSdlOkay = 0;
  SDL_FreeSurface(pScreen);
  
  if (!(pScreen = SDL_SetVideoMode(iXWin, iYWin, DEFAULT_BPP, DEFAULT_FLAGS))) 
    {
      DISPLAY("Couldn't switch to window mode\n");
      if (!bFullscreenFailed)
	switch_fullscreen();
      else
	exit(1);

      bFullscreenFailed = 1;
    }
  
  bFullscreen = 0;
  bWindowedFailed = 0;
  bSdlOkay = 1;
  redraw_skin();
}
