/*
    This file is part of SUPPL - the supplemental library for DOS
    Copyright (C) 1996-2000 Steffen Kaiser

    This library is free software; you can redistribute it and/or
    modify it under the terms of the GNU Library General Public
    License as published by the Free Software Foundation; either
    version 2 of the License, or (at your option) any later version.

    This library is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
    Library General Public License for more details.

    You should have received a copy of the GNU Library General Public
    License along with this library; if not, write to the Free
    Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/* $RCSfile: INIFILE.H $
   $Locker: ska $	$Name:  $	$State: Exp $

	Implementation of an interface to access INI files.

	The interface can be applied to .INI and registry-style files.
	However, the current implementation does support .INI files
	only.

	The format of .INI files is:
	+ Lines matching:
			/^\s*[;#]/ || /^\s*$/
		are completely ignored
	+ Lines matching:
			/^\[\s*([^\]+)\s*\]/
		are interpreted as keys or sections, with the name "$1".
		Any values preceeding the very first section belong to section
		NULL; section "" is supported, too.
	+ Lines matching:
			/^\s*([^=]+)\s*([\$\%\&])?(\!)?(\<\d+\>)?=/
		are interpreted as values within the current section/key.
		"$1" is the name of the value, "$2" is the type of the
		value data, "$'" is the value.
		"$3", if present, characterizes that the data is to be
		expanded (environment variables; comments stripped _before_
		expansion of the vars). If $3 is present, but $2 not, then
		$2 defaults to '$'.
		"$4", if present, specifies a trust level, which is ignored
		currently. (It's not retrieveble, too!)
		Note: All explicitly named data types ignore comments preceeded by ';'.
		The types are:
			$	--> string, possibly enclosed in "quotes", leading and
				trailing non-quoted whitespaces are stripped.
			%	--> integer
			&	--> boolean
	+ Any other lines might be interpreted as follow-ups of preceeding
		lines, or are ignored.

	Each application knows about two INI files, one system INI file
	and one user INI file.

	The system INI file is picked, what is found first:
		1. <appPath>\..\ETC\<appName>.INI
		2. <appPath>\<appName>.INI
	The user INI file is:
		%HOME%\ETC\<appName>.INI

   ----

	=== Function interface

	= General functions

	int cfgEqKey(const char * const key1, const char * const key2)
		Compare the names of the key names. According the convention:
			+ leading and trailing whitespaces are ignored
			+ key names are case-insensitive
			+ within a name any non-empty combination of whitespaces,
				dashes '-' and underscores '_' are interpreted as
				a single space (as a word delimiter).
			exp: "long-option" == " Long Option  " == "long _-OPTION"
			but: "-option" != "option" != " option"
			but: "-option" == "_option" == "   -option"

		Use this function for all:
			+ section names (== key names)
			+ names of values

		Return:  < 0: key1 < key2 (lexicographical order)
				== 0: key1 = key2
				 > 0: key1 > key2

	int cfgChangeNameINI(char *name)
		Changes the "<appName>" part within the names of the
		INI files above. This name must not contain path delimiters
		or an extension.
		Must be called before cfgLoadINI(), cfgCreateINI() and
		cfgInitINI() to take effect. 'name' is _not_ duplicated into the
		heap. If 'name' is NULL, <appName> is used.
		Return: 0: on success

	int cfgChangePathINI(char *path)
		Changes the <appPath> part in the search paths for the INI files
		above. If the path is not absolute, it will be fully-qualified
		within each call of cfgLoadINI(), cfgCreateINI() and cfgInitINI().
		Must be called before cfgLoadINI(), cfgCreateINI() and
		cfgInitINI() to take effect. 'path' is _not_ duplicated into the
		heap. If 'path' is NULL, <appPath> is used.
		Return: 0: on success

	== INI load interface

	= This set of functions is intended for situations, in which
	= a configuration is completely read from the INI file into
	= local configuration variables. After reading the INI file,
	= the file is closed and no "pending" actions or allocated
	= resources are left active.

	= Note that all of these functions end in 'INI' and start with
	= 'cfg'.
	= In addition all "writing" functions start with 'cfgCreate'.
	= The 'write' functions cannot coexist with the 'Random
	= access interface'.

	void cfgLoadINI(Cfg_parseFct fct, struct Cfg_Getopt * optstru)
		Load the INI file of the current application and pass
		all read options (values) through the option-scanning
		function.
		Actually this function reads two INI files, if they exist:
			* First, the system INI file
			* Second, the user INI file
		One or both files may be absent or unable to open/read,
		this is silently ignored.
		Because, probably, both files are scanned, the same value
		may be parsed two times, in which case the second one shall
		superceed the previous one.

	int cfgCreateINI(void)
		Create the user INI file %HOME%\ETC\<appName>.INI and open
		it for storing _a_complete_set_ of configuration options.
		Use in conjunction with: cfgCreateSectionINI(),
		cfgCreateValueINI() and cfgCreateCloseINI().
		Note: Until the cfgCreateCloseINI() function succeeded, the
		configuration is held in a temporary file, that means if the
		program dies during writing the configuration, the original
		configration file is left intact.
		Return: 0: on success

	int cfgCreateFileINI(const char * const fnam)
		as CreateINI(), but creates the specific file. The string
		is dupplicated into the heap.

	int cfgCreateSectionINI(char * section_name)
		Create a new section within the INI file already opened
		with cfgCreateINI(). The section name is preceeded by a blank
		line. section_name may be "", but not NULL, and is not
		processed in any way (no whitespaces are compressed or
		stripped, no case is mangled etc.)
		If section_name == NULL, this call is ignored; this shall
		help to generate automated lists to write out an INI file.
		Return: 0: on success

	int cfgCreateValueINI(char * name, char * data)
		Create a new value with the given data within the INI file already
		opened with cfgCreateINI().
		If this function is called prior cfgCreateSectionINI(), these
		values corresponse to the section NULL (== preceeding any
		section specification).
		Neither "name" nor "data" is not modified in any way.
		To add a type or trust level to the value, just append it
		to the value name, e.g.: "TEMP !" to apply "expandable string"
			to the data of value "TEMP".
		"data" can also contain a comment, e.g.: "5 ; <<comment>>"
		Return: 0: on success

	int cfgCreateStringINI(char *name, char *string)
		Create a new value as string type "$", with value string and the
		comment.
		The type ID is appended automatically to the name.
		To have a trust level, append it to the name already.
		"string" may be NULL, in which case it is assumed as "".
		'string' is automatically quoted.
		Return: 0: on success

	int cfgCreateExpandINI(char *name, char *string)
		As CreateString, but with an expandable string. This
		is a string, which might contain environment variables
		in the "%varname%" notion, which are automatically
		expanded before the data is returned.

	int cfgCreateIntegerINI(char *name, int num)
		As CreateString, but with a signed integer.

	int cfgCreateUnsignedINI(char *name, unsigned num)
		As CreateString, but with an unsigned integer.

	int cfgCreateBooleanINI(char *name, int num)
		As CreateString, but with a boolean value:
			num == 0	--> "true"
			else 		--> "false"

	int cfgCreateCommentINI(char *comment)
		Insert a full-line comment.
		'\n' (newline) characters may be used to form multi-line
		comments and/or prepend or append empty lines in front of or
		after the comment.
		Exactly one '\n' character is appended to the string.
		exp: CreateComment("\nThis is a comment\n")
		results in:
		|
		|; This is a comment
		|
		The '|' only mark the generated lines in this example and do
		not show up in the INI file.
		This function may be called immediately after another
		cfgCreate*() function, in which case the first line is
		appended to the value or section previously written. To
		prevent this put a '\n' character at the first character of
		the comment.
		Note: This mechanism was implemented into this function
		rather than into the other value-writing functions, because
		it is assumed that program-generated INI files usually do not
		contain comments a lot.

	int cfgCreateCloseINI(void)
		Close the user INI file.
		If this succeeds the user INI file is overwritten by the
		actually created file, however, for security reasons the
		old file is backup'ed into *.BAK.
		Return: 0: on success

	int cfgCreateCloseINI(void)
		Close the user INI file, but throw away all changes.
		Return: cfg error code

	int cfgViewErrorINI(char *msg, int errnr)
		Display a string corresponding to the error number returned
		by any of the cfgCreate*() functions (if errnr != 0)
		in the form:
			printf("%s: %s\n", msg, cfgErrorString(errnr))
		'msg' maybe NULL.
		Return:
			0: on success
			CFG_ERR_ARG: if unknown error number



	== Random access interface

	= These functions are intended for situations when one cannot read
	= all the configuration into the application.

	= All "write" functions directly goes into the user INI file.
	= If none exists, it is created.
	= If the path %HOME%\ETC does not exist, it is NOT created.
	= "Read" functions are served from the user INI file first, then,
	= if failed, from the system INI file.

	= Algorithm to access values.
	= The cases in which the requested section is in only one of the
	= two files are handled as expected (e.g. one file absent or
	= section only in one file present).
	= Otherwise: The cfgHasValue() first looks in the user INI file,
	= then in the system INI file.
	= The cfgNxtValue() will ignore any values of the system INI file
	= that are already present in the user INI file; this is different
	= to the behaviour that within the user INI file multiple values
	= named equally are returned.
	= The cfgNxtAnyValue() does not make this difference.
	= The cfgHasUserValue() will look in the user INI file only.
	= The cfgHasSystemValue() will look in the system INI file only.
	= If a section appears more than once within the same INI file,
	= the behaviour is undefined. It is strongly discouraged to
	= rely on the current implementation.

	= Any functions, that receives the name of a value as argument,
	= have the following behaviour: (NOT for sections!)
	= + if the passed name is NULL, the currently parsed value is
	=	processed. If none is parsed already, the function fails.
	= + if the passed name is the name of currently parsed value,
	=	this value is re-used. To be absolutely sure to get returned
	=	the first appearance of the value, call cfgSectionReset()
	=	before.
	= + otherwise, the complete section is re-scanned for the
	=	first appearance of the requested value.
	= + Neither the passed pointer nor the passed string itself
	=	will be modified by the functions.

	= Note: Until cfgExitINI() is called, there are opened/allocated
	= internal/external resources. One of these ones are two file
	= handles, one for the system and one for the user INI file.
	= This interface cannot co-exist with the 'write' functions of the
	= 'INI load interface'.

	= Deleting a section or value does not require that the section
	= or value disappears from retreival, because only the user
	= INI files are processed, but the system INI file remains
	= unchanged.

	int cfgInitINI(void)
		Initializes the INI interface for random access.
		To change the default INI filename or the path to the system
		INI file, call cfgChangeNameINI() or cfgChangePathINI()
		before calling this function.
		Must be called _before_ any of the other "random access" functions
		can be called.
		Return: 0: on success

	int cfgChangeFlagI(CFG_FLAG flag, int value)
		Change an internal flag:
			name		meaning [valid values|default]
			EXPAND		always expand the data [boolean | no]
			SIGN		prepend positive integer numbers by a
						plus sign '+' (never unsigned) [boolean | no]
		Return:
			-1: on error (unknown flag / invalid value)
			else: previous value of flag
				boolean values return: 0 == no, 1 == yes

	int cfgExitINI(void)
		Free all resources opened/allocated by cfgInitINI(). After this
		function was called, no other "random access" function can be
		called, except for cfgInitINI().
		Return: 0: on success

	int cfgReset(void)
		Resets all internal variables and open section NULL.
		All currently cached information is discarded.
		This function will also make sure that asynchroneously changes
		to the INI file(s) are detected and the internal variables
		are adjusted accordingly.
		This function can be called any time.
		Returns the same value as cfgOpenSection(NULL)

	int cfgOpenSection(char *name)
		Seek to section 'name'. cfgGetNxtValue() will return the first
		value available within this section (user INI file first).
		If name == NULL, the file rewinds to the start of it, meaning
		before any '[<section_name>]' definition.
		If name == "", an empty section is searched, e.g. "[  ]".
		Any cfgVal() function will process this section only.
		Return: 0: on success

	int cfgDynOpenSection(char *name)
		As cfgOpenSection(), but the supplied name is already located
		in the heap and will not be duplicated again. The system
		will free this string when necessary.

	int cfgHasUserSection(char *name)
		Check and positions to, if so, if a section of this name
		exists in the user INI file.
		Return: 0: on success

	int cfgHasSystemSection(char *name)
		Check and positions to, if so, if a section of this name
		exists in the system INI file.
		Return: 0: on success

	char *cfgGetCurSecName(void)
		return the name of the current section.
		Return:
			NULL: "out of section" <-> the values at the start of
				an INI file without a preceeding section definition.
			else: pointer to the name of the section as it is
				written within the INI file. If the section
				was opened with OpenSection(), it is the name from the
				user INI file, if present there.

	int cfgGetCurSecLocation(void)
		Return the location of the current section. Only valid after
		NxtSection() or AnyNxtSection().
		Return:
			<0: cfg error code
			=0: system INI file
			>0: user INI file

	int cfgFirstSection(int userfile);
		Open the very first section in either the user or the system
		INI file. The 'NULL' section is skipped.
		Return:	cfg error code

	int cfgAnyNxtSection(void)
		Advance to the next section (from the currently opened section).
		If the same section appears in both the user and system INI file,
		it is returned two times.
		Return:	cfg error code
			ERR_END: is returned if there is no section to advance to
			ERR_CHGINI:	is returned if the user INI file was changed
				in a way that the function lost its synchroneousity and
				cannot proceed.

	int cfgNxtSection(void)
		Advance to the next section (from the currently opened section).
		In opposite of the AnyNxtSection() function all sections of the
		system INI file are skipped that are also located in the user
		INI file.
		Return: cfg error code

	int cfgHasValue(char *name)
		Try to find and parse the value with the specified name,
		first in the user INI file, if failed, in the system
		INI file.
		If name == NULL, the function returns immediately with an
		return value if there was a value already paresed.
		On success the value is parsed, too.
		Return: 0: on success

	int cfgHasUserValue(char *name)
		Try to find and parse the value with the specified name,
		in the user INI file only.
		On success the value is parsed, too.
		Return: 0: on success

	int cfgHasSystemValue(char *name)
		Try to find and parse the value with the specified name,
		in the system INI file only.
		On success the value is parsed, too.
		Return: 0: on success

	int cfgFirstValue(int userfile)
		Advance to and parse the first value of the opened section within
		the user or system INI file.
		Return: cfg error code
			ERR_END:	is returned if the section has no value in the
						specified INI file

	int cfgAnyNxtValue(void)
		Advance to and parse the next value (from the current one).
		If the value is multiplely located in the same INI file, it is
		returned all the time.
		If the value is located in both the user and system INI file,
		all values are returned.
		Immediately after this call the function 'cfgIsUserValue()' returns
		where the found value is located.
		The retreive the data of this value, call any cfgVal*() function
		with name == 'NULL'.
		The "next" value is determined by current file position, any function
		that seeks within the opened INI file will cause to let AnyNxtValue()
		return wrong information.
		Any function that opens a section or parses a value will reset the
		system to make realible calls to this function.
		Return:	cfg error code
			ERR_END: is returned if there is no value to advance to
			ERR_CHGINI:	is returned if the user INI file was changed
				in a way that the function lost its synchroneousity and
				cannot proceed.

	int cfgNxtValue(void)
		As AnyNxtValue(), but if the value is located in the user INI
		file, any appearances in the system INI file are ignored.
		Return: cfg error code

	int cfgGetValLocation(char *name)
		Return the location (which INI file) the value is located in.
		Return:
			<0: error
			=0: system INI file
			>0: user INI file (be prepared for more than one user file)

	int cfgGetValType(char *name)
		Return the type of the data of the value with the specified name.
		The type is that one supplied in the definition of the value,
		left of the equal sign '='.
		Return:
			CFG_TERROR: on error
			else: type ID

	char *cfgTypeString(int type)
		Return a string corresponding to the binary type, e.g.
		"expandable string" or "integer".
		Return: always != NULL, pointer to internal read-only buffer
			"unknown type": on error

	char *cfgGetValName(char *name)
		Return the internal pointer to the name of the value.
		Use cfgGetValName(NULL) to return the name of the current
		value.
		Return:
			NULL: no such value
			else: pointer to the name

	char *cfgGetValComment(char *name)
		Return the comment attached to the data of the value with the
		specified name.
		This function destroys the currently parsed value in such
		way, that cfgValBuffer() can no longer succeed. However,
		ValBuffer() can be called and the data be used _before_
		calling GetValComment().
		The returned string points to an internal buffer might
		be overwritten, when the next cfg*() function is called.
		Return:
			NULL: no comment attached
			else: pointer to comment

	char *cfgGetValData(char *name)
		Return a pointer to the data of the value. Previous calls to
		GetValComment() or one of cfgVal*() functions may modify the
		contents of this string later or, if called after such function,
		the contents might not directly corresponse to the data read from
		the INI file, e.g. the environment variables could have been 
		expanded already.
		If this function is called before any of these functions, the
		data is returned in unmodified form, including the comment,
		leading and trailing whitespaces and environment variables.

	int cfgValInteger(char *name, int *num, int dflt)
		Return the data of value 'name' re-parsed as signed integer.
		If the parsing fails, dflt is returned.
		The value can be of the (unsigned) range.
		A negative value could be constructed by applying one minus '-'
		sign to a value less than or equal to INT_MAX, or by using an
		unsigned value greater than INT_MAX.
		There will be applied a range check: INT_MIN <= x <= INT_MAX
		for values _with_ a sign, and 0 <= x <= UINT_MAX for values
		without a sign.

	int cfgValBoolean(char *name, int *num, int dflt)
		Return the data of value 'name' re-parsed as boolean.
		If the parsing fails, dflt is returned.

	char *cfgValString(char *name, char **num, char *dflt)
		Return the data of the value 'name' re-parsed as string.
		If the parsing fails, dflt is returned. Note: The returned
		pointer is either 'dflt' or points to an internal buffer, that
		might change the next time one of the cfg*() functions is called.

	char *cfgValExpand(char *name, char **num, char *dflt)
		As cfgValString(), but even expand environment variables,
		if the type information within the INI file doesn't tell so.

	int cfgDeleteSection(char *name)
		Delete the whole section(s) from the user INI file.
		If the section appears more than once in the user INI file,
		all are deleted.
		Return: 0: on success; section 'NULL' is opened

	int cfgDeleteCurSection(void)
		Delete the current section, and only it.
		Return: 0: on success; section 'NULL' is opened

	int cfgDeleteValue(char *name)
		Delete the value with the specified name from the current
		section. If the value appears more than once in the section,
		all are deleted.
		If name == NULL, the current value is deleted, and only it.
		Return: 0: on success; no value is currently parsed.

	int cfgAddValue(char * name, char * data)
		Append a new value with the given data to the end of the
		current section.
		If the value already exists, the elder one is not deleted.
		Neither "name" nor "data" is not modified in any way.
		To add a type or trust level to the value, just append it
		to the value name, e.g.: "TEMP !" to apply "expandable string"
			to the data of value "TEMP".
		"data" can also contain a comment, e.g.: "5 ; <<comment>>"
		Return: 0: on success

	int cfgAddString(char *name, char *string, char *comment)
		wrapper of AddValue().
		Add a new value as string type "$", with value string and the
		comment.
		The type ID is appended automatically to the name.
		To have a trust level, append it to the name already.
		To have the 'expandable' flag set, append to the name.
		Both "string" and "comment" may be NULL, in which:
			string --> "" and the comment is not appended.
		Return: 0: on success

	int cfgAddExpand(char *name, int num, char *comment)
		As AddString, but with an expandable string. This
		is a string, which might contain environment variables
		in the "%varname%" notion, which are automatically
		expanded before the data is returned.

	int cfgAddInteger(char *name, int num, char *comment)
		As AddString, but with a signed integer.

	int cfgAddUnsigned(char *name, unsigned num, char *comment)
		As AddString, but with an unsigned integer.

	int cfgAddBoolean(char *name, int num, char *comment)
		As AddString, but with a boolean value:
			num == 0	--> "true"
			else 		--> "false"

	int cfgSetValue(char *name, char *data)
	int cfgSetString(char *name, char *string, char *comment)
	int cfgSetExpand(char *name, int num, char *comment)
	int cfgSetInteger(char *name, int num, char *comment)
	int cfgSetInteger(char *name, unsigned num, char *comment)
	int cfgSetBoolean(char *name, int num, char *comment)
		these functions corresponse to the cfgAdd*() functions,
		but remove any appearance of the value first.
		If name == NULL, it will pass NULL to the DeleteValue()
		function, but will use the current name to create the
		value. If there is no current value, the functions fail.

	int cfgNewSection(char *name)
		Create a new section with the specified name.
		If such section already exists within the user INI file,
		this function fails.
		If the operation succeeds, the section is opened for
		writing and, if it exists in the system INI file, for
		read access as well.
		Return:
			0: on success
			CFG_ERR_EXIST: if the section already exists
			else: on failure

	int cfgEraseSection(char *name)
		Open the specified section for read & write. No value is
		copied in the section automatically.
		If none such section already exists, a new section
		is created.
		Return: 0: on success

	int cfgEraseCurSection(void)
		As EraseSection(), but with the currently opened section.

	int cfgCloseWrite(void)
		Closes the output INI file and replace the user INI file
		by the new one. After this, cfgOpenSection(NULL) is
		performed.
		Note: This function should not be called, if one of the
		INI write functions returned an error code, because in this
		case the generated INI file might contain errors. Call
		cfgReset() to ignore the changes.
		Return:	cfg error code, but the output file is closed in all
			circumstances

   ----

	Target compilers:
   		Micro-C v3.14, Borland C++ v3.1+

   ----
	
	Known bugs:

   ----
	
*/

#ifndef __INIFILE__H_
#define __INIFILE__H_

#ifndef _MICROC_
#include <string.h>
#endif
#include <portable.h>
#include "cfg_base.h"

/* error codes */
#define CFG_ERR_NONE	0		/* no error, must be 0 (zero) ! */
#define CFG_ERR_ARG		-1		/* passed argument(s) invalid */
#define CFG_ERR_MEMORY	-2		/* out of memory */
#define CFG_ERR_NOTOPEN	-3		/* output file not open */
#define CFG_ERR_ACCESS	-4		/* general access error */
#define CFG_ERR_OPEN	-5		/* cannot open file */
#define CFG_ERR_ALREADY	-6		/* INI file already opened */
#define CFG_ERR_USRHOME	-7		/* user home directory does not exist */
#define CFG_ERR_KEY		-8		/* key not found */
#define CFG_ERR_EXIST	-9		/* section/value already exists */
#define CFG_ERR_END		-10		/* no section/value to advance to */
#define CFG_ERR_INICHG	-11		/* unrecoverable change of INI file */
#define CFG_ERR_COMMENT	-12		/* line is a comment */
#define CFG_ERR_LINE	-13		/* ill-formed line in INI file */
#define CFG_ERR_USRFILE	-14		/* no user INI file */
#define CFG_ERR_WRITE	-15		/* cannot write to INI file */
#define CFG_ERR_READ	-16		/* cannot read from INI file */

#define CFG_ERR_MAXNUM	16		/* maxmimum cfg error number */

#define CFG_NOTRUSTLEVEL	0	/* no associated trust level */
#define CFG_DEFAULT_TRUSTLEVEL	(-1)	/* use I(trustlevel) */

/*
 *	INI sub system flags
 */
#define CFG_FLAG int


/*
 *	Enumeration declarations
 */

/* Which sections/values to be included within Enum() */
#define CFG_FSTD	0		/* all from user INI, only those from system
								INI, which are not present in user INI */
#define CFG_FUSER	1		/* only all from user INI */
#define CFG_FSYSTEM	2		/* only all from system INI */
#define CFG_FALL	3		/* all from both user and system INI */

/* Structure passed to a section enumeration function: Cfg_secEnumData */
/* Type of callback function for enumerating sections: Cfg_secEnumFunc */

/* !!! Note: Only the members with exactly one underscore are portably
	supported by future versions of SUPPL !!!
	They are necessary to let cfgOpenEnumSection() work properly.

	All members are read-only */

/* The function receives a pointer to the filled key information
	structure and the arbitary pointer passed to the Enum() function.
	The return value is evaluated as follows:
		<0: cfg error code --> break enumeration, return this value
		=0: continue enumeration (== CFG_ERR_NONE)
		>0: function exit code --> break enumeration, return this value
*/
struct CFG_SECENUM {
	fpos_t cfgi___kpos;
	fpos_t cfgi___vpos;
	char *cfgi_keyname;		/* name of section currently found */
	int cfgi_location;		/* location: 0 == system INI file, else user */
	int cfgi___mode;
	int cfgi___mods;
};
#ifdef _MICROC_
#define Cfg_secEnumData struct CFG_SECENUM
#define Cfg_secEnumFunc int
#else
typedef struct CFG_SECENUM Cfg_secEnumData;
typedef int (*Cfg_secEnumFunc)(const Cfg_secEnumData * const enumData
	, void * const arg);
#endif

/* Enumeration function for values of the currently active section */
/* Return value as Cfg_secEnumFunc.
	The found value is made the current value, therefore the normal
	cfgGelVal*() functions with name == NULL can be called to retrieve
	the information about the value.
*/
#ifdef _MICROC_
#define Cfg_valEnumFunc int
#else
typedef int (*Cfg_valEnumFunc)(void * const arg);
#endif


/* Type of the add/set functions (see cfgAddValue() & cfgSetValue()) */
#ifdef _MICROC_
#define Cfg_addSetFct int
#else
typedef int (*Cfg_addSetFct)(const char * const name, const char * const data
 , const char * const comment, int type, int trustlevel);
#endif


/****************************************
 ***** cfg write INI functions **********
 ****************************************/
int cfgCreateINI(void);
int cfgCreateSectionINI(const char * const section_name);
//int cfgCreateValueINI(const char * const name, const char * const data);
#define cfgCreateValueINI(name,data)	\
	cfgCreateDataLine((name), (data), CFG_TNONE, 0)
int cfgCreateDataLineINI(const char * const name, const char * const data
 , int type, int trustlevel);
int cfgCreateStringINI(const char * const name, const char * const string);
int cfgCreateExpandINI(const char * const name, const char * const string);
int cfgCreateIntegerINI(const char * const name, int num);
int cfgCreateUnsignedINI(const char * const name, unsigned num);
int cfgCreateBooleanINI(const char * const name, int num);
int cfgCreateCommentINI(const char * const comment);
int cfgCreateCloseINI(void);
int cfgCreateIgnoreINI(void);
int cfgViewErrorINI(const char * const msg, int errnr);


/****************************************
 ***** cfg random access INI functions **
 ****************************************/

/***** read access ********************** 
 ****************************************/

/***** common functions *****************/
int cfgInitINI(void);
int cfgChangeFlagI(CFG_FLAG flag, int value);
int cfgExitINI(void);
int cfgReset(void);

/***** section related functions ********/
int cfgOpenSection(const char * const name);
int cfgDynOpenSection(char * const name);
int cfgHasUserSection(const char * const name);
int cfgHasSystemSection(const char * const name);
char *cfgGetCurSecName(void);
//int cfgGetCurSecLocation(void);
int cfgEnumSections(int mode, Cfg_secEnumFunc fct, void * const arg);
int cfgOpenEnumSection(const Cfg_secEnumData * const keyinfo);
//int cfgFirstSection(int userfile);
//int cfgAnyNxtSection(void);
//int cfgNxtSection(void);

/***** value related functions **********/
int cfgEnumValues(int mode, Cfg_valEnumFunc fct, void * const arg);
//int cfgFirstValue(int userfile);
//int cfgAnyNxtValue(void);
//int cfgNxtValue(void);
int cfgHasUserValue(const char * const name);
int cfgHasSystemValue(const char * const name);
int cfgHasValue(const char * const name);
int cfgGetValType(const char * const name);
int cfgGetValTrustlevel(const char * const name);
char *cfgGetValName(const char * const name);
char *cfgGetValData(const char * const name);
int cfgGetValLocation(const char * const name);
char *cfgGetValComment(const char * const name);
char *cfgValString(const char * const name, char * const dflt);
char *cfgValExpand(const char * const name, char * const dflt);
int cfgValInteger(const char * const name, int const dflt);
int cfgValBoolean(const char * const name, int const dflt);
char *cfgTypeString(int type);
	

/***** write access *********************
 ****************************************/

/***** delete information, method 2.1) **/
int cfgDeleteSection(const char * const name);
int cfgDeleteCurSection(void);
int cfgDeleteValue(const char * const name);
//int cfgDeleteCurValue(void);
#define cfgDeleteCurValue()		cfgDeleteValue(0)

/***** add/alter information, method 2.1) */
int cfgAddValue(const char * const name, const char * const data, const char * const comment, int type, int trustlevel);
int cfgSetValue(const char * const name, const char * const data, const char * const comment, int type, int trustlevel);

//int cfgAddString(const char * const name, const char * const string, const char * const comment);
//int cfgAddExpand(const char * const name, const char * const string, const char * const comment);
//int cfgAddInteger(const char * const name, int num, const char * const comment);
//int cfgAddUnsigned(const char * const name, unsigned num, const char * const comment);
//int cfgAddBoolean(const char * const name, int num, const char * const comment);
//int cfgSetString(const char * const name, const char * const string, const char * const comment);
//int cfgSetExpand(const char * const name, const char * const string, const char * const comment);
//int cfgSetInteger(const char * const name, int num, const char * const comment);
//int cfgSetUnsigned(const char * const name, unsigned num, const char * const comment);
//int cfgSetBoolean(const char * const name, int num, const char * const comment);

int cfgASString(Cfg_addSetFct fct, const char * const name, const char * const string, const char * const comment);
int cfgASExpand(Cfg_addSetFct fct, const char * const name, const char * const string, const char * const comment);
int cfgASInteger(Cfg_addSetFct fct, const char * const name, int num, const char * const comment);
int cfgASUnsigned(Cfg_addSetFct fct, const char * const name, unsigned num, const char * const comment);
int cfgASBoolean(Cfg_addSetFct fct, const char * const name, int num, const char * const comment);

#define cfgAddString(n,s,c) cfgASString(aF(cfgAddValue), (n), (s), (c))
#define cfgAddExpand(n,s,c) cfgASExpand(aF(cfgAddValue), (n), (s), (c))
#define cfgAddInteger(n,s,c) cfgASInteger(aF(cfgAddValue), (n), (s), (c))
#define cfgAddUnsigned(n,s,c) cfgASUnsigned(aF(cfgAddValue), (n), (s), (c))
#define cfgAddBoolean(n,s,c) cfgASBoolean(aF(cfgAddValue), (n), (s), (c))
#define cfgSetString(n,s,c) cfgASString(aF(cfgSetValue), (n), (s), (c))
#define cfgSetExpand(n,s,c) cfgASExpand(aF(cfgSetValue), (n), (s), (c))
#define cfgSetInteger(n,s,c) cfgASInteger(aF(cfgSetValue), (n), (s), (c))
#define cfgSetUnsigned(n,s,c) cfgASUnsigned(aF(cfgSetValue), (n), (s), (c))
#define cfgSetBoolean(n,s,c) cfgASBoolean(aF(cfgSetValue), (n), (s), (c))


/***** method 2.2) **********************/
int cfgNewSection(const char * const name);
int cfgEraseSection(const char * const name);
int cfgEraseCurSection(void);
int cfgCloseWrite(void);


#endif /*#	!defined(__HASHFILE__H_) */
