/*
 * $Header: /home/cvspublic/jakarta-commons/httpclient/src/java/org/apache/commons/httpclient/HttpConnectionManager.java,v 1.10 2002/09/03 01:36:26 jsdever Exp $
 * $Revision: 1.10 $
 * $Date: 2002/09/03 01:36:26 $
 * ====================================================================
 *
 * The Apache Software License, Version 1.1
 *
 * Copyright (c) 1999-2002 The Apache Software Foundation.  All rights
 * reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 *
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in
 *    the documentation and/or other materials provided with the
 *    distribution.
 *
 * 3. The end-user documentation included with the redistribution, if
 *    any, must include the following acknowlegement:
 *       "This product includes software developed by the
 *        Apache Software Foundation (http://www.apache.org/)."
 *    Alternately, this acknowlegement may appear in the software itself,
 *    if and wherever such third-party acknowlegements normally appear.
 *
 * 4. The names "The Jakarta Project", "HttpClient", and "Apache Software
 *    Foundation" must not be used to endorse or promote products derived
 *    from this software without prior written permission. For written
 *    permission, please contact apache@apache.org.
 *
 * 5. Products derived from this software may not be called "Apache"
 *    nor may "Apache" appear in their names without prior written
 *    permission of the Apache Group.
 *
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND ANY EXPRESSED OR IMPLIED
 * WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
 * OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED.  IN NO EVENT SHALL THE APACHE SOFTWARE FOUNDATION OR
 * ITS CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
 * SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
 * LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF
 * USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT
 * OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 * ====================================================================
 *
 * This software consists of voluntary contributions made by many
 * individuals on behalf of the Apache Software Foundation.  For more
 * information on the Apache Software Foundation, please see
 * <http://www.apache.org/>.
 *
 * [Additional notices, if required by prior licensing conditions]
 *
 */

package org.apache.commons.httpclient;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.LinkedList;

/**
 * Manages a set of HttpConnections for various host:ports.  This class is
 * used by HttpMultiClient.  
 *
 * @author Marc A. Saegesser
 */
public class HttpConnectionManager
{
    // -------------------------------------------------------- Class Variables
    /** Log object for this class. */
    private static final Log log = LogFactory.getLog(HttpConnectionManager.class);

    // ----------------------------------------------------- Instance Variables
    private HashMap mapHosts = new HashMap();
    private HashMap mapNumConnections = new HashMap();
    private int maxConnections = 2;   // Per RFC 2616 sec 8.1.4
    private String proxyHost = null;
    private int proxyPort = -1;

    /**
     * No-args constructor
     */
    public HttpConnectionManager() {
    }

    /**
     * Set the proxy host to use for all connections.
     * 
     * @param proxyHost - the proxy host name
     */
    public void setProxyHost(String proxyHost) {
        this.proxyHost = proxyHost;
    }

    /**
     * Get the proxy host.
     *
     * @return the proxy host name
     */
    public String getProxyHost() {
        return proxyHost;
    }

    /**
     * Set the proxy port to use for all connections.
     *
     * @param proxyPort - the proxy port number
     */
    public void setProxyPort(int proxyPort) {
        this.proxyPort = proxyPort;
    }

    /**
     * Get the proxy port number.
     *
     * @return the proxy port number
     */
    public int getProxyPort() {
        return proxyPort;
    }

    /**
     * Set the maximum number of connections allowed for a given host:port.
     * Per RFC 2616 section 8.1.4, this value defaults to 2.
     *
     * @param maxConnections - number of connections allowed for each host:port
     */
    public void setMaxConnectionsPerHost(int maxConnections) {
        this.maxConnections = maxConnections;
    }

    /**
     * Get the maximum number of connections allowed for a given host:port.
     *
     * @return The maximum number of connections allowed for a given host:port.
     */
    public int getMaxConnectionsPerHost() {
        return maxConnections;
    }

    /**
     * Get an HttpConnection for a given URL.  The URL must be fully
     * specified (i.e. contain a protocol and a host (and optional port number).
     * If the maximum number of connections for the host has been reached, this
     * method will block forever until a connection becomes available.
     *
     * @param sURL - a fully specified URL.
     * @return an HttpConnection for the given host:port
     * @exception java.net.MalformedURLException
     * @exception org.apache.commons.httpclient.HttpException -
     *            If no connection becomes available before the timeout expires
     */
    public HttpConnection getConnection(String sURL)
    throws HttpException, MalformedURLException {
        return getConnection(sURL, 0);
    }

    /**
     * Return the port provided if not -1 (default), return 443 if the
     * protocol is HTTPS, otherwise 80. 
     *
     * This functionality is a URLUtil and may be better off in URIUtils
     *
     * @param protocol the protocol to use to get the port for, e.g. http or
     *      https
     * @param port the port provided with the url (could be -1)
     * @return the port for the specified port and protocol.
     */
    private static int getPort(String protocol, int port) {
        log.trace("HttpConnectionManager.getPort(String, port)");

        // default to provided port
        int portForProtocol = port;
        if (portForProtocol == -1) {
            if (protocol.equalsIgnoreCase("HTTPS")) {
                portForProtocol = 443;
            } else {
                portForProtocol = 80;
            }
        }
        return portForProtocol;
    }
    
    /**
     * Get an HttpConnection for a given URL.  The URL must be fully
     * specified (i.e. contain a protocol and a host (and optional port number).
     * If the maximum number of connections for the host has been reached, this
     * method will block for <code>timeout</code> milliseconds or until a 
     * connection becomes available.  If no connection becomes available before
     * the timeout expires an HttpException exception will be thrown.
     *
     * @param sURL - a fully specified URL.
     * @param timeout - the time (in milliseconds) to wait for a connection 
     *      to become available
     * @return an HttpConnection for the given host:port
     * @exception java.net.MalformedURLException
     * @exception org.apache.commons.httpclient.HttpException -
     *            If no connection becomes available before the timeout expires
     */
    public HttpConnection getConnection(String sURL, long timeout) 
    throws HttpException, MalformedURLException {
        log.trace("enter HttpConnectionManager.getConnection(String, long)");

        // FIXME: This method is too big
        if (sURL == null) {
            throw new MalformedURLException("URL is null");
        }

        URL url = new URL(sURL);
        // Get the protocol and port (use default port if not specified)
        String protocol = url.getProtocol();
        String host = url.getHost();
        int port = HttpConnectionManager.getPort(protocol, url.getPort());
        final String hostAndPort = host + ":" + port;

        if (log.isDebugEnabled()) {
            log.debug("HttpConnectionManager.getConnection:  key = "
                + hostAndPort);
        }

        // Look for a list of connections for the given host:port
        LinkedList listConnections = getConnections(hostAndPort);
        
        HttpConnection conn = null;
        // get a connection from the 'pool', waiting for 'timeout' if no
        // connections are currently available
        synchronized(listConnections) {
            if (listConnections.size() > 0) {
                conn = (HttpConnection)listConnections.removeFirst();
            } else {
                // get number of connections to host:port
                Integer numConnections =  getConnectionsInUse(hostAndPort);
                if (numConnections.intValue() < maxConnections) {
                    // Create a new connection
                    boolean isSecure = protocol.equalsIgnoreCase("HTTPS");
                    conn = new HttpConnection(proxyHost, proxyPort, host, port,
                        isSecure);
                    numConnections = new Integer(numConnections.intValue() + 1);
                    mapNumConnections.put(hostAndPort, numConnections);
                } else {
                    conn = waitForConnection(listConnections, timeout);
                }
            }
        }

        return conn;
    }

    /**
     * Get the pool (list) of connections available for the given host and port
     *
     * @param hostAndPort the key for the connection pool
     * @return a pool (list) of connections available for the given key
     */
    private LinkedList getConnections(String hostAndPort) {
        log.trace("enter HttpConnectionManager.getConnections(String)");

        // Look for a list of connections for the given host:port
        LinkedList listConnections = null;
        synchronized (mapHosts) {
            listConnections = (LinkedList) mapHosts.get(hostAndPort);
            if (listConnections == null) {
                // First time for this host:port
                listConnections = new LinkedList();
                mapHosts.put(hostAndPort, listConnections);
                mapNumConnections.put(hostAndPort, new Integer(0));
            }
        }
        return listConnections;
    }
    
    /**
     * Get the number of connections in use for the key
     *
     * @param hostAndPort the key that connections are tracked on
     * @return the number of connections in use for the given key
     */
    public Integer getConnectionsInUse(String hostAndPort) {
        log.trace("enter HttpConnectionManager.getConnectionsInUse(String)");
        // FIXME: Shouldn't this be synchronized on mapNumConnections? or
        //        mapHosts?

        Integer numConnections = (Integer)mapNumConnections.get(hostAndPort);
        if (numConnections == null) {
            log.error("HttpConnectionManager.getConnection:  "
                + "No connection count for " + hostAndPort);
            // This should never happen, but just in case we'll try to recover.
            numConnections = new Integer(0);
            mapNumConnections.put(hostAndPort, numConnections);
        }
        return numConnections;
    }
    
    /**
     * wait for a connection from the pool
     */
    private HttpConnection waitForConnection(LinkedList pool, long timeout)
    throws HttpException {
        log.trace("enter HttpConnectionManager.waitForConnection(LinkedList, long)");

        // No connections available, so wait
        // Start the timeout thread
        TimeoutThread threadTimeout = new TimeoutThread();
        threadTimeout.setTimeout(timeout);
        threadTimeout.setWakeupThread(Thread.currentThread());
        threadTimeout.start();

        HttpConnection conn = null;
        // wait for the connection to be available
        while(conn == null){    // spin lock
            try {
                log.debug("HttpConnectionManager.getConnection:  waiting for "
                    + "connection from " + pool);
                pool.wait();
            } catch (InterruptedException e) {
                throw new HttpException("Timeout waiting for connection.");
            }
            if (pool.size() > 0) {
                conn = (HttpConnection)pool.removeFirst();
                threadTimeout.interrupt();
            }
        }
        return conn;
    }
    
    /**
     * Make the given HttpConnection available for use by other requests.
     * If another thread is blocked in getConnection() waiting for a connection
     * for this host:port, they will be woken up.
     * 
     * @param conn - The HttpConnection to make available.
     */
    public void releaseConnection(HttpConnection conn) {
        log.trace("enter HttpConnectionManager.releaseConnection(HttpConnection)");

        String host = conn.getHost();
        int port = conn.getPort();
        String key = host + ":" + port;

        if(log.isDebugEnabled()){
            log.debug("HttpConnectionManager.releaseConnection:  Release connection for " + host + ":" + port);
        }

        LinkedList listConnections = null;
        synchronized(mapHosts){
            listConnections = (LinkedList)mapHosts.get(key);
            if(listConnections == null){
                // This is an error, but we'll try to recover
                log.error("HttpConnectionManager.releaseConnection:  No connect list for " + key);
                listConnections = new LinkedList();
                mapHosts.put(key, listConnections);
                mapNumConnections.put(key, new Integer(1));
            }
        }

        synchronized(listConnections){
            // Put the connect back in the available list and notify a waiter
            listConnections.addFirst(conn);
            listConnections.notify();
        }
    }

    /**
     * In getConnection, if the maximum number of connections has already
     * been reached the call will block.  This class is used to help provide
     * a timeout facility for this wait.  Because Java does not provide a way to
     * determine if wait() returned due to a notify() or a timeout, we need
     * an outside mechanism to interrupt the waiting thread after the specified
     * timeout interval.
     */
    private static class TimeoutThread extends Thread {

        private long timeout = 0;
        private Thread thrdWakeup = null;

        public void setTimeout(long timeout)
        {
            this.timeout = timeout;
        }

        public long getTimeout()
        {
            return timeout;
        }

        public void setWakeupThread(Thread thrdWakeup)
        {
            this.thrdWakeup = thrdWakeup;
        }

        public Thread getWakeupThread()
        {
            return thrdWakeup;
        }

        public void run() {
	    log.trace("TimeoutThread.run()");
            if(timeout == 0){
                return;
            }
            if(thrdWakeup == null){
                return;
            }

            try{
                sleep(timeout);
                thrdWakeup.interrupt();
            }catch(InterruptedException e){
	        log.debug("InterruptedException caught as expected");
                // This is expected
            }
        }
    }
}
