/*
 * Copyright (C) The Apache Software Foundation. All rights reserved.
 *
 * This software is published under the terms of the Apache Software License
 * version 1.1, a copy of which has been included with this distribution in
 * the LICENSE.txt file.
 */
package org.apache.avalon.excalibur.source;

/**
 * Describes a {@link Source} object whose data content can change.
 * The overall handling of this object is very similar to the handling
 * of the source object: using the getInputStream() method you get
 * always the upto-date content. When you're done with using the
 * source object, you have to release it.
 * If you want to track changes of the source object, this interface
 * offers you some support for it by providing a SourceValidity object.
 *
 * How does the caching work?
 * The first time you get a ModifiableSource object, you simply ask
 * it for it's content via getInputStream() and then get the validity
 * object by invoking getValidity. (Further calls to getValidity always
 * return the same object! This is not updated!)
 * The caching algorithm can now store this validity object together
 * with the system identifier of the source.
 * The next time, the caching algorithm wants to check if the cached
 * content is still valid. It has a validity object already to check
 * against.
 * If it is still the same ModifiableSource than the first time, you
 * have to call discardValidity() in order to discard the stored validity
 * in the ModifiableSource object. If it is a new ModifiableSource object,
 * calling discardValidity() should do no harm.
 * After that an upto-date validity object can retrieved by calling
 * getValidity(). This can be used to test if the content is still valid
 * as discribed in the source validity documentation.
 * If the content is still valid, the cache knows what to do, if not,
 * the new content can be get using getInputStream().
 * So either after a call to getValidity() or the getInputStream the
 * validity object must be the same until discardValidity is called!
 *
 * @author <a href="mailto:ovidiu@cup.hp.com">Ovidiu Predescu</a>
 * @author <a href="mailto:cziegeler@apache.org">Carsten Ziegeler</a>
 */
public interface ModifiableSource
  extends Source
{
    /**
     *  Get the Validity object. This can either wrap the last modification
     *  date or the expires information or...
     *  If it is currently not possible to calculate such an information
     *  <code>null</code> is returned.
     */
    SourceValidity getValidity();

    /**
     * Refresh the content of this object after the underlying data
     * content has changed.
     */
    void discardValidity();
}
