/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                   Author :  Alan W Black                              */
/*                   Date   :  March 1998                                */
/*-----------------------------------------------------------------------*/
/*                EST_Utterance i/o functions                            */
/*                                                                       */
/*=======================================================================*/
#include <stdlib.h>
#include <stdio.h>
#include <iostream.h>
#include <fstream.h>
#include "EST_string_aux.h"
#include "EST_FileType.h"
#include "EST_Token.h"
#include "ling_class/EST_Utterance.h"

static EST_read_status load_est_ascii(EST_TokenStream &ts,EST_Utterance &u);
static EST_read_status load_all_contents(EST_TokenStream &ts,
					 KVL<int,EST_Item_Content *> &sitems);
static EST_read_status load_relations(EST_TokenStream &ts,
				      EST_Utterance &utt,
				      KVL<int,EST_Item_Content *> &sitems);
static EST_write_status save_est_ascii(ostream &outf,const EST_Utterance &utt);
static EST_write_status utt_save_all_contents(ostream &outf,
					      const EST_Utterance &utt,
					      KVL<void *,int> &sinames);
static EST_write_status utt_save_all_contents(ostream &outf,
					      EST_Item *n, 
					      KVL<void *,int> &sinames,
					      int &si_count);
static EST_write_status utt_save_ling_content(ostream &outf,
					      EST_Item *si,
					      KVL<void *,int> &sinames,
					      int &si_count);

EST_read_status EST_Utterance::load(const EST_String &filename)
{   
    EST_TokenStream ts;
    EST_read_status v=format_ok;
    
    if (((filename == "-") ? ts.open(cin) : ts.open(filename)) != 0)
    {
	cerr << "load_utt: can't open utterance input file " 
	    << filename << endl;
	return misc_read_error;
    }
    // set up the character constant values for this stream
    ts.set_SingleCharSymbols(";");
    ts.set_quotes('"','\\');
    f.set("filename", filename);

    v = load(ts);

    ts.close();

    return v;
}

EST_read_status EST_Utterance::load(EST_TokenStream &ts)
{
    EST_read_status v=format_ok;

    init();  // we're committed to reading something so clear utterance
    
    if ((v = load_est_ascii(ts,*this)) == wrong_format)
    {   
	cerr << "utt.load: unknown utterance file type " << 
	    ts.pos_description() << endl;
	v = wrong_format;
    }

    return v;
}

static EST_read_status load_est_ascii(EST_TokenStream &ts,
				      EST_Utterance &u)
{
    EST_Option hinfo;
    bool ascii;
    EST_EstFileType t;
    EST_read_status r;
    KVL<int,EST_Item_Content *> sitems;

    if ((r = read_est_header(ts, hinfo, ascii, t)) != format_ok)
	return r;
    if (t != est_file_utterance)
	return misc_read_error;
    if (hinfo.ival("version") != 2)
    {
	cerr << "utt_load: " << ts.pos_description() <<
	    " wrong version of utterance format expected 2 but found " <<
		hinfo.ival("version") << endl;
	return misc_read_error;
    }

    // Utterance features
    if (ts.get() != "Features")
    {
	cerr << "utt_load: " << ts.pos_description() <<
	    " missing utterance features section" << endl;
	return misc_read_error;
    }
    else
	u.f.load(ts);
    // items
    if (ts.get() != "Stream_Items")
    {
	cerr << "utt_load: " << ts.pos_description() <<
	    " missing Items section" << endl;
	return misc_read_error;
    }
    r = load_all_contents(ts,sitems);

    // Only exist in older form utterances so sonn wont be necessary
    if (ts.peek() == "Streams")
    {
	cerr << "utt.load: streams found in utterance file, " <<
	    "no longer supported" << endl;
	return misc_read_error;
    }

    // Relations
    if ((r == format_ok) && (ts.get() != "Relations"))
    {
	cerr << "utt_load: " << ts.pos_description() <<
	    " missing Relations section" << endl;
	return misc_read_error;
    }
    r = load_relations(ts,u,sitems);

    if ((r == format_ok) && (ts.get() != "End_of_Utterance"))
    {
	cerr << "utt_load: " << ts.pos_description() <<
	    " End_of_Utterance expected but not found" << endl;
	return misc_read_error;
    }

    if (r != format_ok)
    {
	// This works because even if some of these si's have been
	// linked to nodes they will be unlink when the si is destroyed
	for (EST_Litem *p = sitems.list.head(); p != 0; p=next(p))
	    delete sitems.list(p).v;
    }

    return format_ok;

}

static EST_read_status load_all_contents(EST_TokenStream &ts,
					 KVL<int,EST_Item_Content *> &sitems)
					 
{
    // Load items into table with names for later reference
    // by relations
    while (ts.peek() != "End_of_Stream_Items")
    {
	EST_Item_Content *si = new EST_Item_Content;
	int id;
	
	if (ts.peek().string().matches(RXint))
	    id = atoi(ts.get().string());
	else
	{
	    cerr << "utt_load: " << ts.pos_description() << 
		" Item name not a number: " << ts.peek().string() <<
		    endl;
	    return misc_read_error;
	}
	sitems.add_item(id,si);
	if (si->f.load(ts) != format_ok)
	    return misc_read_error;
	if (ts.eof())
	    return misc_read_error;  // just in case this happens
    }

    ts.get(); // skip "End_of_Stream_Items"

    return format_ok;
}

static EST_read_status load_relations(EST_TokenStream &ts,
				      EST_Utterance &utt,
				      KVL<int,EST_Item_Content *> &sitems)
{
    // Load relations

    while (ts.peek() != "End_of_Relations")
    {
	// can't use create relation as we don't know its name until
	// after its loaded
	EST_Relation *r = new EST_Relation;

	if (r->load(ts,sitems) != format_ok)
	    return misc_read_error;

	utt.relations.add_item(r->name(),r);
	r->set_utt(&utt);

	if (ts.eof())
	    return misc_read_error;
    }

    ts.get();  // Skip "End_of_Relations"

    return format_ok;
}
	
EST_write_status EST_Utterance::save(const EST_String &filename,
				     const EST_String &type) const
{
    EST_write_status v;
    ostream *outf;
    
    if (filename == "-")
	outf = &cout;
    else
	outf = new ofstream(filename);
    
    if (!(*outf))
	return write_fail;

    v = save(*outf,type);

    if (outf != &cout)
	delete outf;

    return v;
}

EST_write_status EST_Utterance::save(ostream &outf,
				     const EST_String &type) const
{
    if (type == "est_ascii")
	return save_est_ascii(outf,*this);
    else
    {
	cerr << "EST_Utterance::save unknown type " << type << endl;
	return write_fail;
    }
}

static EST_write_status save_est_ascii(ostream &outf,const EST_Utterance &utt)
{
    EST_Litem *r;
    EST_write_status v = write_ok;
    
    outf.precision(8);
    outf.setf(ios::fixed, ios::floatfield);
    outf.width(8);
    
    outf << "EST_File utterance\n"; // EST header identifier.
    outf << "DataType ascii\n";
    outf << "version 2\n";
    outf << "EST_Header_End\n"; // EST end of header identifier.

    // Utterance features
    outf << "Features ";
    utt.f.save(outf);
    outf << endl;

    outf << "Stream_Items\n";
    KVL<void *,int> sinames;
    v = utt_save_all_contents(outf,utt,sinames);
    if (v == write_fail) return v;
    outf << "End_of_Stream_Items\n";

    // Relations
    outf << "Relations\n";
    for (r = utt.relations.list.head(); r != 0; r=next(r))
    {
	v = utt.relations.list(r).v->save(outf,sinames);
	if (v == write_fail) return v;
    }
    outf << "End_of_Relations\n";

    outf << "End_of_Utterance\n";
    return write_ok;
}

static EST_write_status utt_save_all_contents(ostream &outf,
					      const EST_Utterance &utt,
					      KVL<void *,int> &sinames)
{
    // Write out all stream items in the utterance, as they mayu appear in
    // various places in an utterance keep a record of which ones
    // have been printed and related them to names for reference by
    // the Relations (and older Stream architecture).
    int si_count = 1;
    EST_write_status v = write_ok;

    // Find the stream items in the relations
    for (EST_Litem *r = utt.relations.list.head(); r != 0; r=next(r))
    {
	v = utt_save_all_contents(outf,utt.relations.list(r).v->head(),
				  sinames,si_count);
	if (v == write_fail) return v;
    }

    return v;
}

static EST_write_status utt_save_all_contents(ostream &outf,
					      EST_Item *n, 
					      KVL<void *,int> &sinames,
					      int &si_count)
{
    if (n == 0)
	return write_ok;
    else 
    {
	utt_save_ling_content(outf,n,sinames,si_count);
	// As we have more complex structures this will need to
	// be updated (i.e. we'll need a marking method for nodes)
	utt_save_all_contents(outf,next(n),sinames,si_count);
	utt_save_all_contents(outf,down(n),sinames,si_count);
    }
    return write_ok;
}

static EST_write_status utt_save_ling_content(ostream &outf,
					      EST_Item *si,
					      KVL<void *,int> &sinames,
					      int &si_count)
{
    // Save item and features if not already saved
					     
    if ((si != 0) && (!sinames.present(si->contents())))
    {
	sinames.add_item(si->contents(),si_count);
	outf << si_count << " ";
	si->features().save(outf);
	outf << endl;
	si_count++;
    }
    return write_ok;
}

