/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                     Copyright (c) 1995,1996                           */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/

#ifndef __EST_SIG2FV_H__
#define __EST_SIG2FV_H__

#include "EST_Option.h"
#include "sigpr/EST_sigpr_utt.h"


/**@name  High-Level Functions for Feature Vector Building 

This part of the library contains a series of high level functions
which construct feature vectors from waveforms. In the
\Ref{Utterance-based signal processing functions.} section, there are a
number of functions defined such as \Ref{sig2lpc}, {\tt sig2cep},
which take a waveform and produce a set of coefficients of the
required type. The functions described here provide an easy to use
wrap-around for the signal processing functions, and are particularly
useful for cases where a variety of different coefficient types is
required in the same Track.

* @see Sig2fv:example
* @see Signal_processing_example_code
*/
//@{

/** Generate base, delta and acc feature vectors.

This functions processes {\tt sig} and fills the track {\tt fv}
with the required feature vectors. The required types are specified in
the sring list {\tt base_list} and should be a selection of the
currently available legal types\footnote{This is sometging.}. In
addition to this, delta and acceleration (delta delta) coefficients
may be specified in the string lists {\tt delta_list} and {\tt
acc_list}.

This function (unlike most others) calculates the number of channels
required, and resizes the {\tt fv} accordingly. As with virtually all
other signal processing functions, the time positions of the analysis
frames should exist already and by given in the time array of the {\tt fv}.

For example usage, see \Ref{Producing_a_single_type_of_feature_vector_for_an_utterance.}



@param  sig is the waveform to be processed
@param  fv is where the features will be written to
@param  op contains the options regarding pitch tracking parameters
@param  base_list a list of basic signal processing operations, eg. lpc
@param  delta_list a list of names of delta coefficients to be made
@param  acc_list a list of names of acceleration coefficients to be made
@see sig2fv
@see sig2fv_fixed


*/
void sig2fv(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	    const EST_StrList &base_list, 
	    const EST_StrList &delta_list,
	    const EST_StrList &acc_list);

inline void sig2fv(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	    const EST_StrList &base_list, 
	    const EST_StrList &delta_list)
	{ const EST_StrList empty; sig2fv(sig ,fv, op, base_list, delta_list, empty); }

inline void sig2fv(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	    const EST_StrList &base_list)
	{ const EST_StrList empty; sig2fv(sig ,fv, op, base_list, empty, empty); }

/**  Generate base, delta and acc feature vectors at fixed intervals.

This is a wrap-around for \Ref{sig2fv} which operates on an empty
track ({\tt fv}. This function fills the time array of {\tt fv} at
{\tt shift} intervals and uses analysis windows of length {\tt
length} (both specified in seconds).

@param  sig the waveform to be processed
@param  fv where the features will be written to
@param  op the options regarding pitch tracking parameters
@param  shift frame spacing in seconds
@param  length frame length in seconds
@param  base_list a list of basic signal processing operations, eg. lpc
@param  delta_list a list of names of delta coefficients to be made
@param  acc_list a list of names of acceleration coefficients to be made
*/

void sig2fv_fixed(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		  float shift, float length,
		  EST_StrList &base_list, 
		  const EST_StrList &delta_list,
		  const EST_StrList &acc_list);

inline void sig2fv_fixed(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		  float shift, float length,
		  EST_StrList &base_list, 
		  const EST_StrList &delta_list)
	{ const EST_StrList empty; sig2fv_fixed(sig, fv, op, shift, length, base_list, delta_list, empty); }

inline void sig2fv_fixed(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		  float shift, float length,
		  EST_StrList &base_list)
	{ const EST_StrList empty; sig2fv_fixed(sig, fv, op, shift, length, base_list, empty, empty); }

/** Generate a set of base feature vectors.

Generate a set of feature vectors specified by the entries of {\tt
slist} and place them in the pre-allocated track {\tt fv}. This
function basically calls \Ref{sigpr_track} for the number of entries
in the list.

For example usage, see \Ref{Producing_a_single_type_of_feature_vector_for_an_utterance.}

@param  sig the waveform to be processed
@param  fv where the features will be written to
@param  op the options regarding pitch tracking parameters
@param  slist a list of names of the types of processing required

@see sig2coef
@see sig2fv

*/
void sigpr_base(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		 const EST_StrList &slist);

/** Generate a delta feature vector.

This functions generates a set of delta coefficients of type {\tt
name} and places them in the pre-allocated track {\tt fill}. If the
base types of the named coefficients is present in {\tt fv} this is
used, otherwise the base types are generated on the fly and discarded.
*/


void sigpr_delta_track(EST_Wave &sig, EST_Track &fill, const EST_String &name,
		       EST_Option &op, int order = 0, 
		       const EST_Track &fv = EST_Track());

/** Generate a set of delta feature vectors.

Generate a set of delta feature vectors specified by the entries of
{\tt slist} and place them in the pre-allocated track {\tt fv}. This
function operates by calling \Ref{sigpr_delta_track} for each entry in
the list.

For example code see \Ref{Producing_delta_and_acceleration_coefficients.}

@see delta
@see sigpr_acc
@see sigpr_delta_track
@see sig2fv
*/

void sigpr_delta(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
		 const EST_StrList &slist);


/** Generate a set of acceleration feature vectors.

Generate a set of acceleration feature vectors specified by the
entries of {\tt slist} and place them in the pre-allocated track {\tt
fv}. This function operates by calling \Ref{sigpr_delta_track} twice
for each entry in the list.

For example code see \Ref{Producing_delta_and_acceleration_coefficients.}

@see delta
@see sigpr_delta
@see sigpr_delta_track
@see sig2fv
*/

void sigpr_acc(EST_Wave &sig, EST_Track &fv, EST_Option &op, 
	       const EST_StrList &slist);

//@}


    
/**@name Functions for naming, and allocating space in tracks.

*/
//@{


/** Fill time array at {\tt shift} seconds intervals.

*/
void make_track(EST_Track &fv, const EST_String &name, 
		int coef_order, int del_order = 0);


/** Fill time values.
*/
void add_to_track_map(const EST_String &name, EST_TrackMap &map, 
			   int coef_order, int delta_order, int &c);

/** Fill time values.
*/
void add_to_track_map(const EST_StrList &slist, EST_TrackMap &map, 
			   EST_Option &op, int delta_num, int &c);


/** Fill time values.
*/
inline void add_to_track_map(const EST_StrList &slist, EST_TrackMap &map, 
			   EST_Option &op, int delta_num = 0)
{
    int c = 0;
    add_to_track_map(slist, map, op, delta_num, c);
}

/** Fill time values.
*/
void make_fixed_times(EST_Track &pm, EST_TrackMap &map, 
		      float end, float shift);


/** Fill time values.
*/
void make_fixed_times(EST_Track &pm, float end, float shift);


/** Allocate space in a track and name the channels.

Firstly, this function calculates how many frames are need by using the
{\tt end} and {\tt shift} arguments. Using this, and {\tt coef_order},
an appropriately sized track is created. A \Ref{EST_TrackMap} is then
created for the track using {\tt name}, {\tt coef_order} and
{\tt delta_order}.
*/

void make_fixed_timed_track(EST_Track &fv, const EST_String &name, 
			    float end, int coef_order, 
			    float shift, int del_order = 0);

/** Copy time points from a track, allocate space and name the channels.

The number and position of the frames are calculated from the
{\tt main} track.
*/

void make_timed_track(const EST_Track &main, EST_Track &fv, 
		      const EST_String &name, int coef_order, int del_order);

//@}


#endif /* __EST_SIG2FV_H__ */

