/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                     Copyright (c) 1995,1996                           */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/

/**@name EST_lpc.h
  * Edinburgh Speech Tools signal processing functions.
  * <p>
  * @author Paul Taylor<pault@cstr.ed.ac.uk>
  * @version $Id: EST_lpc.h,v 1.1 1998/08/12 09:29:25 awb Exp $
  */
//@{   

#ifndef __EST_LPC_H__
#define __EST_LPC_H__

#include "EST_Wave.h"
#include "EST_Track.h"
#include "EST_FMatrix.h"
#include "EST_Option.h"
#include "EST_TNamedEnum.h"
#include "EST_TBuffer.h"
#include "sigpr/EST_Window.h"

/**@name Cepstrum functions. */
//@{
EST_Track make_cepstra_unnorm(EST_Wave &sig, EST_Option &op);
EST_Track make_cepstra(EST_Wave &sig, EST_Option &op);
int init_cepstra(EST_Option &op, EST_String bc_file);
//@}


/**@name LPC analysis and synthesis */
//@{

/// Pre-emphasis to correct for spectral tilt.
void lpc_pre_emphasis(EST_Wave &signal, EST_Wave &psignal, float a);
/// De-emphasis for use after synthesis.
void lpc_post_deemphasis(EST_Wave &signal, EST_Wave &dsignal, float a);

/// Pre-emphasis to correct for spectral tilt.
void lpc_pre_emphasis(EST_Wave &signal, float a);
/// De-emphasis for use after synthesis.
void lpc_post_deemphasis(EST_Wave &signal, float a);

/// A typical description of pitchmarks, contains length and offset channels.
extern EST_TrackMap EST_PMTrackMap;


/**@name Pitchmarks */
//@{


float get_time_pitch_period(EST_Track &pms, int i, 
			    int prefer_prev=0);

int get_pitch_period(EST_Track &pms, int i, int sample_rate, 
		     int prefer_prev=0);

/// make a set of evenly spaced pitchmarks.
void make_fixed_frame_pitchmarks(EST_Track &pms,
				 EST_Wave &sig,
				 float shift);

/// Take pitchmarks and add frame size and offset
void make_offset_pitchmarks(EST_Track &pms,
			    EST_Wave &sig,
			    int offset);

/// Take pitchmarks and add frames size and offset -- offset a percentage of a frame
void make_percent_offset_pitchmarks(EST_Track &pms,
				    EST_Wave &sig,
				    int percent_offset);

/** Adjust the pitchmarks so that the whole sequence of times, offsets and lengths
  * represents a sequence of analysis frames which meet each other.
  */
void pm_frames_must_meet(EST_Track &pms, EST_Wave &sig,
			 int (*find_pos)(int near, EST_Wave &sig));
//@}

/**@name LPC analysis functions
  * Analysis is controled by the description.
  */
//@{
/// Version with optional pitchmarks.
void lpc_analyse(EST_Track &lpct, 
		 EST_Wave &sig, 
		 EST_WindowFunc *make_window,
		 EST_Track &pms,
		 float factor,
		 bool do_energy,
		 EST_TrackMap *description=NULL
		 );
inline void lpc_analyse(EST_Track &lpct, 
		 EST_Wave &sig, 
		 EST_WindowFunc *make_window,
		 EST_Track &pms,
		 float factor,
		 bool de,
		 EST_TrackMap &description
		 )
	{ lpc_analyse(lpct, sig, make_window, pms, factor, de, &description); }
//@}

/**@name Misc Analysis Functions.
  * Further analysis for pitchmarked signals.
  * They use the pitchmarks to work out which parts of the
  * signal are associated with each frame.
  */
//@{
/// Inverse filter a signal to get a residual.
void lpc_residual(EST_Wave &residual, 
		  EST_Track &lpcs,  EST_Track &pms, 
		  EST_Wave &sig);

/// Calculate the RMS power of each frame.
void fill_power_channel(EST_Track &lpcs,  EST_Track &pms, 
			EST_Wave &sig);

/// Make voicing decisionfor each frame.
void fill_voice_channel(EST_Track &lpcs,  EST_Track &pms, 
			EST_Wave &sig);
//@}

/**@name LPC synthesis functions */
//@{

/// Basic resynthesis function
void lpc_resynth(EST_Wave &signal, const EST_Track &lpcs, 
		 const EST_Wave &residual, const EST_Track *modifications=NULL, 
		 EST_PpModFunc* pp_mod=NULL, EST_PpModFunc *pp_mod_u=NULL,
		 short *res_return = NULL);

/// Allow modifications to be passed in as reference
inline void lpc_resynth(EST_Wave &signal, 
			const EST_Track &lpcs, 
			const EST_Wave &residual, 
			const EST_Track &modifications, 
			EST_PpModFunc *pp_mod=NULL, 
			EST_PpModFunc *pp_mod_u=NULL,
			short *res_return = NULL)
	{ lpc_resynth(signal, lpcs, residual, &modifications, 
		      pp_mod, pp_mod_u, res_return); }

/// Inner function which synthesises a small section of speech.
int lpc_resynth_chunk(short *sig,
		       const EST_Track &lpcs,
		       const short *res,
		       int res_num_samples,
		       int sample_rate,
		       int num_samples, 
		       int start_frame,
		       int num_frames, 
		       int order,	       
		       const EST_Track *modifications=NULL,
		       EST_PpModFunc *pp_mod=NULL,
		       EST_PpModFunc *pp_mod_u=NULL,
		       short *res_return=NULL);

/// Allow modifications to be passed in as reference
inline int lpc_resynth_chunk(short *sig,
			      const EST_Track &lpcs,
			      const short *res,
			      int res_num_samples,
			      int sample_rate,
			      int num_samples, 
			      int start_frame,
			      int num_frames, 
			      int order,
			      const EST_Track &modifications,
			      EST_PpModFunc *pp_mod=NULL,
			      EST_PpModFunc *pp_mod_u=NULL,
			      short *res_return=NULL)
	{ return lpc_resynth_chunk(sig, lpcs, res, res_num_samples, sample_rate,
			    num_samples, start_frame, num_frames, order, 
			    &modifications, pp_mod, pp_mod_u, res_return); }


/// Basic resynthesis function, two pass version.
void lpc_resynth_split(EST_Wave &signal, 
		       const EST_Track &lpcs, 
		       const EST_Wave &residual, 
		       const EST_Track *modifications=NULL, 
		       EST_PpModFunc* pp_mod=NULL, 
		       EST_PpModFunc *pp_mod_u=NULL,
		       short *res_return = NULL);

/// Allow modifications to be passed in as reference
inline void lpc_resynth_split(EST_Wave &signal,
			      const EST_Track &lpcs, 
			      const EST_Wave &residual,
			      const EST_Track &modifications, 
			      EST_PpModFunc *pp_mod=NULL, 
			      EST_PpModFunc *pp_mod_u=NULL,
			      short *res_return = NULL)
	{ lpc_resynth_split(signal, lpcs, residual, &modifications, 
			    pp_mod, pp_mod_u, res_return); }

/// Inner function which synthesises a small section of speech, two pass version.
int lpc_resynth_chunk_split(short *sig,
			     const EST_Track &lpcs,
			     const short *res,
			     int res_num_samples,
			     int sample_rate,
			     int num_samples, 
			     int start_frame,
			     int num_frames, 
			     int order,	       
			     const EST_Track *modifications=NULL,
			     EST_PpModFunc *pp_mod=NULL,
			     EST_PpModFunc *pp_mod_u=NULL,
			     short *res_return=NULL);

/// Allow modifications to be passed in as reference
inline int lpc_resynth_chunk_split(short *sig,
			      const EST_Track &lpcs,
			      const short *res,
			      int res_num_samples,
			      int sample_rate,
			      int num_samples, 
			      int start_frame,
			      int num_frames, int order,
			      const EST_Track &modifications,
			      EST_PpModFunc *pp_mod=NULL,
			      EST_PpModFunc *pp_mod_u=NULL,
			      short *res_return=NULL)
	{ return lpc_resynth_chunk_split(sig, lpcs, res, res_num_samples, sample_rate,
				  num_samples, start_frame, num_frames, order, 
				  &modifications, pp_mod, pp_mod_u, res_return); }

//@}
//@}

/**@name Convertion procedures. */
//@{

void ref2truearea(const float *ref, float *area, int order);
void ref2logarea(const float *ref, float *logarea, int order);
void ref2area(const float *ref, float *area, int order);
void lpc2ceps(const float *lpc, int nlpc, float *ceps, int nceps);
void lpc2ref(const float *lpc, float *ref, int order);
void ref2lpc(const float *ref, float *lpc, int order);

void lpc_to_reflection(EST_Track &lpcs);

/// Convert a track of LPC coefficients to line spectral frequencies.
void lpc_to_lsf(EST_Track &lpcs);
//@}

/**@name Depreciated */
//@{

/**@name Old LPC code */
//@{
EST_Wave make_lpc_residual(EST_Wave &sig, EST_Track &lpc, EST_Option &op);

void lpc_cepstra(EST_Track &c, EST_Wave &sig, float length, float
		 shift, int lporder, int genlpc = 0);

//@}

//@}

#endif /* __EST_LPC_H__ */


//@}
