/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1995 (redone May 1998)               */
/*-----------------------------------------------------------------------*/
/*                  EST_Utterance Class header file                      */
/*                                                                       */
/*=======================================================================*/
#ifndef __Utterance_H__
#define __Utterance_H__

#include "EST_String.h"
#include "EST_TList.h"
#include "ling_class/EST_Relation.h"
#include "ling_class/EST_Item.h"
#include "EST_Features.h"

/** A class that contains \Ref{EST_Item}s and \Ref{EST_Relation}s
between them.  Used for holding interrelated linguistic structures.

\Ref{EST_Item}s may be used to represent phones, words etc.
\Ref{EST_Relation}s which may be lists or trees, may be defined over
these \Ref{EST_Item}s.  Relations may used to represent
tree structures like syntax and syllable structure as well as more
simple lists of words, segments etc.

*/

class EST_Utterance{
private:
    void copy(const EST_Utterance &u);
public:
    /**@name Constructor and initialisation Functions */
    //@{
    /// default constructor
    EST_Utterance();
    EST_Utterance(const EST_Utterance &u) { copy(u); }
    ~EST_Utterance() {clear();}
    ///
    void init();
    /// remove everything in utterance
    void clear();

    /// clear the contents of the relations only
    void clear_relations();
    //@}

    /**@name File i/o */
    //@{
    /** load an ascii file containing an Utterance
      */
    EST_read_status load(const EST_String &filename);
    EST_read_status load(EST_TokenStream &ts);

    /** save an ascii file containing an Utterance
      */
    EST_write_status save(const EST_String &filename,
			  const EST_String &type="est_ascii") const;
    EST_write_status save(ostream &outf,const EST_String &type) const;

    EST_Utterance &operator=(const EST_Utterance &s);
    friend ostream& operator << (ostream &s, const EST_Utterance &u);
    EST_Relation * operator() (const EST_String &name)
    { return relation(name);}
    //@}  


    /// Utterance level features
    EST_Features f;
    
    /// The list of named relations over stream items
    KVL <EST_String, EST_Relation *> relations;

    // possible public interface (by RJC)
    int num_relations() const { return relations.length(); }

    /** returns true if utterance contains named relations.
      {\bf name} can be either a single string or a bracketed list
      of strings e.g. "(Word Phone Syl)".  */
    bool has_relation(const EST_String name) const;

    /** returns true if utterance contains all the relations
      named in the list {\bf name}. */
    bool has_relation(EST_StrList &names) const;

    /// get nth relation 
    EST_Relation *relation(int i); 
    /// get relation by name
    EST_Relation *relation(const EST_String &name,int err_on_not_found=1);
    EST_Relation *relation(const char *name,int err_on_not_found=1);

    EST_Relation *create_relation(const EST_String &n);
    void delete_relation(const EST_String &n);
		
    void sub_utterance(EST_Item *i);
};

int utterance_merge(EST_Utterance &utt,
		    EST_Utterance &sub_utt,
		    EST_Item *at,
		    const EST_String &relname);
void sub_utterance(EST_Utterance &sub,EST_Item *i);

#endif
