/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                      Copyright (c) 1995,1996                          */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                    Author :  Paul Taylor                              */
/*                    Date   :  May 1995                                 */
/*-----------------------------------------------------------------------*/
/*                   Stream Class header file                            */
/*                                                                       */
/*   This is to be deleted soon to be replaced with general relations    */
/*                                                                       */
/*=======================================================================*/
#ifndef __EST_STREAM_H__
#define __EST_STREAM_H__

#include "EST_common.h"
#include "EST_String.h"
#include "EST_TList.h"
#include "EST_KV.h"
#include "EST_Val.h"
#include "EST_types.h"
#include "EST_Features.h"

class EST_Stream;

#include "ling_class/EST_Stream_Item.h"
#include "ling_class/EST_Relation.h"  // new architecture

/** Stream class for storing a list of linguistic objects. This class
contains a list of {\tt Stream_Items} which represent individual
linguistic objects such as words or phones etc. Each Stream is
identified by a stream name, which gives the type of the linguistic
units it holds.

Note that this is not a derived class of EST_TList, becuase mechanism
of the list is necessarily different: specifically, the items in the
list need to be able to see each other.

Iteration is accomplished by use of a pointer of type EST_Stream_Item:

EST_Stream_Item *p;
EST_Stream words;

for (p = words.head(); p != 0; p = next(p))
    cout << p->name();

*/

class EST_Stream {
    EST_Stream_Item *h;		//head of list
    EST_Stream_Item *t;		//tail of list
    int next_addr;		// next unused "address" tag.
    EST_String p_stream_name;   // typically "phoneme" etc
    EST_String p_name;          // unique name, often filename

    void default_vals();
    void copy(const EST_Stream &s);    
    EST_Stream_Item *prepend_item(EST_Stream_Item *item);
    EST_Stream_Item *append_item(EST_Stream_Item *item);
public:
    /**@name Constructors and Destructors*/
    //@{
    /// Initialise stream to name {\tt type}
    EST_Stream(const EST_String &type);
    /// Copy constructor.
    EST_Stream(const EST_Stream &s);
    /// Default constructor
    EST_Stream();
    ~EST_Stream() { clear(); }
    //@}

    EST_Features f;

    /**@name Initialisation functions*/
    //@{
    /// Initialise a stream to a type.
    void init(const EST_String &type = "blank");
    //@}

    /**@name File i/o functions*/
    //@{

    /** load in stream. It is difficult to determine the file format
      of most label files automatically, and hence this function is
      foxed to read ESPS/xlabel style label files only.
      @return EST_read_status
      @param filename is the name of the file to be read
      */
    EST_read_status load(const EST_String &filename);

    /** load in stream of specified file type. 
      @return EST_read_status
      @param filename is the name of the file to be read
      @param type is the format of the file, which can be ascii, timit, htk, words, ogi or esps.
      */
    EST_read_status load_type(const EST_String &filename, const EST_String &type = "", int literal = 1);
    /** save stream of specified file type. 
      @return EST_write_status
      @param filename is the name of the file to be read
      @param type is the format of the file, which can be htk, esps, spn or names
      */
    EST_write_status save(const EST_String &filename,
			  const EST_String &type = "", 
			  const EST_String &features = "None") const;
    //@}

    /**name Information functions */
    //@{
    /// returns true if the stream is empty
	bool empty() const;
    /// number of items in stream
	int length() const; 
    
    /// stream name, often linguistic class, e.g. "phoneme"
	const EST_String &stream_name() const {return p_stream_name;}
    /// set stream name to {\tt a}
	void set_stream_name(const EST_String &a);
    
    /// general name, often file name
	const EST_String &name() const {return p_name;}
    /// set stream name to {\tt a}
	void set_name(const EST_String &a);

    
    
    //@}
    
    /**@name Functions for adding items to streams*/
    //@{
    /// Insert {\tt item} after the position marked by ptr
	EST_Stream_Item *insert_after(EST_Stream_Item *ptr, 
				      EST_Stream_Item &item);
    /// Insert {\tt item} before the position marked by ptr
	EST_Stream_Item *insert_before(EST_Stream_Item *ptr, 
				       EST_Stream_Item &item);
    
    /// Add {\tt item} on to the end of the stream
	EST_Stream_Item *append(const EST_Stream_Item &item);

    /// Make a new one, add it to the end and return a pointer to it
	EST_Stream_Item *append();
    

    /// Add {\tt item} on to the beginning of the stream
	EST_Stream_Item *prepend(const EST_Stream_Item &item);
    /// Make a new one adding to to the beginning of the stream
	EST_Stream_Item *prepend();
    //@}
    
    /**@name Removing from stream */
    //@{
    /// remove all items from stream
	void clear(); 
    /// remove item from stream, return pointer to previous item
	EST_Stream_Item *remove(EST_Stream_Item *ptr); 
    /// remove item of peusdo address {\tt a} from list
	void remove(int a);
    //@}
    
    /**@name Access Functions */
    //@{
    
    /** pointer to first item in list, especially useful for initialisation of
      an iteration */
    EST_Stream_Item *head() const {return h;}	
    /** pointer to last item in list, especially useful for initialisation of
      an iteration */
    EST_Stream_Item *tail() const {return t;}
    
    /** pointer to iterm indexed by address*/
    EST_Stream_Item *item(int a) const;	
    /** return nth item from stream */
    EST_Stream_Item *nth(int a) const;	
    //@}
    
    /**@name Operators */
    //@{
    /// print stream
    friend ostream& operator << (ostream &s, const EST_Stream &u);
    /// copy stream
    EST_Stream &operator=(const EST_Stream &s);
    //@}

};

typedef EST_TList<EST_Stream> EST_StreamList;

EST_read_status load_mlf(EST_String filename, EST_StreamList &l);

EST_String options_stream_filetypes(void);

#endif				//  __EST_STREAM_H__
