 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /************************************************************************/
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)            */
 /*                   Date: Fri Apr 11 1997                              */
 /************************************************************************/
 /*                                                                      */
 /* Simple frames synthesis -- just glue together LPCs and              */
 /* resynthesise the result. This is just a trivial test of the API.     */
 /*                                                                      */
 /************************************************************************/

#include "festival.h"
#include "UnitDatabase.h"
#include "FramesUnit.h"
#include "frames_lpc_concatenation.h"
#include "module_support.h"
#include "lpc_utils.h"
#include "sigpr/EST_PpMod.h"
#include "sigpr/EST_lpc.h"

static ModuleDescription description =
{
  "frames_lpc_concatenation", 1.0,
  "CSTR",
  "Richard Caley <rjc@cstr.ed.ac.uk>",
  {
    "Waveform concatenation for units derived from the",
    "FramesUnit class.",
    NULL
  },
  {
    { "ModifiedUnit",	"Units to be synthesised." },
    {NULL,NULL}
  },
  {
    { "Join",		"Information on how to join units."},
    {NULL,NULL}
  },
  {
    { "Wave",		"Synthesised waveform." },
    {NULL,NULL}
  },
  { {NULL,NULL,NULL,NULL}
  }
};

LISP frames_lpc_concatenation(LISP args)
{
    EST_Utterance *utt;

    EST_String unit_stream_name("ModifiedUnit");	
    EST_String join_stream_name("Join");
    EST_String wave_stream_name("Wave");

    EST_Relation *unit_stream=NULL, *join_stream=NULL, *wave_stream=NULL;

    unpack_module_args(args, 
		       utt, 
		       unit_stream_name, unit_stream, sat_existing,
		       join_stream_name, join_stream, sat_as_is,
		       wave_stream_name, wave_stream, sat_replace);
	
    const FramesUnit::Chunk *chunks = FramesUnit::chunk_utterance(utt, unit_stream, join_stream);

    FramesUnit::dump_chunks(cout, chunks);

    const FramesUnit::Chunk *ch;

    int num_samples=0, order= -1, sample_rate= -1;

    for(ch=chunks; ch->n != 0; ch++)
	for(int i=0; i < ch->n ; i++)
	{
	    EST_Track *lpcs = ch->bit[i].unit->lpc();
	    if (!lpcs)
		return NIL;

	    num_samples  += sum_lengths(*lpcs, 
					ch->bit[i].start_frame, 
					ch->bit[i].end_frame);

	    if (order < 0)
		order = get_order(*lpcs);
	    else if (order != get_order(*lpcs))
		cwarn << "lpcs of different order " << order << " " << get_order(*lpcs) << "\n";

	}

    short *sig = new short[num_samples+order];
    short *sig_at = sig;

    while (sig_at < sig+order)
	*(sig_at++) = 0;

    for(ch=chunks; ch->n != 0; ch++)
	for(int i=0; i < ch->n ; i++)
	{
	    EST_Track *lpcs = ch->bit[i].unit->lpc();
	    EST_Wave *res = ch->bit[i].unit->residual();
  
	    if (!res)
		return NIL;

	    if (sample_rate < 0)
		sample_rate = res->sample_rate();
	    else if (sample_rate != res->sample_rate())
		cwarn << "residuals of different sample rate " << sample_rate << " " << res->sample_rate() << "\n";

	    int frame_num_samples = sum_lengths(*lpcs, 
						ch->bit[i].start_frame, 
						ch->bit[i].end_frame);
	    
	    // Hacky copy until this either uses Wave's or we
	    // get a pointer to the shorts
	    short *data = walloc(short,res->num_samples());
	    for (int ii=0; ii < res->num_samples(); ii++)
		data[i] = res->a_no_check(i);
	    lpc_resynth_chunk(sig_at,
			      *lpcs,
			      data,
			      res->num_samples(),
			      res->sample_rate(),
			      frame_num_samples, 
			      ch->bit[i].start_frame, 
			      ch->bit[i].end_frame-ch->bit[i].start_frame,
			      order,	       
			      NULL,
			      EST_PpMod::mod_function("stretch_1_3"),
			      EST_PpMod::mod_function("chop_1_3")
			      );
	    wfree(data);

	    sig_at += frame_num_samples;
	}

    EST_Wave *result = new EST_Wave;

    result->resize(num_samples,1);
    for (int i=0; i<result->length(); i++)
	result->a_no_check(i) = sig[order+i];
    result->set_sample_rate(sample_rate);

    EST_Item *item = wave_stream->append();

    item->set_name("[waveform]");

    item->f.set("Wave", result, gc_wave);

    delete[] sig;
    return NIL;
}

void frames_lpc_concatenation_init(void)
{
  proclaim_module("frames_lpc_concatenation", &description);
  init_module_subr("frames_lpc_concatenation", frames_lpc_concatenation, &description);
}
