/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                 Authors:  Alan W Black                                */
/*                 Date   :  December 1997                               */
/*-----------------------------------------------------------------------*/
/*  Access to WFST classes                                               */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "festival.h"
#include "festivalP.h"

static LISP wfst_loaded_list = NIL;
static EST_WFST *load_wfst(const EST_String &filename);
static LISP add_wfst(const EST_String &name,EST_WFST *n);
static int tc_wfst = -1;

static EST_WFST *get_c_wfst(LISP x)
{
    if (TYPEP(x,tc_wfst))
	return (EST_WFST *)USERVAL(x);
    else
	err("wta to get_c_wfst",x);
    return NULL;  // err doesn't return but compilers don't know that
}

static LISP siod_make_wfst(EST_WFST *n)
{
    if (n==0)
	return NIL;
    else
	return siod_make_typed_cell(tc_wfst,n);
}

static LISP lisp_load_wfst(LISP name, LISP filename)
{
    EST_WFST *n;

    n = load_wfst(get_c_string(filename));
    add_wfst(get_c_string(name),n);

    return name;
}

static EST_WFST *load_wfst(const EST_String &filename)
{
    EST_WFST *n = new EST_WFST();
    if (n->load(filename) != 0)
    {
	cerr << "WFST: failed to read wfst from \"" <<
		filename << "\"" << endl;
	festival_error();
    }

    return n;
}

static LISP add_wfst(const EST_String &name,EST_WFST *n)
{
    EST_WFST *old_n;
    LISP lpair; 

    lpair = siod_assoc_str(name,wfst_loaded_list);

    if (wfst_loaded_list == NIL)
    {
	gc_protect(&wfst_loaded_list);
	tc_wfst = siod_register_user_type("WFST");
    }

    LISP wfst = siod_make_wfst(n);
    
    if (lpair == NIL)
	wfst_loaded_list = 
	    cons(cons(strintern(name),cons(wfst,NIL)),wfst_loaded_list);
    else
    {
	cwarn << "WFST: " << name << " recreated" << endl;
	old_n = get_c_wfst(car(cdr(lpair)));
	delete old_n;
	setcar(cdr(lpair),wfst);
    }
    return wfst;
}

EST_WFST *get_wfst(const EST_String &name,const EST_String &filename)
{
    //  Find ngram named name, returns NULL if none;
    LISP lpair;
    
    lpair = siod_assoc_str(name,wfst_loaded_list);

    if (lpair == NIL)
    {
	if (filename != EST_String::Empty)
	{
	    EST_WFST *n = load_wfst(filename);
	    add_wfst(name,n);
	    return n;
	}
	else
	{
	    cwarn << "WFST: no wfst named \"" << name << "\" loaded" << endl;
	    return 0;
	}
    }
    else
	return get_c_wfst(car(cdr(lpair)));
}

LISP lisp_wfst_transduce(LISP wfstname, LISP input)
{
    EST_WFST *wfst = get_wfst(get_c_string(wfstname));
    EST_StrList in,out;
    int r;

    if (consp(input))
	siod_list_to_strlist(input,in);
    else
	siod_list_to_strlist(stringexplode(get_c_string(input)),in);

    r = transduce(*wfst,in,out);

    if (r == FALSE)
	return rintern("FAILED");
    else
	return siod_strlist_to_list(out);
}

void festival_wfst_init()
{

    init_subr_2("wfst.load",lisp_load_wfst,
 "(wfst.load NAME FILENAME)\n\
  Load a WFST from FILENAME and store it named NAME for later access.");
    init_subr_2("wfst.transduce",lisp_wfst_transduce,
 "(wfst.trasduce WFSTNAME INPUT)\n\
  Transduce list INPUT (or exploded INPUT if its an atom) to a list of \n\
  outputs.  The atom FAILED is return if the transduction fails.");
    

}
