/*************************************************************************/
/*                                                                       */
/*                Centre for Speech Technology Research                  */
/*                     University of Edinburgh, UK                       */
/*                       Copyright (c) 1996,1997                         */
/*                        All Rights Reserved.                           */
/*                                                                       */
/*  Permission to use, copy, modify, distribute this software and its    */
/*  documentation for research, educational and individual use only, is  */
/*  hereby granted without fee, subject to the following conditions:     */
/*   1. The code must retain the above copyright notice, this list of    */
/*      conditions and the following disclaimer.                         */
/*   2. Any modifications must be clearly marked as such.                */
/*   3. Original authors' names are not deleted.                         */
/*  This software may not be used for commercial purposes without        */
/*  specific prior written permission from the authors.                  */
/*                                                                       */
/*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK        */
/*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING      */
/*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT   */
/*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE     */
/*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES    */
/*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN   */
/*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,          */
/*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF       */
/*  THIS SOFTWARE.                                                       */
/*                                                                       */
/*************************************************************************/
/*                     Author :  Alan W Black                            */
/*                     Date   :  October 1996                            */
/*-----------------------------------------------------------------------*/
/*                                                                       */
/*   Interface to various low level waveform functions from Lisp         */
/*                                                                       */
/*=======================================================================*/
#include <stdio.h>
#include "EST_unix.h"
#include <stdlib.h>
#include "festival.h"
#include "festivalP.h"

static void utt_save_f0_from_targets(EST_Utterance *u,EST_String &filename);
static float f0_interpolate(EST_Item *s, float t);

static EST_Wave *get_utt_wave(EST_Utterance *u)
{
    EST_Relation *r;

    if (((r = u->relation("Wave")) == 0) || (r->head() == 0))
    {
	cerr << "no waveform in utterance" << endl;
	festival_error();
    }

    const void *p = r->head()->fP("wave", 1);
    EST_Wave *w = (EST_Wave *) p;
//    wave_info(*w);
    return w;
//    return (EST_Wave *) r->head()->fP("wave", 1));
}

static LISP utt_save_wave(LISP utt, LISP fname, LISP ltype)
{
    EST_Utterance *u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);
    EST_Wave *w = get_utt_wave(u);
    EST_String type;
    EST_String sampletype;

    if (fname == NIL)
	filename = "save.wav";

    if (ltype)
	type = get_c_string(ltype);
    else if (ft_get_param("Wavefiletype"))
	type = get_c_string(ft_get_param("Wavefiletype"));
    else 
	type = "nist";

    if (ft_get_param("Wavesampletype"))
	sampletype = get_c_string(ft_get_param("Wavesampletype"));
    else 
	sampletype = "short";

    if (w->save_file(filename,type,sampletype,EST_NATIVE_BO) != write_ok)
    {
	cerr << "utt.save.wave: failed to write wave to \"" << filename 
	    << "\"" << endl;
	festival_error();
    }

    return utt;
}

static LISP utt_save_f0(LISP utt, LISP fname)
{
    // Save utt's F0 in fname as an ESPS file
    EST_Utterance *u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);

    if ((u->has_relation("F0")) && (u->relation("F0")->head() != 0))
    {
	EST_Track *f0 = F0C(*(u->relation("F0")->head()));
	if (f0->save(filename,"esps") != write_ok)
	{
	    cerr << "utt.save.f0: failed to write f0 to \"" << 
		filename << "\"" << endl;
	    festival_error();
	}
    }
    else if (u->relation("Target") != 0)
	utt_save_f0_from_targets(u,filename);
    else
    {
	cerr << "utt.save.f0: utterance doesn't contain F0 or Target stream"
	    << endl;
	festival_error();
    }
    return utt;
}

static void utt_save_f0_from_targets(EST_Utterance *u,EST_String &filename)
{
    EST_Item *s;
    EST_Track f0;
    float p = 0.0;
    float length = u->relation("Segment")->last()->f("end");
    int i,frames = (int)(length / 0.010);
    f0.resize(frames,4);
    
    for (i=0,s=u->relation("Segment")->first(); s != 0; s=next(s))
    {
	if (i >= frames)
	    break;  // may hit here one before end
	for ( ; p < s->fF("end"); p+=0.010,i++)
	{
	    if (i >= frames)
		break;  // may hit here one before end
	    if ((ffeature(s,"ph_vc") == "+") ||
		(ffeature(s,"ph_cvox") == "+"))
	    {
		f0(i,0) = f0_interpolate(s,p);
		f0(i,1) = 1;
	    }
	    else
	    {
		f0(i,0) = 0;
		f0(i,1) = 0.0; // unvoiced;
	    }
	}
    }
    f0.set_channel_name("F0",0);
    f0.set_channel_name("prob_voice",1);
    f0.fill_time(0.01);

    f0.set_contour_type(ct_f0);

    if (f0.save(filename,"esps") != write_ok)
    {
	cerr << "utt.save.f0: failed to write F0 to \"" << 
	    filename << "\"" << endl;
	festival_error();
    }

    return;
}

static float f0_interpolate(EST_Item *s, float t)
{
    // Return interpolated F0 at time t (which should be within s)
    float p1,p0,d1,d0;
    float mid = ffeature(s,"segment_mid");

    if (t < mid)
    {
	p1 = ffeature(s,"seg_pitch");
	p0 = ffeature(s,"p.seg_pitch");
	d1 = mid;
	d0 = ffeature(s,"p.segment_mid");
    }
    else
    {
	p0 = ffeature(s,"seg_pitch");
	p1 = ffeature(s,"n.seg_pitch");
	d0 = mid;
	d1 = ffeature(s,"n.segment_mid");
    }
    if (p0 == 0.0)
	return  p1;       // first in stream
    else if (p1 == 0.0)
	return p0;        // last in stream
    else
	return p0 + (p1-p0)*(t-d0)/(d1-d0);
}

static LISP utt_send_wave_client(LISP utt)
{
    // Send the waveform to a client (must be acting as server)
    EST_Utterance *u = get_c_utt(utt);
    EST_Wave *w;
    EST_String tmpfile = make_tmp_filename();
    LISP ltype;
    EST_String type;

    w = get_utt_wave(u);
    if (ft_server_socket == -1)
    {
	cerr << "utt_send_wave_client: not in server mode" << endl;
	festival_error();
    }
	
    ltype = ft_get_param("Wavefiletype");
    if (ltype == NIL)
	type = "nist";
    else
	type = get_c_string(ltype);
    w->save(tmpfile,type);
    write(ft_server_socket,"WV\n",3);
    socket_send_file(ft_server_socket,tmpfile);
    unlink(tmpfile);

    return utt;
}

static LISP utt_import_track(LISP utt, LISP fname, LISP rname, LISP iname)
{
    // Create an wave stream and load fname into it
    EST_Utterance *u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);
    EST_String relname = get_c_string(rname);
    EST_String itemname = get_c_string(iname);
    EST_Track *track = new EST_Track;
    EST_Item *item;

    if (track->load(filename) != format_ok)
    {
	cerr << "Cannot load Trackfile: " << filename << endl;
	festival_error();
    }

    if (!u->has_relation(relname))
	u->create_relation(relname);

    item = u->relation(relname)->append();
    item->fset(itemname, track, gc_track);

    return utt;
}

static LISP utt_import_wave(LISP utt,LISP fname,LISP append)
{
    // Create an wave stream and load fname into it
    EST_Utterance *u = get_c_utt(utt);
    EST_String filename = get_c_string(fname);
    EST_Wave *wave = new EST_Wave;
    EST_Wave *existing_wave;
    EST_Item *item;

    if (wave->load(filename) != format_ok)
    {
	cerr << "Cannot load wavefile: " << filename << endl;
	festival_error();
    }

    if ((append) && (u->has_relation("Wave")))
    {
	// Append it to existing one
	item = u->relation("Wave")->head();
	existing_wave = get_utt_wave(u);
	*existing_wave += *wave;
	delete wave;
    }
    else 
    {
	item = u->create_relation("Wave")->append();
	item->fset("wave",wave,gc_wave);
    }

    return utt;
}

void play_wave(EST_Wave *w)
{
    EST_Option al;
    LISP audio;
	
    if ((audio = ft_get_param("Audio_Method")) != NIL)
	al.add_item("-p",get_c_string(audio));
    if ((audio = ft_get_param("Audio_Command")) != NIL)
	al.add_item("-command",quote_string(get_c_string(audio)));
    if ((audio = ft_get_param("Audio_Required_Rate")) != NIL)
	al.add_item("-rate",get_c_string(audio));
    if ((audio = ft_get_param("Audio_Required_Format")) != NIL)
	al.add_item("-otype",get_c_string(audio));
    al.add_item("-quality","HIGH");
    play_wave(*w,al);
}

static LISP utt_play(LISP utt)
{
    // Play the wave through the selected audio device 
    EST_Utterance *u = get_c_utt(utt);
    EST_Wave *w;

    w = get_utt_wave(u);
    
    if (audsp_mode)  // asynchronous mode
	audsp_play_wave(w);
    else
	play_wave(w);
    return utt;

}

static LISP utt_rescale(LISP utt,LISP lfactor, LISP ltype)
{
    // Modify the waveform gain, either relative or absolute.
    EST_Utterance *u = get_c_utt(utt);
    EST_Wave *w = get_utt_wave(u);
    float factor = get_c_float(lfactor);
	
    if ((factor < 0.01) || (factor > 100.0))  // sanity check
    {
	cerr << "Wave rescale: modification factor too weird: " <<
	    factor << endl;
	festival_error();
    }
    
    if ((ltype == NIL) || (streq(get_c_string(ltype),"absolute")))
	w->rescale(factor);
    else
    {
	if (factor > 1.0)
	{
	    cerr << "Wave rescale: in relative mode factor cannot be greater than 1.0" << endl;
	    festival_error();
	}
	w->rescale(factor,TRUE);  // normalize wave first
    }

    return utt;
}

static LISP utt_resample(LISP utt, LISP lrate)
{
    // Resample wave to desirsed rate
    EST_Utterance *u = get_c_utt(utt);
    EST_Wave *w = get_utt_wave(u);
    int rate = get_c_int(lrate);

    w->resample(rate);

    return utt;
}

static LISP utt_wave_info(LISP utt, LISP item)
{
    // Returns information about the waveform in utt;
    EST_Utterance *u = get_c_utt(utt);
    EST_Wave *w = get_utt_wave(u);

    if (item != NIL)
    {
	if (streq("num_samples",get_c_string(item)))
	    return flocons(w->num_samples());
	else if (streq("sample_rate",get_c_string(item)))
	    return flocons(w->sample_rate());
	else if (streq("data_addr",get_c_string(item)))
	{
	    // This is an awful hack, but its better than 
	    // possible alternatives
	    char addr[256];
	    // This doesn't work any more
	    sprintf(addr,"0x%p",w);
	    return strintern(addr);
	}
	else
	{
	    cerr << "wave.info: unknown info type \"" << 
		get_c_string(item) << "\"" << endl;
	    festival_error();
	}
    }
    else  // a list of items
    {
	LISP l,ritems = NIL;
	for (l=item; l != NIL; l=cdr(l))
	{
	    ritems = cons(make_param_lisp(get_c_string(car(l)),
					  utt_wave_info(utt,car(l))),
			  ritems);
	}
	return reverse(ritems);
    }

    return NIL;
}

void festival_wave_init(void)
{
    // declare utterance (wave) specific Lisp functions 

    init_subr_3("utt.save.wave",utt_save_wave,
 "(utt.save.wave UTT FILENAME TYPE)\n\
  Save the waveform of UTT in FILENAME in the format specified by TYPE. \n\
  If TYPE is nil or unspecified use the Parameter Wavefiletype for the \n\
  type.  The Parameter Wavesampletype is respect, short is the default if \n\
  this parameter is unset, mulaw is a potential value.  Note that is you \n\
  want the save file to be in a particular sample rate use \n\
  utt.wave.resample.");
    init_subr_1("utt.play",utt_play,
 "(utt.play UTT)\n\
  Send the waveform of UTT to the audio device as specified by the\n\
  Parameter Audio_Method.");
    init_subr_3("utt.import.wave",utt_import_wave,
 "(utt.import.wave UTT FILENAME APPEND)\n\
  Load waveform from FILENAME into UTT. If APPEND is nil or unspecified\n\
  create a new Wave relation and set it to the loaded wave, if APPEND is\n\
  non null them append the loaded wave to the existing one (or create \n\
  a new one if there is none.");

    init_subr_4("utt.import.track",utt_import_track,
 "(utt.import.wave UTT FILENAME RELATION FEATURE_NAME)\n\
  Load track from FILENAME into UTT. The track is placed in an item which \n\
  is appended to relation RELATION. The track is the value of the feature \n\
  FEATURE_NAME.");

    init_subr_2("utt.wave.resample",utt_resample,
 "(utt.wave.resample UTT RATE)\n\
  Resample waveform in UTT to RATE (if it is already at that rate it remains\n\
  unchanged).");
    init_subr_3("utt.wave.rescale",utt_rescale,
 "(utt.wave.rescale UTT FACTOR TYPE)\n\
  Modify the gain of the waveform in UTT by FACTOR (which should be in the\n\
  range 0.01 > FACTOR > 100.0).  If TYPE is specified and not 'absolute\n\
  then is normalized first.  It is normalized by making the magnitude of\n\
  of the largest sample point 32767.");  
    init_subr_2("utt.wave.info",utt_wave_info,
 "(utt.wave.info UTT TAG)\n\
  Return info about wave.  TAG may be num_samples, sample_rate or data_addr,\n\
  or a list of any of these.  If TAG is atomic an atomic value is returned,\n\
  if tag is a list then a list of values is returned (in the order they\n\
  are specified in TAG.");
    init_subr_1("utt.send.wave.client",utt_send_wave_client,
 "(utt.send.wave.client UTT)\n\
  Sends wave in UTT to client.  If not in server mode gives an error\n\
  Note the client must be expecting to receive the waveform.");
    init_subr_2("utt.save.f0",utt_save_f0,
 "(utt.save.f0 UTT FILENAME)\n\
 Save F0 of UTT as esps track file in FILENAME.");

}

    

