 /************************************************************************/
 /*                                                                      */
 /*                Centre for Speech Technology Research                 */
 /*                     University of Edinburgh, UK                      */
 /*                       Copyright (c) 1996,1997                        */
 /*                        All Rights Reserved.                          */
 /*                                                                      */
 /*  Permission to use, copy, modify, distribute this software and its   */
 /*  documentation for research, educational and individual use only, is */
 /*  hereby granted without fee, subject to the following conditions:    */
 /*   1. The code must retain the above copyright notice, this list of   */
 /*      conditions and the following disclaimer.                        */
 /*   2. Any modifications must be clearly marked as such.               */
 /*   3. Original authors' names are not deleted.                        */
 /*  This software may not be used for commercial purposes without       */
 /*  specific prior written permission from the authors.                 */
 /*                                                                      */
 /*  THE UNIVERSITY OF EDINBURGH AND THE CONTRIBUTORS TO THIS WORK       */
 /*  DISCLAIM ALL WARRANTIES WITH REGARD TO THIS SOFTWARE, INCLUDING     */
 /*  ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO EVENT  */
 /*  SHALL THE UNIVERSITY OF EDINBURGH NOR THE CONTRIBUTORS BE LIABLE    */
 /*  FOR ANY SPECIAL, INDIRECT OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES   */
 /*  WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN  */
 /*  AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION,         */
 /*  ARISING OUT OF OR IN CONNECTION WITH THE USE OR PERFORMANCE OF      */
 /*  THIS SOFTWARE.                                                      */
 /*                                                                      */
 /*************************************************************************/
 /*                                                                       */
 /*                 Author: Richard Caley (rjc@cstr.ed.ac.uk)             */
 /*                   Date: Mon June 30 1997                              */
 /* ------------------------------------------------------------------    */
 /* A wrapper around the ModuleDescription object to allow it to be       */
 /* passed around in scheme.                                              */
 /*                                                                       */
 /*************************************************************************/


#include "SchemeModuleDescription.h"

EST_String SchemeModuleDescription::type_name(void) const { return "SchemeModuleDescription";}

#if defined(INCLUDE_LISP)
int SchemeModuleDescription::s_lisp_type = tc_festival_schememoduledescription;
#endif

SchemeModuleDescription::SchemeModuleDescription(const ModuleDescription *description, bool allocated)
{
  p_description = description;
  if (description)
    p_name = description->name;
  p_allocated = allocated;
}

SchemeModuleDescription::~SchemeModuleDescription(void)
{
//    if (p_allocated)
//	delete p_alloc_description;
    p_description = NULL; 
}


void SchemeModuleDescription::property_names(EST_TList<EST_String> &list) const
{
  SchemeObject::property_names(list);
  list.append("version");
  list.append("organisation");
  list.append("author");
  list.append("description");
  list.append("input_streams");
  list.append("optional_streams");
  list.append("output_streams");
  list.append("parameters");
}

static EST_String combine_lines(const char * const *lines)
{
  EST_String text("");

  for(int i=0; i < MD_MAX_DESCRIPTION_LINES && lines[i]; i++)
    {
      if (i>0)
	text += "\n";
      text += lines[i];
    }

  return text;
}

static LISP streams_to_list(const struct ModuleDescription::stream_parameter streams[])
{
  LISP res =NIL;

  for(int i=0; i < MD_MAX_INPUT_STREAMS && streams[i].name; i++)
    {
      LISP stream = cons(rintern(streams[i].name),
			 cons(c_string_as_value(streams[i].description),
			      NIL));

      res = cons(stream, res);
    }
  return res;
}
static LISP parameters_to_list(const struct ModuleDescription::parameter params[])
{
  LISP res =NIL;

  for(int i=0; i < MD_MAX_PARAMETERS && params[i].name; i++)
    {
      LISP param = cons(rintern(params[i].name),
			cons(c_string_as_value(params[i].type),
			     cons(c_string_as_value(params[i].default_val),
				  cons(c_string_as_value(params[i].description),
				       NIL))));

      res = cons(param, res);
    }
  return res;
}

ValueType SchemeModuleDescription::property(EST_String property) const
{
  if (!p_description)
    return c_string_as_value("NULL DESCRIPTION");
  else if (property == "version")
    return float_as_value(p_description->version);
  else if (property == "organisation")
    return c_string_as_value(p_description->organisation);
  else if (property == "author")
    return c_string_as_value(p_description->author);
  else if (property == "description")
    return string_as_value(combine_lines(p_description->description));
  else if (property == "input_streams")
    return list_as_value(streams_to_list(p_description->input_streams));
  else if (property == "optional_streams")
    return list_as_value(streams_to_list(p_description->optional_streams));
  else if (property == "output_streams")
    return list_as_value(streams_to_list(p_description->output_streams));
  else if (property == "parameters")
    return list_as_value(parameters_to_list(p_description->parameters));
  else
    return SchemeObject::property(property);
}

int SchemeModuleDescription::set_property(EST_String property, ValueType value)
{
  (void) property;
  (void) value;
  error_string = "ModuleDescription objects are read only";
  return 0;
}

void  SchemeModuleDescription::print_description(FILE *stream) const
{
  if (p_description)
    ModuleDescription::print(stream, *p_description);
  else
    fputs("[null description]\n", stream);
}

#if defined(INCLUDE_LISP)

static void list_to_description(const char * d[MD_MAX_DESCRIPTION_LINES], 
				LISP list)
{
  int i=0;
  for (LISP dlist = list; CONSP(dlist) && i < MD_MAX_DESCRIPTION_LINES-1 ; dlist = CDR(dlist), i++)
    {
      d[i] = wstrdup(get_c_string(CAR(dlist)));
    }
  d[i] = NULL;
}

static void list_to_streams(ModuleDescription::stream_parameter s[MD_MAX_INPUT_STREAMS],
				LISP list)
{
  int i=0;
  for (LISP dlist = list; CONSP(dlist) && i <MD_MAX_INPUT_STREAMS-1 ; dlist = CDR(dlist), i++)
    {
      LISP line = CAR(dlist);
      if (LIST2P(line))
	{
	  s[i].name        = wstrdup(get_c_string(CAR1(line)));
	  s[i].description = wstrdup(get_c_string(CAR2(line)));
	}
    }
  s[i].name = NULL;
}

static void list_to_parameters(ModuleDescription::parameter p[MD_MAX_PARAMETERS],
				LISP list)
{
  int i=0;
  for (LISP dlist = list; CONSP(dlist) && i <MD_MAX_INPUT_STREAMS-1 ; dlist = CDR(dlist), i++)
    {
      LISP line = CAR(dlist);
      if (LIST4P(line))
	{
	  p[i].name        = wstrdup(get_c_string(CAR1(line)));
	  p[i].type        = wstrdup(get_c_string(CAR2(line)));
	  p[i].default_val = CAR3(line)?wstrdup(get_c_string(CAR3(line))):NULL;
	  p[i].description = wstrdup(get_c_string(CAR4(line)));
	}
    }
  p[i].name = NULL;
}

LISP SchemeModuleDescription::lisp_create(LISP list_form)
{
  struct ModuleDescription *desc = ModuleDescription::create();
  SchemeModuleDescription *sdesc = new SchemeModuleDescription(desc, TRUE);
  LISP scheme_desc = siod_make_typed_cell(tc_festival_schememoduledescription,
					  sdesc
					  );

  // NAME
  if (!CONSP(list_form))
    return scheme_desc;
  desc->name = wstrdup(get_c_string(CAR(list_form)));
  list_form = CDR(list_form);
  sdesc->set_name(desc->name);

  // VERSION
  if (!CONSP(list_form))
    return scheme_desc;
  desc->version = get_c_float(CAR(list_form));
  list_form = CDR(list_form);

  // ORGANISATION
  if (!CONSP(list_form))
    return scheme_desc;
  desc->organisation = wstrdup(get_c_string(CAR(list_form)));
  list_form = CDR(list_form);

  // AUTHOR
  if (!CONSP(list_form))
    return scheme_desc;
  desc->author = wstrdup(get_c_string(CAR(list_form)));
  list_form = CDR(list_form);

  // DESCRIPTION
  if (!CONSP(list_form))
    return scheme_desc;
  list_to_description(desc->description, CAR(list_form));
  list_form = CDR(list_form);

  // INPUT_STREAMS
  if (!CONSP(list_form))
    return scheme_desc;
  list_to_streams(desc->input_streams, CAR(list_form));
  list_form = CDR(list_form);

  // OPTIONAL_STREAMS
  if (!CONSP(list_form))
    return scheme_desc;
  list_to_streams(desc->optional_streams, CAR(list_form));
  list_form = CDR(list_form);

  // OUTPUT_STREAMS
  if (!CONSP(list_form))
    return scheme_desc;
  list_to_streams(desc->output_streams, CAR(list_form));
  list_form = CDR(list_form);

  // PARAMETERS
  if (!CONSP(list_form))
    return scheme_desc;
  list_to_parameters(desc->parameters, CAR(list_form));
  list_form = CDR(list_form);

  return scheme_desc;
}

LISP SchemeModuleDescription::lisp_print_description(LISP ldesc, LISP file)
{
  FILE *stream=get_c_file(file,stdout);

  if (ldesc == NIL || TYPE(ldesc) != tc_festival_schememoduledescription)
    {
      fputs("[not a module description]\n", stream);
      return NIL;
    }

  SchemeModuleDescription *desc = (SchemeModuleDescription *)USERVAL(ldesc);

  desc->print_description(stream);
  return ldesc;
}

void SchemeModuleDescription::lisp_declare(void)
{
  long gc_kind;
  set_gc_hooks(tc_festival_schememoduledescription,
	       1,
	       NULL,
	       SchemeObject::lisp_gc_mark,
	       NULL,
	       SchemeObject::lisp_gc_free,
	       SchemeObject::lisp_gc_clear_mark,
	       &gc_kind);
  set_print_hooks(tc_festival_schememoduledescription,
		  SchemeModuleDescription::lisp_print,
		  SchemeModuleDescription::lisp_print_string
		  );

  init_subr_1("create_module_description", SchemeModuleDescription::lisp_create,
  "(create_module_description LIST_FORM)\n\
   Create a module desription from LIST_FORM.");

  init_subr_2("print_module_description", SchemeModuleDescription::lisp_print_description,
 "(print_module_description MODULEDESCRIPTION FILE)\n\
  Print a description of the database.");
}
#endif


