%{
/********************************************************************************
 * Copyright (c) Erik Kunze 1997
 *
 * Permission to use, distribute, and sell this software and its documentation
 * for any purpose is hereby granted without fee, provided that the above
 * copyright notice appear in all copies and that both that copyright notice and
 * this permission notice appear in supporting documentation, and that the name
 * of the copyright holder not be used in advertising or publicity pertaining to
 * distribution of the software without specific, written prior permission.  The
 * copyright holder makes no representations about the suitability of this
 * software for any purpose.  It is provided "as is" without express or implied
 * warranty. THE CODE MAY NOT BE MODIFIED OR REUSED WITHOUT PERMISSION!
 *
 * THE COPYRIGHT HOLDER DISCLAIMS ALL WARRANTIES WITH REGARD TO THIS SOFTWARE,
 * INCLUDING ALL IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS, IN NO
 * EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY SPECIAL, INDIRECT OR
 * CONSEQUENTIAL DAMAGES OR ANY DAMAGES WHATSOEVER RESULTING FROM LOSS OF USE,
 * DATA OR PROFITS, WHETHER IN AN ACTION OF CONTRACT, NEGLIGENCE OR OTHER
 * TORTIOUS ACTION, ARISING OUT OF OR IN CONNECTION WITH THE USE OR
 * PERFORMANCE OF THIS SOFTWARE.
 *
 * Author: Erik Kunze
 *
 * changed by EKU
 *******************************************************************************/
#ifndef lint
static char monitor1_y[] = "$Id: monitor1.y,v 4.1 1998/01/06 21:21:30 erik Rel $";
#endif

#include <X11/keysym.h>
#include <stdio.h>
#include <unistd.h>
#include <string.h>
#include "config.h"
#include "z80.h"
#include "resource.h"
#include "mem.h"
#include "dialog.h"
#include "dis.h"
#include "screen.h"
#include "monitor.h"

struct reginfo {
	uns8 *high_reg;
	uns8 *low_reg;
	int x;
	int y;
};

static int yyparse (void);
static void yyerror(char *);
static void dbgPrintString(char *);
static void dbgPrintPanel(void);
static void dbgDisassemble(uns16);
static uns16 dbgGetMnemo(uns16);

int NumberBase = 10;
char *CmdLine;

static char buf[COLS + 1];
static int leaveDebugger;
static uns16 memoryPtr = 0;
static int activeReg = 0;
static struct reginfo regs[] = {
	{ &A,   &F,    2,  2 },
	{ &A1,  &F1,  13,  2 },
	{ &B,   &C,    2,  3 },
	{ &B1,  &C1,  13,  3 },
	{ &D,   &E,    2,  4 },
	{ &D1,  &E1,  13,  4 },
	{ &H,   &L,    2,  5 },
	{ &H1,  &L1,  13,  5 },
	{ &HIX, &LIX,  2,  6 },
	{ &HIY, &LIY,  2,  7 },
	{ &HPC, &LPC,  2,  8 },
	{ &HSP, &LSP,  2,  9 },
	{ &I,   &R,   13,  7 }
};

%}
%token BAS2
%token BAS8
%token BAS10
%token BAS16
%token BASE
%token COPY
%token DIS
%token EQUAL
%token FILL
%token GO
%token INT_MODE
%token INT_FF1
%token INT_FF2
%token MINUS
%token NL
%token NUMBER
%token PLUS
%token QUIT
%token REGISTER
%token UNKNOWN
%%
line
	: NL
	| command
	| UNKNOWN
	| error
	{
		yyerrok;
		yyclearin;
		YYABORT;
	}
	;
command
	: register
	| memory
	| fill
	| copy
	| dis
	| go
	| quit
	| number_base
	| interrupt
	;
register
	: REGISTER NL
	{
		activeReg = ++activeReg % (sizeof(regs)/sizeof(regs[0]));
	}
	| REGISTER NUMBER NL
	{
		*(regs[activeReg].high_reg) = HIGHBYTE($2);
		*(regs[activeReg].low_reg) = LOWBYTE($2);
	}
	;
memory
	: PLUS NL
	{
		memoryPtr++;
	}
	| PLUS NUMBER NL
	{
		memoryPtr += $2;
	}
	| MINUS NL
	{
		memoryPtr--;
	}
	| MINUS NUMBER NL
	{
		memoryPtr -= $2;
	}
	| EQUAL NL
	{
		memoryPtr =
			(*(regs[activeReg].high_reg) << 8) | *(regs[activeReg].low_reg);
	}
	| EQUAL NUMBER NL
	{
		memoryPtr = $2;
	}
	| NUMBER NL
	{
		if ($1 <= 0xff)
		{
			WR_BYTE(memoryPtr, (uns8)$1);
			memoryPtr++;
		}
		else
		{
			WR_WORD(memoryPtr, (uns16)$1);
			memoryPtr += 2;
		}
	}
	;
fill
	: FILL NUMBER NUMBER NL
	{
		uns16 i;
		for (i = (uns16)$2; i <= (uns16)$3; i++)
		{
			WR_BYTE(i, 0);
		}
	}
	| FILL NUMBER NUMBER NUMBER NL
	{
		uns16 i;
		if ($4 > 0xff)
		{
			YYERROR;
		}
		else
		{
			for (i = (uns16)$2; i <= (uns16)$3; i++)
			{
				WR_BYTE(i, (uns8)$4);
			}
		}
	}
	;
copy
	: COPY NUMBER NUMBER NUMBER NL
	{
		uns16 src = $2, dst = $3;
		uns16 cnt = $4;
		if (src + cnt < dst || src > dst)
		{

			for (; cnt > 0; cnt--)
			{
				WR_BYTE(dst++, RD_BYTE(src++));
			}
		}
		else
		{

			src += cnt;
			dst += cnt;
			for (; cnt > 0; cnt--)
			{
				WR_BYTE(dst--, RD_BYTE(src--));
			}
		}
	}
	;
dis
	: DIS NL
	{
		dbgDisassemble(PC);
	}
	| DIS NUMBER NL
	{
		dbgDisassemble((uns16)$2);
	}
	;
go
	: GO NL
	{
		leaveDebugger = 1;
	}
	| GO NUMBER NL
	{
		PC = $2;
		leaveDebugger = 1;
	}
	;
quit
	: QUIT NL
	{
		leaveDebugger = 1;
	}
	;
number_base
	: BASE NL
	{
		char *p = NULL;
		switch (NumberBase)
		{
			case 2:
				p = "2";
				break;
			case 8:
				p = "8";
				break;
			case 10:
				p = "10";
				break;
			case 16:
				p = "16";
				break;
		}
		dbgPrintString(p);
	}
	| BAS2 NL
	{
		NumberBase = 2;
	}
	| BAS8 NL
	{
		NumberBase = 8;
	}
	| BAS10 NL
	{
		NumberBase = 10;
	}
	| BAS16 NL
	{
		NumberBase = 16;
	}
	;
interrupt
	: INT_MODE NUMBER NL
	{
		if ($2 >= 0 && $2 <= 2)
		{
			IM = $2;
		}
		else
		{
			YYERROR;
		}
	}
	| INT_FF1 NUMBER NL
	{
		if ($2 >= 0 && $2 <= 1)
		{
			IFF1 = $2;
		}
		else
		{
			YYERROR;
		}
	}
	| INT_FF2 NUMBER NL
	{
		if ($2 >= 0 && $2 <= 1)
		{
			IFF2 = $2;
		}
		else
		{
			YYERROR;
		}
	}
	;
%%

static void
yyerror(char *s)
{
	dbgPrintString("eh?");
}

void
Debugger(int why)
{
	int oldInDialog;
	char input[COLS];

	oldInDialog = EnterOSD();
	SetAttr((BLACK << 3) | WHITE);
	SetBorderColor(BLACK);
	for (leaveDebugger = 0; leaveDebugger == 0;)
	{

		dbgPrintPanel();

		do
		{
			SetCursor(0, 23);
			PrintLetter('>');
		}
		while (!GetString(input, sizeof(input) - 1, COLS - 1));

		(void)strcat(input, "\n");
		CmdLine = input;
		yylex_init();
		(void)yyparse();
	}
	SetBorderColor((MENU_TEXT_ATTR & PAPER) >> 3);
	LeaveOSD(oldInDialog);
}

static void
dbgPrintString(char *string)
{
	SetCursor(1, 23);
	DrawBox(COLS - 2, 1);
	PrintString(string);
	ScreenRefresh();
	(void)GetKey();
}

static void
dbgPrintPanel(void)
{
	int i;
	uns16 mem;
	ClearScreen();

	(void)dbgGetMnemo(PC);
	PrintString(buf);

	SetCursor(0, 2);
	(void)sprintf(buf, "AF %04X   AF' %04X      SZXHXVPC", AF, AF1);
	PrintString(buf);
	SetCursor(0, 3);
	(void)sprintf(buf, "BC %04X   BC' %04X   F  ", BC, BC1);
	PrintString(buf);
	for (i = 0x80; i > 0; i >>= 1)
	{
		PrintLetter(F & i ? '1' : '0');
	}
	SetCursor(0, 4);
	(void)sprintf(buf, "DE %04X   DE' %04X   F' ", DE, DE1);
	PrintString(buf);
	for (i = 0x80; i > 0; i >>= 1)
	{
		PrintLetter(F1 & i ? '1' : '0');
	}
	SetCursor(0, 5);
	(void)sprintf(buf, "HL %04X   HL' %04X", HL, HL1);
	PrintString(buf);
	SetCursor(0, 6);
	(void)sprintf(buf, "IX %04X              IM 1 2 RMVL", IX);
	PrintString(buf);
	SetCursor(0, 7);
	(void)sprintf(buf, "IY %04X   IR  %02X%02X    %1d %1d %1d ",
				  IX,I, R, IM, IFF1, IFF2);
	PrintString(buf);

	switch (GETCFG(machine))
	{
		case SP_48_2:
		case SP_48_3:
			(void)sprintf(buf, "%1d---", RPAGE(0));
			break;
		case SP_128:
			(void)sprintf(buf, "%1d%1d%1d%1d",
						  RPAGE(0),
						  Last0x7FFD & B_SELRAM,
						  Last0x7FFD & B_SELSCREEN ? 7 : 5,
						  (Last0x7FFD & B_PAGING) >> 5);
			break;
#ifdef XZX_PLUS3
		case SP_3:
			if (!EXTENDED_RAM(Last0x1FFD))
			{
				(void)sprintf(buf, "%1d%1d%1d%1d",
							  RPAGE(0),
							  Last0x7FFD & B_SELRAM,
							  Last0x7FFD & B_SELSCREEN ? 7 : 5,
							  (Last0x7FFD & B_PAGING) >> 5);
			}
			else
			{
				(void)sprintf(buf, "-%1d-0", (Last0x1FFD & B_SELRAM3) >> 1);
			}
			break;
#endif
	}
	PrintString(buf);

	SetCursor(0, 8);
	(void)sprintf(buf, "PC %04X", PC);
	PrintString(buf);
	SetCursor(0, 9);
	(void)sprintf(buf, "SP %04X", SP);
	PrintString(buf);

	for (mem = memoryPtr - 12, i = 0; i < 8; mem++, i++)
	{
		SetCursor(0, 13 + i);
		(void)sprintf(buf, "%04X %02X   %04X %02X   %04X %02X",
					  mem, RD_BYTE(mem),
					  (uns16)(mem + 8), RD_BYTE(mem + 8),
					  (uns16)(mem + 16), RD_BYTE(mem +16));
		PrintString(buf);
	}

	SetCursor(9, 17);
	PrintLetter('>');
	SetCursor(17, 17);
	PrintLetter('<');

	SetCursor(regs[activeReg].x, regs[activeReg].y);
	PrintLetter('*');
	ScreenRefresh();
}

static void
dbgDisassemble(uns16 addr)
{
	int i;
	do
	{
		ClearScreen();
		for (i = 0; i < ROWS; i++)
		{
			SetCursor(0, i);
			addr = dbgGetMnemo(addr);
			PrintString(buf);
		}
		ScreenRefresh();
	}
	while (GetKey() != XK_Escape);
}

static uns16
dbgGetMnemo(uns16 addr)
{
	char *p;
	int i, j;
	p = buf;
	(void)sprintf(p, "%04X  ", addr);
	p += 6;
	j = Disassemble(buf + 16, addr) - addr;
	for (i = 0; i < j; i++)
	{
		(void)sprintf(p, "%02X", RD_BYTE(addr + i));
		p += 2;
	}
	while (p < &buf[16])
	{
		*p++ = ' ';
	}
	return (addr + j);
}

