//============================================================================
//
//    SSSS    tt          lll  lll              
//   SS  SS   tt           ll   ll                
//   SS     tttttt  eeee   ll   ll   aaaa    "An Atari 2600 VCS Emulator"
//    SSSS    tt   ee  ee  ll   ll      aa      
//       SS   tt   eeeeee  ll   ll   aaaaa   Copyright (c) 1995,1996,1997
//   SS  SS   tt   ee      ll   ll  aa  aa         Bradford W. Mott
//    SSSS     ttt  eeeee llll llll  aaaaa    
//
//============================================================================

/**
  RIOT

  @author  Bradford W. Mott
  @version $Id: M6532.cxx,v 1.2 1997/05/17 19:00:06 bwmott Exp $
*/

#include <iostream.h>

#include "Random.hxx"
#include "System.hxx"
#include "M6532.hxx"

//============================================================================
// Constructor
//============================================================================
M6532::M6532(System& system)
    : Device(system),
      myRAMOffset(0)
{
  // Map all of my addresses in the system
  for(int address = 0; address < 8192; ++address)
  {
    if((address & 0x1080) == 0x0080)
    {
      if((address & 0x0200) == 0x0000)
      {
        mySystem.mapPeek(address, *this, &myRAM[address & 0x7f], &myRAMOffset);
        mySystem.mapPoke(address, *this, &myRAM[address & 0x7f]);
      }
      else
      {
        mySystem.mapPeek(address, *this);
        mySystem.mapPoke(address, *this);
      }
    }
  }

  // Get the timer adjustment value from the properties
  myTimerAdjustment = mySystem.properties().integer("Timer.Adjustment");

  // Put myself in power-up state
  reset();
}
 
//============================================================================
// Destructor
//============================================================================
M6532::~M6532()
{
}
 
//============================================================================
// Reset to my power on state
//============================================================================
void M6532::reset()
{
  myTimer = 100;
  myIntervalShift = 6;
  myCyclesWhenTimerSet = 0;
  myTimerReadAfterInterrupt = false;

  // Zero the I/O registers
  myDDRA = 0x00;
  myDDRB = 0x00;

  // Randomize the 128 bytes of memory
  Random random;

  for(uWord t = 0; t < 128; ++t)
    myRAM[t] = (uByte)random;
}

//============================================================================
// Answer the byte at the given address
//============================================================================
uByte M6532::peek(uWord addr)
{
  switch(addr & 0x07)
  {
    case 0x00:    // Port A I/O Register (Joystick)
    {
      return ((mySystem.controller(0).readPIA() & 0x0f) << 4) |
          (mySystem.controller(1).readPIA() & 0x0f);
    }

    case 0x01:    // Port A Data Direction Register 
    {
      return myDDRA;
    }

    case 0x02:    // Port B I/O Register (Console switches)
    {
      return mySystem.console().switches();
    }

    case 0x03:    // Port B Data Direction Register
    {
      return myDDRB;
    }

    case 0x04:    // Timer Output
    case 0x06:
    {
      uLong cycles = mySystem.m6507().cycles() - 1;
      uLong delta = cycles - myCyclesWhenTimerSet - myTimerAdjustment;
      Long timer = (Long)myTimer - (Long)(delta >> myIntervalShift) - 1;

      // See if the timer has expired yet?
      if(timer >= 0)
      {
        return (uByte)timer; 
      }
      else
      {
        timer = (Long)(myTimer << myIntervalShift) - (Long)delta - 1;

        if(timer <= -2)
        {
          // Indicate that timer has been read after interrupt occured
          myTimerReadAfterInterrupt = true;
        }

        return (uByte)timer;
      }
    }

    case 0x05:    // Interrupt Flag
    case 0x07:
    {
      uLong cycles = mySystem.m6507().cycles() - 1;
      uLong delta = cycles - myCyclesWhenTimerSet - myTimerAdjustment;
      Long timer = (Long)myTimer - (Long)(delta >> myIntervalShift) - 1;

      if((timer >= 0) || myTimerReadAfterInterrupt)
        return 0x00;
      else
        return 0x80;
    }

    default:
    {    
#ifdef DEBUG_ACCESSES
      cerr << "BAD M6532 Peek: " << hex << addr << endl;
#endif
      return 0;
    }
  }
}

//============================================================================
// Write value at the given address
//============================================================================
void M6532::poke(uWord addr, uByte value)
{
  if((addr & 0x07) == 0x00)         // Port A I/O Register (Joystick)
  {
    return;
  }
  else if((addr & 0x07) == 0x01)    // Port A Data Direction Register 
  {
    myDDRA = value;
  }
  else if((addr & 0x07) == 0x02)    // Port B I/O Register (Console switches)
  {
    return;
  }
  else if((addr & 0x07) == 0x03)    // Port B Data Direction Register
  {
//        myDDRB = value;
    return;
  }
  else if((addr & 0x17) == 0x14)    // Write timer divide by 1 
  {
    myTimer = value;
    myIntervalShift = 0;
    myCyclesWhenTimerSet = mySystem.m6507().cycles();
    myTimerReadAfterInterrupt = false;
  }
  else if((addr & 0x17) == 0x15)    // Write timer divide by 8
  {
    myTimer = value;
    myIntervalShift = 3;
    myCyclesWhenTimerSet = mySystem.m6507().cycles();
    myTimerReadAfterInterrupt = false;
  }
  else if((addr & 0x17) == 0x16)    // Write timer divide by 64
  {
    myTimer = value;
    myIntervalShift = 6;
    myCyclesWhenTimerSet = mySystem.m6507().cycles();
    myTimerReadAfterInterrupt = false;
  }
  else if((addr & 0x17) == 0x17)    // Write timer divide by 1024
  {
    myTimer = value;
    myIntervalShift = 10;
    myCyclesWhenTimerSet = mySystem.m6507().cycles();
    myTimerReadAfterInterrupt = false;
  }
  else if((addr & 0x14) == 0x04)    // Write Edge Detect Control
  {
#ifdef DEBUG_ACCESSES
    cerr << "M6532 Poke (Write Edge Detect): "
        << ((addr & 0x02) ? "PA7 enabled" : "PA7 disabled")
        << ", "
        << ((addr & 0x01) ? "Positive edge" : "Negative edge")
        << endl;
#endif
  }
  else
  {
#ifdef DEBUG_ACCESSES
    cerr << "BAD M6532 Poke: " << hex << addr << endl;
#endif
  }
}
 

