
/*
 * DREADERD/THREAD.C - select() support for threads
 *
 *	Thread module used to support non-blocking multi-threaded I/O
 *	(see also mbuf.c)
 *
 * (c)Copyright 1998, Matthew Dillon, All Rights Reserved.  Refer to
 *    the COPYRIGHT file in the base directory of this distribution
 *    for specific rights granted.
 */

#include "defs.h"

Prototype ForkDesc *AddThread(const char *id, int fd, pid_t pid, int type, int slot);
Prototype ForkDesc *FindThread(int fd, pid_t pid);
Prototype ForkDesc *FindLeastUsedThread(int type, int maxCount);
Prototype ForkDesc *FindThreadId(int type, const char *id);
Prototype void ScanThreads(int type, void (*func)(ForkDesc *desc));
Prototype void DelThread(ForkDesc *desc);
Prototype void ResetThreads(void);
Prototype void DumpThreads(void);

Prototype fd_set SFds;
Prototype fd_set RFds;
Prototype fd_set WFds;
Prototype int MaxFds;

fd_set SFds;
fd_set RFds;
fd_set WFds;
int MaxFds;

ForkDesc *FDes[MAXFDS];
MemPool	 *TMemPool;

ForkDesc *
AddThread(const char *id, int fd, pid_t pid, int type, int slot)
{
    ForkDesc *desc;

    if (fd < 0 || fd >= MAXFDS || (desc = FDes[fd]) != NULL) {
	logit(LOG_CRIT, "AddThread: descriptor %d already in use", fd);
	exit(1);
    }
    desc = FDes[fd] = zalloc(&TMemPool, sizeof(ForkDesc));

    fcntl(fd, F_SETFL, O_NONBLOCK);

    desc->d_Id = zallocStr(&TMemPool, id);
    desc->d_Slot = slot;
    desc->d_Pid = pid;
    desc->d_Fd = fd;
    desc->d_Type = type;
    desc->d_FdPend = -1;
    FD_SET(fd, &SFds);
    if (MaxFds <= fd)
	MaxFds = fd + 1;
    return(desc);
}

ForkDesc *
FindThread(int fd, pid_t pid)
{
    int i;

    if (fd >= 0)
	return(FDes[fd]);
    for (i = 0; i < MaxFds; ++i) {
	if (FDes[i] && FDes[i]->d_Pid == pid)
	    return(FDes[i]);
    }
    return(NULL);
}

ForkDesc *
FindLeastUsedThread(int type, int maxCount)
{
    int i;
    int bestCount = 0x7FFFFFFF;
    ForkDesc *best = NULL;

    for (i = 0; i < MaxFds; ++i) {
	ForkDesc *desc;

	if ((desc = FDes[i]) != NULL && 
	    desc->d_Type == type &&
	    desc->d_Count < bestCount &&
	    desc->d_Count < maxCount
	) {
	    best = desc;
	    bestCount = best->d_Count;
	}
    }
    return(best);
}

ForkDesc *
FindThreadId(int type, const char *id)
{
    int i;

    for (i = 0; i < MaxFds; ++i) {
	ForkDesc *desc;

	if ((desc = FDes[i]) != NULL && 
	    desc->d_Type == type &&
	    strcmp(id, desc->d_Id) == 0
	) {
	    return(desc);
	}
    }
    return(NULL);
}

void
ScanThreads(int type, void (*func)(ForkDesc *desc))
{
    int i;

    for (i = 0; i < MaxFds; ++i) {
	ForkDesc *desc;

	if ((desc = FDes[i]) != NULL && desc->d_Type == type)
	    func(desc);
    }
}

void
DelThread(ForkDesc *desc)
{
    int fd = desc->d_Fd;

    if (fd >= 0) {
	FDes[fd] = NULL;
	close(fd);
	desc->d_Fd = -1;

	FD_CLR(fd, &SFds);
	FD_CLR(fd, &RFds);
	FD_CLR(fd, &WFds);

	if (fd + 1 == MaxFds) {
	    while (fd >= 0) {
		if (FDes[fd] != NULL)
		    break;
		--fd;
	    }
	    MaxFds = fd + 1;
	}
    }
    zfreeStr(&TMemPool, &desc->d_Id);
    zfree(&TMemPool, desc, sizeof(ForkDesc));
}

void
ResetThreads(void)
{
    int i;

    for (i = 0; i < MaxFds; ++i) {
	ForkDesc *desc;
	if ((desc = FDes[i]) != NULL) {
	    if (desc->d_Fd >= 0) {
		close(desc->d_Fd);
		desc->d_Fd = -1;
	    }
	    if (desc->d_FdPend >= 0) {
		close(desc->d_FdPend);
		desc->d_FdPend = -1;
	    }
	}
    }
    FD_ZERO(&SFds);
    FD_ZERO(&RFds);
    FD_ZERO(&WFds);
    freePool(&TMemPool);
    bzero(FDes, sizeof(FDes));
    MaxFds = 0;
}

void
DumpThreads(void)
{
    int i;

    printf("**** DUMP THREADS ****\n");
    sleep(1);

    for (i = 0; i < MAXFDS; ++i) {
	ForkDesc *desc;

	if ((desc = FDes[i]) != NULL) {
	    if (i >= MaxFds)
		printf("DESC > MAXFDS (%d,%d)\n", i, MaxFds);
	    printf("DESC %s type=%d count=%d\n", desc->d_Id, desc->d_Type, desc->d_Count);
	}
    }
}
