/***************************************************************************/
/***************************************************************************/
/*                                                                         */
/*   (c) 1995-1998.  The Regents of the University of California.  All     */
/*   rights reserved.                                                      */
/*                                                                         */
/*   This work was produced at the University of California, Lawrence      */
/*   Livermore National Laboratory (UC LLNL) under contract no.            */
/*   W-7405-ENG-48 (Contract 48) between the U.S. Department of Energy     */
/*   (DOE) and The Regents of the University of California (University)    */
/*   for the operation of UC LLNL.  Copyright is reserved to the           */
/*   University for purposes of controlled dissemination,                  */
/*   commercialization through formal licensing, or other disposition      */
/*   under terms of Contract 48; DOE policies, regulations and orders;     */
/*   and U.S. statutes.  The rights of the Federal Government are          */
/*   reserved under Contract 48 subject to the restrictions agreed upon    */
/*   by the DOE and University.                                            */
/*                                                                         */
/*                                                                         */
/*                              DISCLAIMER                                 */
/*                                                                         */
/*   This software was prepared as an account of work sponsored by an      */
/*   agency of the United States Government.  Neither the United States    */
/*   Government nor the University of California nor any of their          */
/*   employees, makes any warranty, express or implied, or assumes any     */
/*   liability or responsibility for the accuracy, completeness, or        */
/*   usefulness of any information, apparatus, product, or process         */
/*   disclosed, or represents that its specific commercial products,       */
/*   process, or service by trade name, trademark, manufacturer, or        */
/*   otherwise, does not necessarily constitute or imply its               */
/*   endorsement, recommendation, or favoring by the United States         */
/*   Government or the University of California. The views and opinions    */
/*   of the authors expressed herein do not necessarily state or reflect   */
/*   those of the United States Government or the University of            */
/*   California, and shall not be used for advertising or product          */
/*   endorsement purposes.                                                 */
/*                                                                         */
/*   Permission to use, copy, modify and distribute this software and its  */
/*   documentation for any non-commercial purpose, without fee, is         */
/*   hereby granted, provided that the above copyright notice and this     */
/*   permission notice appear in all copies of the software and            */
/*   supporting documentation, and that all UC LLNL identification in      */
/*   the user interface remain unchanged.  The title to copyright LLNL     */
/*   XDIR shall at all times remain with The Regents of the University     */
/*   of California and users agree to preserve same. Users seeking the     */
/*   right to make derivative works with LLNL XDIR for commercial          */
/*   purposes may obtain a license from the Lawrence Livermore National    */
/*   Laboratory's Technology Transfer Office, P.O. Box 808, L-795,         */
/*   Livermore, CA 94550.                                                  */
/*                                                                         */
/***************************************************************************/
/***************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <Xm/Form.h>
#include <Xm/Separator.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#include <Xm/Label.h>
#include <Xm/TextF.h>
#include <Xm/Text.h>
#include <Xm/PushB.h>
#include "list.h"
#include "str.h"

#define MAXFPLINE      200
#define MAXQUESTION    1000

#define F_IDENTIFIER  "FIREWALL:"
#define PROXY_HOST    "Proxy_Host"
#define PROXY_PORT    "Proxy_Port"
#define PROXY_DOMAIN  "Proxy_Domain"
#define USE_PROXY     "Use_Proxy"
#define USE_PASSIVE   "Use_Passive_Mode"

int passive_mode;
int use_proxy;
char *proxy_host;
int proxy_port;
struct sl_struct *proxy_domains_list;

extern Display *display;
extern int need_to_save_prefs;
extern int screen;

static struct {
	Widget w_shell;
	Widget w_form;
	Widget w_controlArea;
	Widget w_proxyUseToggle;
	Widget w_proxyHostLabel;
	Widget w_proxyHost;
	Widget w_proxyPortLabel;
	Widget w_proxyPort;
	Widget w_proxyDomainsLabel;
	Widget w_proxyDomains;
	Widget w_passiveUseToggle;
	Widget w_separator;
	Widget w_actionArea;
	Widget w_okButton;
	Widget w_applyButton;
	Widget w_cancelButton;
	Widget w_helpButton;
} fprefs;

static char *fprefs_help[] = {
	"This dialog enables you to specify two different mechanisms",
	"for using LLNL XDIR through a firewall: passive mode and",
	"proxying. (LLNL XDIR also supports a third type of firewall,",
	"SOCKS, but this has to be specified at compile time.)\n",
	"\n",
	"LLNL XDIR supports FTP proxies that use the simple authentication",
	"technique in which both the remote host and user name are specified",
	"in place of the user name as username@hostname.\n",
	"\n",
	"To use a proxy server, you must: (1) check the \"Use FTP Proxy",
	"Server\" box, (2) specify the host the proxy is running on, and",
	"(3) specify the port the proxy server is listening on.  If \"Use",
	"FTP Proxy Server\" is checked, the OK and APPLY buttons are greyed",
	"out until the proxy host and port are specified.   Use the text",
	"area to specify any domains (e.g., \"llnl.gov\") or host names",
	"(e.g., \"mysystem\" or \"mysystem.llnl.gov\") that you do not wish to",
	"access via the proxy server (this usually consists of local domains",
	"and hosts that are inside of the firewall).\n",
	"\n",
	"Check the \"Use Passive Mode Data Transfers (PASV)\" box to enable",
	"\"passive mode\" (also known as \"PASV mode\").  Normally, the FTP",
	"server opens data connections to the client, something that",
	"firewalls often do not allow.  Use passive mode to instead force",
	"LLNL XDIR to open data connections to the server.   Note that some",
	"FTP servers do not support passive mode, so we recommend that you",
	"use this mode only if you need to.\n",
	"\n",
	"Click the APPLY button to the cause the new changes to take effect.",
	"The OK button has the same effect as APPLY, but the dialog is also",
	"closed.  The CANCEL button undoes the changes made since the OK or",
	"APPLY buttons were last pressed.\n",
	"\n",
	"The firewall preferences can be preserved across LLNL XDIR sessions",
	"in a text file named \".xdirrc\" in the user's home directory (select",
	"SAVE PREFERENCES in a directory window's PREFS menu).\n",
	NULL
};

static int temp_fprefs_visible;
static int initialized_fprefs = False;
static int fprefs_visible = False;

extern Widget w_toplev;

void cb_fprefs_cancel();
void cb_fprefs_help();
void cb_fprefs_add();
void cb_fprefs_delete();
void cb_fprefs_ok();
void cb_fprefs_apply();
void cb_proxy_modified();
struct sl_struct *create_array_list();
struct sl_struct *create_null_array_list();
char *strip_outer_whitespace();


/*
 * cb_firewall_preferences - Callback to pop up Firewall Preferences dialog.
 */
void
cb_firewall_preferences(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
    /* Clear error flag */
    raise_okflag();

	/* Create the Firewall Preferences Dialog */
    create_fprefs_window();

	/* Update the Firewall Prefs dialog */
	update_firewall_dialog();

	/* Pop up viewer preferences window */
	XtPopup(fprefs.w_shell, XtGrabNone);
	add_dialog_to_list(fprefs.w_shell);
	traverse_to_widget(fprefs.w_proxyHost);
	XMapRaised(display, XtWindow(fprefs.w_shell));
	fprefs_visible = True;
}


/*
 * create_fprefs_window - Create window for specifying firewall preferences.
 */
create_fprefs_window()
{
	Arg args[3];
	int i;

    /* Create dialog only once */
    if (initialized_fprefs)
        return;
    initialized_fprefs = True;

	/* Create toplevel shell for directory window */
	fprefs.w_shell = XtVaCreatePopupShell("firewallPrefs",
		topLevelShellWidgetClass, w_toplev, NULL);

	/* Attach custom icon */
	attach_wm_icon(fprefs.w_shell);

	/* Create form */
	fprefs.w_form = XtVaCreateWidget("form", xmFormWidgetClass,
		fprefs.w_shell, NULL);

	/* Add callback for the WM_DELETE_WINDOW protocol */
	add_wm_delete_window_cb(fprefs.w_shell, cb_fprefs_cancel, NULL, False);

	/* Create form for action area */
    fprefs.w_actionArea = XtVaCreateWidget(
        "actionArea",
        xmFormWidgetClass,
        fprefs.w_form,
		XmNmarginHeight,		10,
		XmNbottomAttachment,	XmATTACH_FORM,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
        NULL
    );

	/* Create OK pushbutton */
    fprefs.w_okButton = XtVaCreateManagedWidget(
        "okButton",
        xmPushButtonWidgetClass,
        fprefs.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			3,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			25,
        NULL
    );
    XtAddCallback(fprefs.w_okButton, XmNactivateCallback, cb_fprefs_ok,
        (XtPointer)NULL);

	/* Create Apply pushbutton */
    fprefs.w_applyButton = XtVaCreateManagedWidget(
        "applyButton",
        xmPushButtonWidgetClass,
        fprefs.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			27,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			49,
        NULL
    );
    XtAddCallback(fprefs.w_applyButton, XmNactivateCallback, cb_fprefs_apply,
		(XtPointer)NULL);

	/* Create Cancel pushbutton */
    fprefs.w_cancelButton = XtVaCreateManagedWidget(
        "cancelButton",
        xmPushButtonWidgetClass,
        fprefs.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			51,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			73,
        NULL
    );
    XtAddCallback(fprefs.w_cancelButton, XmNactivateCallback, cb_fprefs_cancel,
		(XtPointer)NULL);
	XtVaSetValues(fprefs.w_form, XmNdefaultButton, fprefs.w_cancelButton,
		NULL);

	/* Create Help pushbutton */
    fprefs.w_helpButton = XtVaCreateManagedWidget(
        "helpButton",
        xmPushButtonWidgetClass,
        fprefs.w_actionArea,
		XmNdefaultButtonShadowThickness,	1,
		XmNmarginHeight,					4,
		XmNtopAttachment,					XmATTACH_FORM,
        XmNbottomAttachment,    			XmATTACH_FORM,
        XmNleftAttachment,      			XmATTACH_POSITION,
        XmNleftPosition,        			75,
        XmNrightAttachment,     			XmATTACH_POSITION,
        XmNrightPosition,       			97,
        NULL
    );
    XtAddCallback(fprefs.w_helpButton, XmNactivateCallback, cb_fprefs_help,
		(XtPointer)NULL);

	XtManageChild(fprefs.w_actionArea);

    /* Create separator */
    fprefs.w_separator = XtVaCreateManagedWidget(
        "separator",
        xmSeparatorWidgetClass,
        fprefs.w_form,
        XmNbottomAttachment,    XmATTACH_WIDGET,
        XmNbottomWidget,        fprefs.w_actionArea,
        XmNleftAttachment,      XmATTACH_FORM,
        XmNrightAttachment,     XmATTACH_FORM,
        NULL
    );

    /* Create form for control area */
    fprefs.w_controlArea = XtVaCreateWidget(
        "controlArea",
        xmFormWidgetClass,
        fprefs.w_form,
		XmNtopAttachment,		XmATTACH_FORM,
		XmNleftAttachment,		XmATTACH_FORM,
		XmNrightAttachment,		XmATTACH_FORM,
		XmNbottomAttachment,	XmATTACH_WIDGET,
		XmNbottomWidget,		fprefs.w_separator,
		XmNmarginHeight,		10,
		XmNmarginWidth,			10,
        NULL
    );

    /* Create proxy use toggle */
    fprefs.w_proxyUseToggle = XtVaCreateManagedWidget(
        "proxyUseToggle",
        xmToggleButtonGadgetClass,
        fprefs.w_controlArea,
		XmNindicatorType,	XmN_OF_MANY,
        XmNmarginHeight,    0,
		XmNtopAttachment,	XmATTACH_FORM,
		XmNleftAttachment,	XmATTACH_FORM,
        NULL
    );
	XtAddCallback(fprefs.w_proxyUseToggle, XmNvalueChangedCallback,
		cb_proxy_modified, (XtPointer)NULL);

    /* Create label for proxy host */
    fprefs.w_proxyHostLabel = XtVaCreateManagedWidget(
        "proxyHostLabel",
        xmLabelWidgetClass,
        fprefs.w_controlArea,
		XmNalignment,			XmALIGNMENT_BEGINNING,
        XmNtopAttachment,   	XmATTACH_WIDGET,
		XmNtopWidget,			fprefs.w_proxyUseToggle,
		XmNtopOffset,			5,
        XmNleftAttachment,  	XmATTACH_POSITION,
		XmNleftPosition,		10,
		XmNrightAttachment,		XmATTACH_POSITION,
		XmNrightPosition,		65,
        NULL
    );

    /* Create label for proxy port */
    fprefs.w_proxyPortLabel = XtVaCreateManagedWidget(
        "proxyPortLabel",
        xmLabelWidgetClass,
        fprefs.w_controlArea,
        XmNalignment,       	XmALIGNMENT_BEGINNING,
        XmNtopAttachment,   	XmATTACH_OPPOSITE_WIDGET,
		XmNtopWidget,			fprefs.w_proxyHostLabel,
		XmNleftAttachment,		XmATTACH_POSITION,
		XmNleftPosition,		70,
		XmNrightAttachment,		XmATTACH_POSITION,
		XmNrightPosition,		90,
        NULL
    );

    /* Create textfield for proxy host */
    fprefs.w_proxyHost = XtVaCreateManagedWidget(
        "proxyHost",
        xmTextFieldWidgetClass,
        fprefs.w_controlArea,
        XmNtopAttachment,   XmATTACH_WIDGET,
        XmNtopWidget,       fprefs.w_proxyHostLabel,
        XmNleftAttachment,  XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,		fprefs.w_proxyHostLabel,
		XmNrightAttachment,	XmATTACH_OPPOSITE_WIDGET,
		XmNrightWidget,		fprefs.w_proxyHostLabel,
        NULL
    );
	XtAddCallback(fprefs.w_proxyHost, XmNvalueChangedCallback,
		cb_proxy_modified, (XtPointer)NULL);

    /* Create textfield for proxy port */
    fprefs.w_proxyPort = XtVaCreateManagedWidget(
        "proxyPort",
        xmTextFieldWidgetClass,
        fprefs.w_controlArea,
        XmNtopAttachment,   	XmATTACH_WIDGET,
        XmNtopWidget,       	fprefs.w_proxyHostLabel,
		XmNleftAttachment,  	XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,			fprefs.w_proxyPortLabel,
		XmNrightAttachment,		XmATTACH_OPPOSITE_WIDGET,
		XmNrightWidget,			fprefs.w_proxyPortLabel,
        NULL
    );
	XtAddCallback(fprefs.w_proxyPort, XmNvalueChangedCallback,
		cb_proxy_modified, (XtPointer)NULL);

    /* Create label for list of ignored domains */
    fprefs.w_proxyDomainsLabel = XtVaCreateManagedWidget(
        "proxyDomainsLabel",
        xmLabelWidgetClass,
        fprefs.w_controlArea,
        XmNalignment,       	XmALIGNMENT_BEGINNING,
        XmNtopAttachment,   	XmATTACH_WIDGET,
		XmNtopWidget,			fprefs.w_proxyHost,
		XmNtopOffset,			10,
		XmNleftAttachment,		XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,			fprefs.w_proxyHostLabel,
		XmNrightAttachment,		XmATTACH_OPPOSITE_WIDGET,
		XmNrightWidget,			fprefs.w_proxyPortLabel,
        NULL
    );

    /* Create passive use toggle */
    fprefs.w_passiveUseToggle = XtVaCreateManagedWidget(
        "passiveUseToggle",
        xmToggleButtonGadgetClass,
        fprefs.w_controlArea,
		XmNindicatorType,		XmN_OF_MANY,
        XmNmarginHeight,    	0,
		XmNleftAttachment,		XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,			fprefs.w_proxyUseToggle,
		XmNbottomAttachment,	XmATTACH_FORM,
        NULL
    );

    /* Create list of ignored domains */
    i = 0;
	XtSetArg(args[i], XmNeditMode, XmMULTI_LINE_EDIT); i++;
	XtSetArg(args[i], XmNeditable, True); i++;
	XtSetArg(args[i], XmNmarginHeight, 3); i++;
    fprefs.w_proxyDomains =
		XmCreateScrolledText(fprefs.w_controlArea, "proxyDomains", args, i);
    XtVaSetValues(
        XtParent(fprefs.w_proxyDomains),
        XmNtopAttachment,   	XmATTACH_WIDGET,
        XmNtopWidget,       	fprefs.w_proxyDomainsLabel,
        XmNleftAttachment,  	XmATTACH_OPPOSITE_WIDGET,
		XmNleftWidget,			fprefs.w_proxyHostLabel,
        XmNrightAttachment, 	XmATTACH_OPPOSITE_WIDGET,
		XmNrightWidget,			fprefs.w_proxyPortLabel,
		XmNbottomAttachment,	XmATTACH_WIDGET,
		XmNbottomWidget,		fprefs.w_passiveUseToggle,
		XmNbottomOffset,		10,
        NULL
    );
    XtManageChild(fprefs.w_proxyDomains);

	XtManageChild(fprefs.w_controlArea);
	XtManageChild(fprefs.w_form);
}


/*
 * cb_fprefs_ok - Callback that applies current firewall preferences and
 *                then closes Firewall Preferences dialog.
 */
void
cb_fprefs_ok(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	apply_firewall_prefs();
	XtUnmapWidget(fprefs.w_shell);
	fprefs_visible = False;
}


/*
 * cb_fprefs_apply - Callback that applies current viewer preferences.
 */
void
cb_fprefs_apply(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	apply_firewall_prefs();
}


/*
 * cb_fprefs_cancel - Callback that restores old viewer preferences 
 *                    and then the closes Viewer Preferences dialog.
 */
void
cb_fprefs_cancel(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	XtUnmapWidget(fprefs.w_shell);
	fprefs_visible = False;
}


/*
 * cb_fprefs_help - Callback that displays help info for the Viewer
 *                  Preferences dialog.
 */
void
cb_fprefs_help(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	help_dialog(widget, False, "Firewall Preferences", fprefs_help);
}


/*
 * apply_firewall_prefs - Set current firewall preferences to displayed
 *                        values.
 */
apply_firewall_prefs()
{
	char *temp;
	char *temp1;
	char *p;
	char *q;
	char *ptr;

	need_to_save_prefs = True;

	/* Proxy server use */
	use_proxy = XmToggleButtonGadgetGetState(fprefs.w_proxyUseToggle);

	/* Proxy host name */
	XtFree(proxy_host);
	proxy_host = XmTextFieldGetString(fprefs.w_proxyHost);

	/* Proxy port */
	temp = XmTextFieldGetString(fprefs.w_proxyPort);
	ptr = temp;
	proxy_port = 0;
	while (*ptr && isdigit(*ptr))
		proxy_port = proxy_port*10+(*ptr++-'0');
	XtFree(temp);

	/* Proxy ignored domains */
	temp = XmTextGetString(fprefs.w_proxyDomains);
	release_array_list(proxy_domains_list);
	proxy_domains_list = create_null_array_list();
	p = temp;
	while ((q = strtok(p, ", \t\n")) != NULL) {
		cvt_to_lower(q);
		add_to_array_list(&proxy_domains_list, q);
		p = NULL;
	}
	XtFree(temp);

	/* Passive mode use */
	passive_mode = XmToggleButtonGadgetGetState(fprefs.w_passiveUseToggle);
}


/*
 * read_fprefs_from_file - Initializes current firewall preferences with
 *                         values from .xdirrc (which is pointed to by
 *                         "fp".  If "fp" is NULL, then use default
 *                         values.
 */
read_fprefs_from_file(fp)
FILE *fp;
{
	char fprefs_line[MAXFPLINE+1];
	char *q;
	char *name;
	char *value;
	char *ptr;

	/* Set firewall preferences to default values */
	passive_mode = False;
	use_proxy = False;
	proxy_host = XtNewString("");
	proxy_port = 0;
	proxy_domains_list = create_null_array_list();

	/* Able to read from preferences file? */
    if (fp == NULL)
		return;

	/* Process viewer preferences from file */
	while (fgets(fprefs_line, MAXFPLINE+1, fp) != NULL) {
		if (fprefs_line[strlen(fprefs_line)-1] != '\n')
			continue;
		if ((q=strtok(fprefs_line, " \t")) == NULL)
			continue;
		if (strcmp(q, F_IDENTIFIER))
			continue;
		if ((q=strtok(NULL, ":")) == NULL)
			continue;
		name = XtNewString(q);
		if ((q=strtok(NULL, " \t\n")) == NULL) {
			XtFree(name);
			continue;
		}
		value = XtNewString(q);
		if (!strcmp(name, USE_PASSIVE)) {
			if (!strcmp(value, "Yes"))
				passive_mode = True;
			else if (!strcmp(value, "No"))
				passive_mode = False;
		} else if (!strcmp(name, USE_PROXY)) {
			if (!strcmp(value, "Yes"))
				use_proxy = True;
			else if (!strcmp(value, "No"))
				use_proxy = False;
		} else if (!strcmp(name, PROXY_HOST))
			proxy_host = XtNewString(value);
		else if (!strcmp(name, PROXY_PORT)) {
			ptr = value;
			proxy_port = 0;
			while (*ptr && isdigit(*ptr))
				proxy_port = proxy_port*10+(*ptr++-'0');
		} else if (!strcmp(name, PROXY_DOMAIN)) {
			cvt_to_lower(value);
			add_to_array_list(&proxy_domains_list, value);
		}
		XtFree(name);
		XtFree(value);
	}
}


/*
 * save_fprefs - Write current viewer preferences to .xdirrc using file
 *               pointer "fp".
 */
save_fprefs(fp)
FILE *fp;
{
	int i;

	/* Passive mode */
	fprintf(fp, "%s %s: ", F_IDENTIFIER, USE_PASSIVE);
	if (passive_mode)
		fprintf(fp, "Yes\n");
	else
		fprintf(fp, "No\n");

	/* Use proxy */
	fprintf(fp, "%s %s: ", F_IDENTIFIER, USE_PROXY);
	if (use_proxy)
		fprintf(fp, "Yes\n");
	else
		fprintf(fp, "No\n");

	/* Proxy host */
	fprintf(fp, "%s %s: %s\n", F_IDENTIFIER, PROXY_HOST, proxy_host);

	/* Proxy port */
	fprintf(fp, "%s %s: %d\n", F_IDENTIFIER, PROXY_PORT, proxy_port);

	/* Domains to ignore */
	for (i=0; i<proxy_domains_list->nentries; i++)
		fprintf(fp, "%s %s: %s\n", F_IDENTIFIER, PROXY_DOMAIN,
			proxy_domains_list->entries[i]);
}


/*
 * update_firewall_dialog - Update the Firewall Preferences dialog
 *                          with the current values.
 */
update_firewall_dialog()
{
	int i;
	char buf[20];

	/* FTP proxy use */
	XmToggleButtonGadgetSetState(fprefs.w_proxyUseToggle, use_proxy, False);

	/* Proxy host and port */
	XmTextFieldSetString(fprefs.w_proxyHost, proxy_host);
	sprintf(buf, "%d", proxy_port);
	XmTextFieldSetString(fprefs.w_proxyPort, buf);

	/* Domains for proxy to ignore */
	XmTextSetString(fprefs.w_proxyDomains, "");
	for (i=0; i<proxy_domains_list->nentries; i++) {
		XmTextInsert(fprefs.w_proxyDomains,
			XmTextGetLastPosition(fprefs.w_proxyDomains),
			proxy_domains_list->entries[i]);
		XmTextInsert(fprefs.w_proxyDomains,
			XmTextGetLastPosition(fprefs.w_proxyDomains), "\n");
	}
	XmTextShowPosition(fprefs.w_proxyDomains, (XmTextPosition)0);

	/* Passive mode use */
	XmToggleButtonGadgetSetState(fprefs.w_passiveUseToggle, passive_mode,
		False);
}


/*
 * iconify_fprefs_window - Iconify viewer preferences window.
 */
iconify_fprefs_window()
{
	if (fprefs_visible)
		XIconifyWindow(display, XtWindow(fprefs.w_shell), screen);
}


/*
 * deiconify_fprefs_window - Deiconify viewer preferences window.
 */
deiconify_fprefs_window()
{
	if (fprefs_visible)
		XMapWindow(display, XtWindow(fprefs.w_shell));
}


/*
 * adjust_button_sensitivity - Make sure that OK and Apply buttons are
 *                             greyed out if settings are inconsistent
 *                             or obviously invalid.
 */
adjust_button_sensitivity()
{
	int valid = True;
	char *temp1;
	char *temp2;
	char *ptr;

	/* Check validity */
	if (XmToggleButtonGadgetGetState(fprefs.w_proxyUseToggle)) {
		temp1 = XmTextFieldGetString(fprefs.w_proxyHost);
		temp2 = strip_outer_whitespace(temp1);
		XtFree(temp1);
		if (strlen(temp2)) {
			temp1 = XmTextFieldGetString(fprefs.w_proxyPort);
			if (strlen(temp1) == 0)
				valid = False;
			else {
				ptr = temp1;
				while (*ptr)
					if (!isdigit(*ptr++)) {
						valid = False;
						break;
					}
			}
		} else
			valid = False;
		XtFree(temp2);
	}

	/* Adjust sensitivity */
	XtSetSensitive(fprefs.w_okButton, valid);
	XtSetSensitive(fprefs.w_applyButton, valid);
}


/*
 * cb_proxy_modified - Callback that is invoked when the value of the
 *                     proxy controls have changed.  The purpose of this
 *                     callback is to make the appropriate action
 *                     buttons selectable.
 */
void
cb_proxy_modified(widget, client_data, call_data)
Widget widget;
XtPointer client_data;
XtPointer call_data;
{
	adjust_button_sensitivity();
}

