/************************************************************************/
/* Module : highlightData.c					        */
/* Purpose: syntax highlighting pattern creation                        */
/* Mod. By: Keith R. Davis 				                */
/* Date   : 3/23/95					                */
/* Notes  : Copyright(c) 1996-98 Mutiny Bay Software	                */
/*          Copyright(c) 1996 Universities Research Association, Inc.	*/
/*          All rights reserved.					*/
/*                                                                      */
/*          Fermilab Nirvana GUI Library				*/
/*          June 24, 1996                      				*/
/*          By Mark Edel			                        */
/************************************************************************/

#include <stdio.h>
#include <limits.h>
#include <sys/param.h>
#include <Xm/Xm.h>
#include <Xm/Form.h>
#include <Xm/Frame.h>
#include <Xm/Text.h>
#include <Xm/LabelG.h>
#include <Xm/PushB.h>
#include <Xm/ToggleB.h>
#include <Xm/RowColumn.h>
#include <Xm/SeparatoG.h>
#include "textBuf.h"
#include "window.h"
#include "highlight.h"
#include "regularExp.h"
#include "highlightData.h"
#include "msgbox.h"             /* message box header           */

/* maximum number of file extensions allowed in a language mode */
#define MAX_FILE_EXTENSIONS 20

/* Maximum allowed number of styles (also limited by representation of
   styles as a byte - 'b') */
#define MAX_HIGHLIGHT_STYLES 128

/* Maximum number of patterns allowed in a pattern set (regular expression
   limitations are probably much more restrictive).  */
#define MAX_PATTERNS 127

/* maximum number of pattern sets allowed (we wish) */
#define N_DEFAULT_PATTERN_SETS 1
#define MAX_PATTERN_SETS 50

/* Names for the fonts that can be used for syntax highlighting */
#define N_FONT_TYPES 4

/* enumerated type preference strings */
static char *SearchMethodStrings[] = {"Literal", "CaseSense", "RegExp", NULL};
#define N_WRAP_STYLES 3
static char *AutoWrapTypes[N_WRAP_STYLES+2] = {"None", "Newline", "Continuous",
    	"True", "False"};
#define N_INDENT_STYLES 3
static char *AutoIndentTypes[N_INDENT_STYLES+2] = {"None", "Auto",
    	"Smart", "True", "False"};

/* suplement wrap and indent styles w/ a value meaning "use default" for
   the override fields in the language modes dialog */
#define DEFAULT_WRAP -1
#define DEFAULT_INDENT -1
#define DEFAULT_TAB_DIST -1
#define DEFAULT_EM_TAB_DIST -1

enum fontTypes {PLAIN_FONT, ITALIC_FONT, BOLD_FONT, BOLD_ITALIC_FONT};
char *FontTypeNames[N_FONT_TYPES] = {"Plain", "Italic", "Bold", "Bold Italic"};

typedef struct {
    char *name;
    char *color;
    int font;
} highlightStyleRec;

/* list of available highlight styles */
static int NHighlightStyles = 0;
static highlightStyleRec *HighlightStyles[MAX_HIGHLIGHT_STYLES];

/* list of available language modes and language specific preferences */
static int NLanguageModes = 0;
typedef struct {
    char *name;
    int nExtensions;
    char **extensions;
    char *recognitionExpr;
    char *delimiters;
    int wrapStyle;	
    int indentStyle;	
    int tabDist;	
    int emTabDist;	
} languageModeRec;
static languageModeRec *LanguageModes[MAX_LANGUAGE_MODES];

/* Pattern sources */
static int NPatternSets = 0;
static patternSet *PatternSets[MAX_PATTERN_SETS];

static int modeError(languageModeRec *lm, char *stringStart, char *stoppedAt,
	char *message);
static int styleError(char *stringStart, char *stoppedAt, char *message);
static int lookupNamedStyle(char *styleName);
static highlightPattern *readHighlightPatterns(char **inPtr, int withBraces,
    	char **errMsg, int *nPatterns);
static int readHighlightPattern(char **inPtr, char **errMsg,
    	highlightPattern *pattern);
/*static patternSet *readDefaultPatternSet(char *langModeName);*/
static int isDefaultPatternSet(patternSet *patSet);
static patternSet *readPatternSet(char **inPtr);
static patternSet *highlightError(char *stringStart, char *stoppedAt,
    	char *message);
static char *intToStr(int i);
static char *createPatternsString(patternSet *patSet, char *indentStr);
static highlightStyleRec *copyHighlightStyleRec(highlightStyleRec *hs);
static void freeHighlightStyleRec(highlightStyleRec *hs);
static int patternSetsDiffer(patternSet *patSet1, patternSet *patSet2);
static void freeNonNull(void *ptr);
static void freeItemCB(void *item);
static void freePatternSrc(highlightPattern *pat, int freeStruct);
static char **readExtensionList(char **inPtr, int *nExtensions);
static void freeLanguageModeRec(languageModeRec *lm);

static char *DefaultPatternSets[N_DEFAULT_PATTERN_SETS] = {
  "SQL:1:0 {\n\
    	comment:\"/\\*\":\"\\*/\"::Comment::\n\
        ansi comment:\"--\":\"$\"::Comment::\n\
    	string:\"'\":\"'\":\"\\n\":String::\n\
	numeric constant:\"<((0(x|X)[0-9a-fA-F]*)|(([0-9]+\\.?[0-9]*)|(\\.[0-9]+))((e|E)(\\+|-)?[0-9]+)?)(L|l|UL|ul|u|U|F|f)?>\":::Numeric Const::D\n\
    	storage keyword:\"<(char|varchar|int|float|real|smallint|date|time)>\":::Storage Type::D\n\
    	keyword:\"<(abort|alter|table|user|aggregrate|between|database|default|foreign|function|language|operator|rule|sequence|text|trigger|type|version|view|index|is|begin|close|cluster|commit|copy|create|declare|delete|drop|end|explain|fetch|grant|insert|into|key|like|listen|load|lock|move|notify|primary|rename|reset|revoke|rollback|select|set|show|unique|union|update|user|vacuum|values|on|and|or|not|from|using|having|group|order|by|where|with|ABORT|ALTER|TABLE|USER|AGGREGRATE|BETWEEN|DATABASE|DEFAULT|FOREIGN|FUNCTION|LANGUAGE|OPERATOR|RULE|SEQUENCE|TEXT|TRIGGER|TYPE|VERSION|VIEW|INDEX|IS|BEGIN|CLOSE|CLUSTER|COMMIT|COPY|CREATE|DECLARE|DELETE|DROP|END|EXPLAIN|FETCH|GRANT|INSERT|INTO|KEY|LIKE|LISTEN|LOAD|LOCK|MOVE|NOTIFY|PRIMARY|RENAME|RESET|REVOKE|ROLLBACK|SELECT|SET|SHOW|UNIQUE|UNION|UPDATE|USER|VACUUM|VALUES|ON|AND|OR|NOT|FROM|USING|HAVING|GROUP|ORDER|BY|WHERE|WITH)>\":::Keyword::D\n\
    	braces:\"[{}]\":::Keyword::D}"
    };

static char *default_styles = 
    "Plain:black:Plain\n\
    Comment:darkGreen:Plain\n\
    Keyword:blue:Plain\n\
    Storage Type:brown:Plain\n\
    String:VioletRed4:Plain\n\
    String1:SeaGreen:Plain\n\
    String2:darkGreen:Bold\n\
    Preprocessor:RoyalBlue4:Plain\n\
    Preprocessor1:blue:Plain\n\
    Character Const:darkGreen:Plain\n\
    Numeric Const:black:Plain\n\
    Identifier:brown:Plain\n\
    Identifier1:RoyalBlue4:Plain\n\
    Subroutine:brown:Plain\n\
    Subroutine1:chocolate:Plain\n\
    Ada Attributes:plum:Bold\n\
    Flag:red:Bold\n\
    Text Comment:SteelBlue4:Italic\n\
    Text Key:VioletRed4:Bold\n\
    Text Key1:VioletRed4:Plain\n\
    Text Arg:RoyalBlue4:Bold\n\
    Text Arg1:SteelBlue4:Bold\n\
    Text Arg2:RoyalBlue4:Plain\n\
    Text Escape:gray30:Bold\n\
    LaTeX Math:darkGreen:Plain";

static char *default_modes = "SQL:.sql::::::\".,/\\`'!|@#%^&*()-=+{}[]\"\":;<>?~\"";

/*
** Read a string (from the  value of the styles resource) containing highlight
** styles information, parse it, and load it into the stored highlight style
** list (HighlightStyles) for this mpsql session.
*/
int LoadStylesString(char *inString)
{    
    char *errMsg, *fontStr, *inPtr;
    highlightStyleRec *hs;
    int i;

    if(inString == NULL)
      inString = default_styles; 

    inPtr = inString;

    for (;;) {
   	
	/* skip over blank space */
	inPtr += strspn(inPtr, " \t");

	/* Allocate a language mode structure in which to store the info. */
	hs = (highlightStyleRec *)XtMalloc(sizeof(highlightStyleRec));

	/* read style name */
	hs->name = ReadSymbolicField(&inPtr);
	if (hs->name == NULL)
    	    return styleError(inString,inPtr, "style name required");
	if (!SkipDelimiter(&inPtr, &errMsg)) {
	    XtFree(hs->name);
	    XtFree((char *)hs);
    	    return styleError(inString,inPtr, errMsg);
    	}
    	
    	/* read color */
	hs->color = ReadSymbolicField(&inPtr);
	if (hs->color == NULL) {
	    XtFree(hs->name);
	    XtFree((char *)hs);
    	    return styleError(inString,inPtr, "color name required");
	}
	if (!SkipDelimiter(&inPtr, &errMsg)) {
	    freeHighlightStyleRec(hs);
    	    return styleError(inString,inPtr, errMsg);
    	}
    	
	/* read the font type */
	fontStr = ReadSymbolicField(&inPtr);
	for (i=0; i<N_FONT_TYPES; i++) {
	    if (!strcmp(FontTypeNames[i], fontStr)) {
	    	hs->font = i;
	    	break;
	    }
	}
	if (i == N_FONT_TYPES) {
	    XtFree(fontStr);
	    freeHighlightStyleRec(hs);
	    return styleError(inString, inPtr, "unrecognized font type");
	}
	XtFree(fontStr);

   	/* pattern set was read correctly, add/change it in the list */
   	for (i=0; i<NHighlightStyles; i++) {
	    if (!strcmp(HighlightStyles[i]->name, hs->name)) {
		freeHighlightStyleRec(HighlightStyles[i]);
		HighlightStyles[i] = hs;
		break;
	    }
	}
	if (i == NHighlightStyles) {
	    HighlightStyles[NHighlightStyles++] = hs;
   	    if (NHighlightStyles > MAX_HIGHLIGHT_STYLES)
   		return styleError(inString, inPtr,
   	    		"maximum allowable number of styles exceeded");
	}
	
    	/* if the string ends here, we're done */
   	inPtr += strspn(inPtr, " \t\n");
    	if (*inPtr == '\0')
    	    return True;
    }
}

/*
** Create a string in the correct format for the styles resource, containing
** all of the highlight styles information from the stored highlight style
** list (HighlightStyles) for this NEdit session.
*/
char *WriteStylesString(void)
{
    int i;
    char *outStr;
    textBuffer *outBuf;
    highlightStyleRec *style;
    
    outBuf = BufCreate();
    for (i=0; i<NHighlightStyles; i++) {
    	style = HighlightStyles[i];
    	BufInsert(outBuf, outBuf->length, "\t");
    	BufInsert(outBuf, outBuf->length, style->name);
    	BufInsert(outBuf, outBuf->length, ":");
    	BufInsert(outBuf, outBuf->length, style->color);
    	BufInsert(outBuf, outBuf->length, ":");
    	BufInsert(outBuf, outBuf->length, FontTypeNames[style->font]);
    	BufInsert(outBuf, outBuf->length, "\\n\\\n");
    }
    
    /* Get the output, and lop off the trailing newlines */
    outStr = BufGetRange(outBuf, 0, outBuf->length - (i==1?0:4));
    BufFree(outBuf);
    return outStr;
}

/*
** Read a string representing highlight pattern sets and add them
** to the PatternSets list of loaded highlight patterns.  Note that the
** patterns themselves are not parsed until they are actually used.
*/
int LoadHighlightString(char *inString)
{
    char *inPtr = inString;
    patternSet *patSet;
    int i;
    
    for (;;) {
   	
   	/* Read each pattern set, abort on error */
   	patSet = readPatternSet(&inPtr);
   	if (patSet == NULL)
   	    return False;
   	
	/* Add/change the pattern set in the list */
	for (i=0; i<NPatternSets; i++) {
	    if (!strcmp(PatternSets[i]->languageMode, patSet->languageMode)) {
		freePatternSet(PatternSets[i]);
		PatternSets[i] = patSet;
		break;
	    }
	}
	if (i == NPatternSets) {
	    PatternSets[NPatternSets++] = patSet;
   	    if (NPatternSets > MAX_PATTERN_SETS)
   		return False;
	}
	
    	/* if the string ends here, we're done */
   	inPtr += strspn(inPtr, " \t\n");
    	if (*inPtr == '\0')
    	    return True;
    }
}

/*
** Create a string in the correct format for the highlightPatterns resource,
** containing all of the highlight pattern information from the stored
** highlight pattern list (PatternSets) for this NEdit session.
*/
char *WriteHighlightString(void)
{
    char *outStr, *str, *escapedStr;
    textBuffer *outBuf;
    int psn, written = False;
    patternSet *patSet;
    
    outBuf = BufCreate();
    for (psn=0; psn<NPatternSets; psn++) {
    	patSet = PatternSets[psn];
    	if (patSet->nPatterns == 0)
    	    continue;
    	written = True;
    	BufInsert(outBuf, outBuf->length, patSet->languageMode);
    	BufInsert(outBuf, outBuf->length, ":");
    	if (isDefaultPatternSet(patSet))
    	    BufInsert(outBuf, outBuf->length, "Default\n\t");
    	else {
    	    BufInsert(outBuf, outBuf->length, intToStr(patSet->lineContext));
    	    BufInsert(outBuf, outBuf->length, ":");
    	    BufInsert(outBuf, outBuf->length, intToStr(patSet->charContext));
    	    BufInsert(outBuf, outBuf->length, "{\n");
    	    BufInsert(outBuf, outBuf->length,
    	    	    str = createPatternsString(patSet, "\t\t"));
    	    XtFree(str);
    	    BufInsert(outBuf, outBuf->length, "\t}\n\t");
    	}
    }
    
    /* Get the output string, and lop off the trailing newline and tab */
    outStr = BufGetRange(outBuf, 0, outBuf->length - (written?2:0));
    BufFree(outBuf);
    
    /* Protect newlines and backslashes from translation by the resource
       reader */
    escapedStr = EscapeSensitiveChars(outStr);
    XtFree(outStr);
    return escapedStr;
}

/*
** Find the font (font struct) associated with a named style.
** This routine must only be called with a valid styleName (call
** NamedStyleExists to find out whether styleName is valid).
*/
XFontStruct *FontOfNamedStyle(WindowInfo *window, char *styleName)
{
    int fontNum = HighlightStyles[lookupNamedStyle(styleName)]->font;
    XFontStruct *font;
    
    if (fontNum == BOLD_FONT)
    	font = window->boldFontStruct;
    else if (fontNum == ITALIC_FONT)
    	font = window->italicFontStruct;
    else if (fontNum == BOLD_ITALIC_FONT)
    	font = window->boldItalicFontStruct;
    else /* fontNum == PLAIN_FONT */
    	font = GetDefaultFontStruct(window->fontList);
    
    /* If font isn't loaded, silently substitute primary font */
    return font == NULL ? GetDefaultFontStruct(window->fontList) : font;
}

/*
** Find the color associated with a named style.  This routine must only be
** called with a valid styleName (call NamedStyleExists to find out whether
** styleName is valid).
*/
char *ColorOfNamedStyle(char *styleName)
{
    return HighlightStyles[lookupNamedStyle(styleName)]->color;
}

/*
** Determine whether a named style exists
*/
int NamedStyleExists(char *styleName)
{
    return lookupNamedStyle(styleName) != -1;
}

/*
** Look through the list of pattern sets, and find the one for a particular
** language.  Returns NULL if not found.
*/
patternSet *FindPatternSet(char *langModeName)
{
    int i;
    
    if (langModeName == NULL)
    	return NULL;
	
    for (i=0; i<NPatternSets; i++)
    	if (!strcmp(langModeName, PatternSets[i]->languageMode))
    	    return PatternSets[i];
    return NULL;
    
}

/*
** Create a pattern string.
*/
static char *createPatternsString(patternSet *patSet, char *indentStr)
{
    char *outStr, *str;
    textBuffer *outBuf;
    int pn;
    highlightPattern *pat;
    
    outBuf = BufCreate();
    for (pn=0; pn<patSet->nPatterns; pn++) {
    	pat = &patSet->patterns[pn];
    	BufInsert(outBuf, outBuf->length, indentStr);
    	BufInsert(outBuf, outBuf->length, pat->name);
    	BufInsert(outBuf, outBuf->length, ":");
    	if (pat->startRE != NULL) {
    	    BufInsert(outBuf, outBuf->length,
    	    	    str=MakeQuotedString(pat->startRE));
    	    XtFree(str);
    	}
    	BufInsert(outBuf, outBuf->length, ":");
    	if (pat->endRE != NULL) {
    	    BufInsert(outBuf, outBuf->length, str=MakeQuotedString(pat->endRE));
    	    XtFree(str);
    	}
    	BufInsert(outBuf, outBuf->length, ":");
    	if (pat->errorRE != NULL) {
    	    BufInsert(outBuf, outBuf->length,
    	    	    str=MakeQuotedString(pat->errorRE));
    	    XtFree(str);
    	}
    	BufInsert(outBuf, outBuf->length, ":");
    	BufInsert(outBuf, outBuf->length, pat->style);
    	BufInsert(outBuf, outBuf->length, ":");
    	if (pat->subPatternOf != NULL)
    	    BufInsert(outBuf, outBuf->length, pat->subPatternOf);
    	BufInsert(outBuf, outBuf->length, ":");
    	if (pat->flags & DEFER_PARSING)
    	    BufInsert(outBuf, outBuf->length, "D");
    	if (pat->flags & PARSE_SUBPATS_FROM_START)
    	    BufInsert(outBuf, outBuf->length, "R");
    	if (pat->flags & COLOR_ONLY)
    	    BufInsert(outBuf, outBuf->length, "C");
    	BufInsert(outBuf, outBuf->length, "\n");
    }
    outStr = BufGetAll(outBuf);
    BufFree(outBuf);
    return outStr;
}

/*
** Read in a pattern set character string, and advance *inPtr beyond it.
** Returns NULL and outputs an error to stderr on failure.
*/
static patternSet *readPatternSet(char **inPtr)
{
    char *errMsg, *stringStart = *inPtr;
    patternSet patSet, *retPatSet;

    /* remove leading whitespace */
    *inPtr += strspn(*inPtr, " \t\n");

    /* read language mode field */
    patSet.languageMode = ReadSymbolicField(inPtr);
    if (patSet.languageMode == NULL)
    	return highlightError(stringStart, *inPtr,
    	    	"language mode must be specified");
    if (!SkipDelimiter(inPtr, &errMsg))
    	return highlightError(stringStart, *inPtr, errMsg);

    /* look for "Default" keyword, and if it's there, return the default
       pattern set */
    if (!strncmp(*inPtr, "Default", 7)) {
    	*inPtr += 7;
    	retPatSet = readDefaultPatternSet(patSet.languageMode);
    	XtFree(patSet.languageMode);
    	if (retPatSet == NULL)
    	    return highlightError(stringStart, *inPtr,
    	    	    "No default pattern set");
    	return retPatSet;
    }
    	
    /* read line context field */
    if (!ReadNumericField(inPtr, &patSet.lineContext))
	return highlightError(stringStart, *inPtr,
	    	"unreadable line context field");
    if (!SkipDelimiter(inPtr, &errMsg))
    	return highlightError(stringStart, *inPtr, errMsg);

    /* read character context field */
    if (!ReadNumericField(inPtr, &patSet.charContext))
	return highlightError(stringStart, *inPtr,
	    	"unreadable character context field");

    /* read pattern list */
    patSet.patterns = readHighlightPatterns(inPtr,
   	    True, &errMsg, &patSet.nPatterns);
    if (patSet.patterns == NULL)
	return highlightError(stringStart, *inPtr, errMsg);

    /* pattern set was read correctly, make an allocated copy and return */
    retPatSet = (patternSet *)XtMalloc(sizeof(patternSet));
    memcpy(retPatSet, &patSet, sizeof(patternSet));
    return retPatSet;
}

/*
** Parse a set of highlight patterns into an array of highlightPattern
** structures, and a language mode name.  If unsuccessful, returns NULL with
** (statically allocated) message in "errMsg".
*/
static highlightPattern *readHighlightPatterns(char **inPtr, int withBraces,
    	char **errMsg, int *nPatterns)
{    
    highlightPattern *pat, *returnedList, patternList[MAX_PATTERNS];
   
    /* skip over blank space */
    *inPtr += strspn(*inPtr, " \t\n");
    
    /* look for initial brace */
    if (withBraces) {
	if (**inPtr != '{') {
    	    *errMsg = "pattern list must begin with \"{\"";
    	    return False;
	}
	(*inPtr)++;
    }
    
    /*
    ** parse each pattern in the list
    */
    pat = patternList;
    while (True) {
    	*inPtr += strspn(*inPtr, " \t\n");
    	if (**inPtr == '\0') {
    	    if (withBraces) {
    		*errMsg = "end of pattern list not found";
    		return NULL;
    	    } else
    	    	break;
	} else if (**inPtr == '}') {
	    (*inPtr)++;
    	    break;
    	}
    	if (!readHighlightPattern(inPtr, errMsg, pat++))
    	    return NULL;
    	if (pat - patternList > MAX_PATTERNS) {
    	    *errMsg = "max number of patterns exceeded\n";
    	    return NULL;
    	}
    }
    
    /* allocate a more appropriately sized list to return patterns */
    *nPatterns = pat - patternList;
    returnedList = (highlightPattern *)XtMalloc(
    	    sizeof(highlightPattern) * *nPatterns);
    memcpy(returnedList, patternList, sizeof(highlightPattern) * *nPatterns);
    return returnedList;
}

static int readHighlightPattern(char **inPtr, char **errMsg,
    	highlightPattern *pattern)
{
    /* read the name field */
    pattern->name = ReadSymbolicField(inPtr);
    if (pattern->name == NULL) {
    	*errMsg = "pattern name is required";
    	return False;
    }
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    
    /* read the start pattern */
    if (!ReadQuotedString(inPtr, errMsg, &pattern->startRE))
    	return False;
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    
    /* read the end pattern */
    if (**inPtr == ':')
    	pattern->endRE = NULL;
    else if (!ReadQuotedString(inPtr, errMsg, &pattern->endRE))
    	return False;
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    
    /* read the error pattern */
    if (**inPtr == ':')
    	pattern->errorRE = NULL;
    else if (!ReadQuotedString(inPtr, errMsg, &pattern->errorRE))
    	return False;
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    
    /* read the style field */
    pattern->style = ReadSymbolicField(inPtr);
    if (pattern->style == NULL) {
    	*errMsg = "style field required in pattern";
    	return False;
    }
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    
    /* read the sub-pattern-of field */
    pattern->subPatternOf = ReadSymbolicField(inPtr);
    if (!SkipDelimiter(inPtr, errMsg))
    	return False;
    	
    /* read flags field */
    pattern->flags = 0;
    for (; **inPtr != '\n' && **inPtr != '}'; (*inPtr)++) {
	if (**inPtr == 'D')
	    pattern->flags |= DEFER_PARSING;
	else if (**inPtr == 'R')
	    pattern->flags |= PARSE_SUBPATS_FROM_START;
	else if (**inPtr == 'C')
	    pattern->flags |= COLOR_ONLY;
	else if (**inPtr != ' ' && **inPtr != '\t') {
	    *errMsg = "unreadable flag field";
	    return False;
	}
    }
    return True;
}

/*
** Free all of the allocated data in a highlightStyleRec, including the
** structure itself.
*/
static void freeHighlightStyleRec(highlightStyleRec *hs)
{
    XtFree(hs->name);
    if (hs->color != NULL)
    	XtFree(hs->color);
    XtFree((char *)hs);
}

/*
** Given a language mode name, determine if there is a default (built-in)
** pattern set available for that language mode, and if so, read it and
** return a new allocated copy of it.  The returned pattern set should be
** freed by the caller with freePatternSet()
*/
patternSet *readDefaultPatternSet(char *langModeName)
{
    int i, modeNameLen;
    char *strPtr;
    
    modeNameLen = strlen(langModeName);
    for (i=0; i<N_DEFAULT_PATTERN_SETS; i++) {
    	if (!strncmp(langModeName, DefaultPatternSets[i], modeNameLen) &&
    	    	DefaultPatternSets[i][modeNameLen] == ':') {
    	    strPtr = DefaultPatternSets[i];
    	    return readPatternSet(&strPtr);
    	}
    }
    return NULL;
}

/*
** Return True if patSet exactly matches one of the default pattern sets
*/
static int isDefaultPatternSet(patternSet *patSet)
{
    patternSet *defaultPatSet;
    int retVal;
    
    defaultPatSet = readDefaultPatternSet(patSet->languageMode);
    if (defaultPatSet == NULL)
    	return False;
    retVal = !patternSetsDiffer(patSet, defaultPatSet);
    freePatternSet(defaultPatSet);
    return retVal;
}

/*
** Return True if "patSet1" and "patSet2" differ
*/
static int patternSetsDiffer(patternSet *patSet1, patternSet *patSet2)
{
    int i;
    highlightPattern *pat1, *pat2;
    
    if (patSet1->lineContext != patSet2->lineContext)
    	return True;
    if (patSet1->charContext != patSet2->charContext)
    	return True;
    if (patSet1->nPatterns != patSet2->nPatterns)
    	return True;
    for (i=0; i<patSet2->nPatterns; i++) {
    	pat1 = &patSet1->patterns[i];
    	pat2 = &patSet2->patterns[i];
    	if (pat1->flags != pat2->flags)
    	    return True;
    	if (AllocatedStringsDiffer(pat1->name, pat2->name))
    	    return True;
    	if (AllocatedStringsDiffer(pat1->startRE, pat2->startRE))
    	    return True;
    	if (AllocatedStringsDiffer(pat1->endRE, pat2->endRE))
    	    return True;
    	if (AllocatedStringsDiffer(pat1->errorRE, pat2->errorRE))
    	    return True;
    	if (AllocatedStringsDiffer(pat1->style, pat2->style))
    	    return True;
    	if (AllocatedStringsDiffer(pat1->subPatternOf, pat2->subPatternOf))
    	    return True;
    }
    return False;
}

/*
** Short-hand functions for formating and outputing errors for
*/
static patternSet *highlightError(char *stringStart, char *stoppedAt,
    	char *message)
{
    ParseError(NULL, stringStart, stoppedAt, "highlight pattern", message);
    return NULL;
}
static int styleError(char *stringStart, char *stoppedAt, char *message)
{
    ParseError(NULL, stringStart, stoppedAt, "style specification", message);
    return False;
}

int LoadLanguageModesString(char *inString)
{    
    char *errMsg, *styleName, *inPtr;
    languageModeRec *lm;
    int i;

    if(inString == NULL)
      inString = default_modes; 

    inPtr = inString;

    for (;;) {
   	
	/* skip over blank space */
	inPtr += strspn(inPtr, " \t\n");
    	
	/* Allocate a language mode structure to return, set unread fields to
	   empty so everything can be freed on errors by freeLanguageModeRec */
	lm = (languageModeRec *)XtMalloc(sizeof(languageModeRec));
	lm->nExtensions = 0;
	lm->recognitionExpr = NULL;
	lm->delimiters = NULL;

	/* read language mode name */
	lm->name = ReadSymbolicField(&inPtr);
	if (lm->name == NULL) {
    	    XtFree((char *)lm);
    	    return modeError(NULL,inString,inPtr,"language mode name required");
	}
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
    	/* read list of extensions */
    	lm->extensions = readExtensionList(&inPtr,
    	    	&lm->nExtensions);
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
	/* read the recognition regular expression */
	if (*inPtr == '\n' || *inPtr == '\0' || *inPtr == ':')
    	    lm->recognitionExpr = NULL;
    	else if (!ReadQuotedString(&inPtr, &errMsg, &lm->recognitionExpr))
    	    return modeError(lm, inString,inPtr, errMsg);
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
    	/* read the indent style */
    	styleName = ReadSymbolicField(&inPtr);
	if (styleName == NULL)
    	    lm->indentStyle = DEFAULT_INDENT;
    	else {
	    for (i=0; i<N_INDENT_STYLES; i++) {
	    	if (!strcmp(styleName, AutoIndentTypes[i])) {
	    	    lm->indentStyle = i;
	    	    break;
	    	}
	    }
	    XtFree(styleName);
	    if (i == N_INDENT_STYLES)
	    	return modeError(lm,inString,inPtr,"unrecognized indent style");
	}
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
    	/* read the wrap style */
    	styleName = ReadSymbolicField(&inPtr);
	if (styleName == NULL)
    	    lm->wrapStyle = DEFAULT_WRAP;
    	else {
	    for (i=0; i<N_WRAP_STYLES; i++) {
	    	if (!strcmp(styleName, AutoWrapTypes[i])) {
	    	    lm->wrapStyle = i;
	    	    break;
	    	}
	    }
	    XtFree(styleName);
	    if (i == N_WRAP_STYLES)
	    	return modeError(lm, inString, inPtr,"unrecognized wrap style");
	}
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
    	/* read the tab distance */
	if (*inPtr == '\n' || *inPtr == '\0' || *inPtr == ':')
    	    lm->tabDist = DEFAULT_TAB_DIST;
    	else if (!ReadNumericField(&inPtr, &lm->tabDist))
    	    return modeError(lm, inString, inPtr, "bad tab spacing");
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
    	/* read emulated tab distance */
    	if (*inPtr == '\n' || *inPtr == '\0' || *inPtr == ':')
    	    lm->emTabDist = DEFAULT_EM_TAB_DIST;
    	else if (!ReadNumericField(&inPtr, &lm->emTabDist))
    	    return modeError(lm, inString, inPtr, "bad emulated tab spacing");
	if (!SkipDelimiter(&inPtr, &errMsg))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
	/* read the delimiters string */
	if (*inPtr == '\n' || *inPtr == '\0')
    	    lm->delimiters = NULL;
    	else if (!ReadQuotedString(&inPtr, &errMsg, &lm->delimiters))
    	    return modeError(lm, inString, inPtr, errMsg);
    	
   	/* pattern set was read correctly, add/replace it in the list */
   	for (i=0; i<NLanguageModes; i++) {
	    if (!strcmp(LanguageModes[i]->name, lm->name)) {
		freeLanguageModeRec(LanguageModes[i]);
		LanguageModes[i] = lm;
		break;
	    }
	}
	if (i == NLanguageModes) {
	    LanguageModes[NLanguageModes++] = lm;
   	    if (NLanguageModes > MAX_LANGUAGE_MODES)
   		return modeError(NULL, inString, inPtr,
   	    		"maximum allowable number of language modes exceeded");
	}
    	
    	/* if the string ends here, we're done */
   	inPtr += strspn(inPtr, " \t\n");
    	if (*inPtr == '\0')
    	    return True;
    }
}

static void freeNonNull(void *ptr)
{
    if (ptr != NULL)
    	XtFree((char *)ptr);
}

/*
** Free the allocated memory contained in a highlightPattern data structure
** If "freeStruct" is true, free the structure itself as well.
*/
static void freePatternSrc(highlightPattern *pat, int freeStruct)
{
    XtFree(pat->name);
    freeNonNull(pat->startRE);
    freeNonNull(pat->endRE);
    freeNonNull(pat->errorRE);
    freeNonNull(pat->style);
    freeNonNull(pat->subPatternOf);
    if (freeStruct)
    	XtFree((char *)pat);
}

/*
** Free the allocated memory contained in a patternSet data structure
** If "freeStruct" is true, free the structure itself as well.
*/
void freePatternSet(patternSet *p)
{
    int i;
    
    for (i=0; i<p->nPatterns; i++)
    	freePatternSrc(&p->patterns[i], False);
    XtFree(p->languageMode);
    XtFree((char *)p->patterns);
    XtFree((char *)p);
}


/*
** Find the index into the HighlightStyles array corresponding to "styleName".
** If styleName is not found, return -1.
*/
static int lookupNamedStyle(char *styleName)
{
    int i;
    
    for (i=0; i<NHighlightStyles; i++)
    	if (!strcmp(styleName, HighlightStyles[i]->name))
    	    return i;
    return -1;
}

/*
** Write the string representation of int "i" to a static area, and
** return a pointer to it.
*/
static char *intToStr(int i)
{
    static char outBuf[12];
    
    sprintf(outBuf, "%d", i);
    return outBuf;
}

int ReadNumericField(char **inPtr, int *value)
{
    int charsRead;
    
    /* skip over blank space */
    *inPtr += strspn(*inPtr, " \t");
    
    if (sscanf(*inPtr, "%d%n", value, &charsRead) != 1)
    	return False;
    *inPtr += charsRead;
    return True;
}

/*
** Parse a symbolic field, skipping initial and trailing whitespace,
** stops on first invalid character or end of string.  Valid characters
** are letters, numbers, _, -, +, $, #, and internal whitespace.  Internal
** whitespace is compressed to single space characters.
*/
char *ReadSymbolicField(char **inPtr)
{
    char *outStr, *outPtr, *strStart, *strPtr;
    int len;
    
    /* skip over initial blank space */
    *inPtr += strspn(*inPtr, " \t");
    
    /* Find the first invalid character or end of string to know how
       much memory to allocate for the returned string */
    strStart = *inPtr;
    while (isalnum(**inPtr) || **inPtr=='_' || **inPtr=='-' ||  **inPtr=='+' ||
    	    **inPtr=='$' || **inPtr=='#' || **inPtr==' ' || **inPtr=='\t')
    	(*inPtr)++;
    len = *inPtr - strStart;
    if (len == 0)
    	return NULL;
    outStr = outPtr = XtMalloc(len + 1);
    
    /* Copy the string, compressing internal whitespace to a single space */
    strPtr = strStart;
    while (strPtr - strStart < len) {
    	if (*strPtr == ' ' || *strPtr == '\t') {
    	    strPtr += strspn(strPtr, " \t");
    	    *outPtr++ = ' ';
    	} else
    	    *outPtr++ = *strPtr++;
    }
    
    /* If there's space on the end, take it back off */
    if (outPtr > outStr && *(outPtr-1) == ' ')
    	outPtr--;
    if (outPtr == outStr) {
    	XtFree(outStr);
    	return NULL;
    }
    *outPtr = '\0';
    return outStr;
}

/*
** parse an individual quoted string.  Anything between
** double quotes is acceptable, quote characters can be escaped by "".
** Returns allocated string "string" containing
** argument minus quotes.  If not successful, returns False with
** (statically allocated) message in "errMsg".
*/
int ReadQuotedString(char **inPtr, char **errMsg, char **string)
{
    char *outPtr, *c;
    
    /* skip over blank space */
    *inPtr += strspn(*inPtr, " \t");
    
    /* look for initial quote */
    if (**inPtr != '\"') {
    	*errMsg = "expecting quoted string";
    	return False;
    }
    (*inPtr)++;
    
    /* calculate max length and allocate returned string */
    for (c= *inPtr; ; c++) {
    	if (*c == '\0') {
    	    *errMsg = "string not terminated";
    	    return False;
    	} else if (*c == '\"') {
    	    if (*(c+1) == '\"')
    	    	c++;
    	    else
    	    	break;
    	}
    }
    
    /* copy string up to end quote, transforming escaped quotes into quotes */
    *string = XtMalloc(c - *inPtr + 1);
    outPtr = *string;
    while (True) {
    	if (**inPtr == '\"') {
    	    if (*(*inPtr+1) == '\"')
    	    	(*inPtr)++;
    	    else
    	    	break;
    	}
    	*outPtr++ = *(*inPtr)++;
    }
    *outPtr = '\0';

    /* skip end quote */
    (*inPtr)++;
    return True;
}

/*
** Adds double quotes around a string and escape existing double quote
** characters with two double quotes.  Enables the string to be read back
** by ReadQuotedString.
*/
char *MakeQuotedString(char *string)
{
    char *c, *outStr, *outPtr;
    int length = 0;

    /* calculate length and allocate returned string */
    for (c=string; *c!='\0'; c++) {
    	if (*c == '\"')
    	    length++;
    	length++;
    }
    outStr = XtMalloc(length + 3);
    outPtr = outStr;
    
    /* add starting quote */
    *outPtr++ = '\"';
    
    /* copy string, escaping quotes with "" */
    for (c=string; *c!='\0'; c++) {
    	if (*c == '\"')
    	    *outPtr++ = '\"';
    	*outPtr++ = *c;
    }
    
    /* add ending quote */
    *outPtr++ = '\"';

    /* terminate string and return */
    *outPtr = '\0';
    return outStr;
}

/*
** Skip a delimiter and it's surrounding whitespace
*/
int SkipDelimiter(char **inPtr, char **errMsg)
{
    *inPtr += strspn(*inPtr, " \t");
    if (**inPtr != ':') {
    	*errMsg = "syntax error";
    	return False;
    }
    (*inPtr)++;
    *inPtr += strspn(*inPtr, " \t");
    return True;
}

/*
** Short-hand error processing for language mode parsing errors, frees
** lm (if non-null), prints a formatted message explaining where the
** error is, and returns False;
*/
static int modeError(languageModeRec *lm, char *stringStart, char *stoppedAt,
	char *message)
{
    if (lm != NULL)
    	freeLanguageModeRec(lm);
    return ParseError(NULL, stringStart, stoppedAt,
    	    "language mode specification", message);
}

/*
** Report parsing errors in resource strings or macros, formatted nicely so
** the user can tell where things became botched.  Errors can be sent either
** to stderr, or displayed in a dialog.  For stderr, pass toDialog as NULL.
** For a dialog, pass the dialog parent in toDialog.
*/
int ParseError(Widget toDialog, char *stringStart, char *stoppedAt,
	char *errorIn, char *message)
{
    int len, nNonWhite = 0;
    char *c, *errorLine;
    
    for (c=stoppedAt; c>=stringStart; c--) {
    	if (c == stringStart)
    	    break;
    	else if (*c == '\n' && nNonWhite >= 5)
    	    break;
    	else if (*c != ' ' && *c != '\t')
    	    nNonWhite++;
    }
    len = stoppedAt - c + (*stoppedAt == '\0' ? 0 : 1);
    errorLine = XtMalloc(len+4);
    strncpy(errorLine, c, len);
    errorLine[len++] = '<';
    errorLine[len++] = '=';
    errorLine[len++] = '=';
    errorLine[len] = '\0';
    fprintf(stderr, "MPSQL: %s in %s:\n%s\n", message, errorIn, errorLine);
    XtFree(errorLine);
    return False;
}

/*
** Make a new copy of a string, if NULL, return NULL
*/
char *CopyAllocatedString(char *string)
{
    char *newString;
    
    if (string == NULL)
    	return NULL;
    newString = XtMalloc(strlen(string)+1);
    strcpy(newString, string);
    return newString;
}

/*
** Compare two strings which may be NULL
*/
int AllocatedStringsDiffer(char *s1, char *s2)
{
    if (s1 == NULL && s2 == NULL)
    	return False;
    if (s1 == NULL || s2 == NULL)
    	return True;
    return strcmp(s1, s2);
}

/*
** Get the set of word delimiters for the language mode set in the current
** window.  Returns NULL when no language mode is set (it would be easy to
** return the default delimiter set when the current language mode is "Plain",
** or the mode doesn't have its own delimiters, but this is usually used
** to supply delimiters for RE searching, and ExecRE can skip compiling a
** delimiter table when delimiters is NULL).
*/
char *GetWindowDelimiters(WindowInfo *window)
{
    if (window->languageMode == PLAIN_LANGUAGE_MODE)
    	return NULL;
    else
    	return LanguageModes[window->languageMode]->delimiters;
}

/*
** Get the XFontStruct that corresponds to the default (first) font in
** a Motif font list.  Since Motif stores this, it saves us from storing
** it or querying it from the X server.
*/
XFontStruct *GetDefaultFontStruct(XmFontList font)
{
    XFontStruct *fs;
    XmFontContext context;
    XmStringCharSet charset;

    XmFontListInitFontContext(&context, font);
    XmFontListGetNextFont(context, &charset, &fs);
    XmFontListFreeFontContext(context);
    XtFree(charset);
    return fs;
}

/*
** Replace characters which the X resource file reader considers control
** characters, such that a string will read back as it appears in "string".
** (So far, newline characters are replaced with with \n\<newline> and
** backslashes with \\.  This has not been tested exhaustively, and
** probably should be.  It would certainly be more asthetic if other
** control characters were replaced as well).
**
** Returns an allocated string which must be freed by the caller with XtFree.
*/
char *EscapeSensitiveChars(char *string)
{
    char *c, *outStr, *outPtr;
    int length = 0;

    /* calculate length and allocate returned string */
    for (c=string; *c!='\0'; c++) {
    	if (*c == '\\')
    	    length++;
    	else if (*c == '\n')
    	    length += 3;
    	length++;
    }
    outStr = XtMalloc(length + 1);
    outPtr = outStr;
    
    /* add backslashes */
    for (c=string; *c!='\0'; c++) {
    	if (*c == '\\')
    	    *outPtr++ = '\\';
    	else if (*c == '\n') {
    	    *outPtr++ = '\\';
    	    *outPtr++ = 'n';
    	    *outPtr++ = '\\';
    	}
    	*outPtr++ = *c;
    }
    *outPtr = '\0';
    return outStr;
}

static char **readExtensionList(char **inPtr, int *nExtensions)
{
    char *extensionList[MAX_FILE_EXTENSIONS];
    char **retList, *strStart;
    int i, len;
    
    /* skip over blank space */
    *inPtr += strspn(*inPtr, " \t");
    
    for (i=0; i<MAX_FILE_EXTENSIONS && **inPtr!=':' && **inPtr!='\0'; i++) {
    	*inPtr += strspn(*inPtr, " \t");
	strStart = *inPtr;
	while (**inPtr!=' ' && **inPtr!='\t' && **inPtr!=':' && **inPtr!='\0')
	    (*inPtr)++;
    	len = *inPtr - strStart;
    	extensionList[i] = XtMalloc(len + 1);
    	strncpy(extensionList[i], strStart, len);
    	extensionList[i][len] = '\0';
    }
    *nExtensions = i;
    if (i == 0)
    	return NULL;
    retList = (char **)XtMalloc(sizeof(char *) * i);
    memcpy(retList, extensionList, sizeof(char *) * i);
    return retList;
}

static void freeLanguageModeRec(languageModeRec *lm)
{
    int i;
    
    XtFree(lm->name);
    if (lm->recognitionExpr != NULL)
    	XtFree(lm->recognitionExpr);
    if (lm->delimiters != NULL)
    	XtFree(lm->delimiters);
    for (i=0; i<lm->nExtensions; i++)
    	XtFree(lm->extensions[i]);
    XtFree((char *)lm->extensions);
    XtFree((char *)lm);
}

/*
** Return the name of the current language mode set in "window", or NULL
** if the current mode is "Plain".
*/
char *LanguageModeName(int mode)
{
    if (mode == PLAIN_LANGUAGE_MODE)
    	return NULL;
    else
    	return LanguageModes[mode]->name;
}
