/*
 * Snes9x - Portable Super Nintendo Entertainment System (TM) emulator.
 *
 * (c) Copyright 1996, 1997, 1998 Gary Henderson (gary@daniver.demon.co.uk) and
 *                                Jerremy Koot (jkoot@snes9x.com)
 *
 * Super FX C emulator code 
 * (c) Copyright 1997, 1998 Ivar (Ivar@snes9x.com) and
 *                          Gary Henderson.
 * Super FX assembler emulator code (c) Copyright 1998 zsKnight and _Demo_.
 *
 * DSP1 emulator code (c) Copyright 1998 Ivar, _Demo_ and Gary Henderson.
 * DOS port code contains the works of other authors. See headers in
 * individual files.
 *
 * Snes9x homepage: www.snes9x.com
 *
 * Permission to use, copy, modify and distribute Snes9x in both binary and
 * source form, for non-commercial purposes, is hereby granted without fee,
 * providing that this license information and copyright notice appear with
 * all copies and any derived work.
 *
 * This software is provided 'as-is', without any express or implied
 * warranty. In no event shall the authors be held liable for any damages
 * arising from the use of this software.
 *
 * Snes9x is freeware for PERSONAL USE only. Commercial users should
 * seek permission of the copyright holders first. Commercial use includes
 * charging money for Snes9x or software derived from Snes9x.
 *
 * The copyright holders request that bug fixes and improvements to the code
 * should be forwarded to them so everyone can benefit from the modifications
 * in future versions.
 *
 * Super NES and Super Nintendo Entertainment System are trademarks of
 * Nintendo Co., Limited and its subsidiary companies.
 */

#include <string.h>
#include <ctype.h>

#ifdef __linux
#include <unistd.h>
#endif

#ifndef NO_INLINE_SET_GET
#define NO_INLINE_SET_GET
#endif

#include "snes9x.h"
#include "memmap.h"
#include "cpuexec.h"
#include "ppu.h"
#include "display.h"
#include "cheats.h"

#ifndef ZSNES_FX
#include "fxemu.h"
extern struct FxInit_s SuperFX;
#endif

extern char *rom_filename;

bool8 CMemory::AllASCII (uint8 *b, int size)
{
    for (int i = 0; i < size; i++)
    {
	if (b[i] < 32 || b[i] > 126)
	    return (FALSE);
    }
    return (TRUE);
}

int CMemory::ScoreHiROM ()
{
    int score = 0;
    if ((Memory.ROM [0xffdc] + (Memory.ROM [0xffdd] << 8) +
	 Memory.ROM [0xffde] + (Memory.ROM [0xffdf] << 8)) == 0xffff)
	score += 2;

    if (Memory.ROM [0xffda] == 0x33)
	score += 2;
    if ((Memory.ROM [0xffd5] & 0xf) < 4)
	score += 2;
    if (!(Memory.ROM [0xfffd] & 0x80))
	score -= 4;
    if (CalculatedSize > 1024 * 1024 * 3)
	score += 4;
    if ((1 << (Memory.ROM [0xffd7] - 7)) > 48)
	score -= 1;
    if (!AllASCII (&Memory.ROM [0xffb0], 6))
	score -= 1;
    if (!AllASCII (&Memory.ROM [0xffc0], ROM_NAME_LEN - 1))
	score -= 1;

    return (score);
}

int CMemory::ScoreLoROM ()
{
    int score = 0;

    if ((Memory.ROM [0x7fdc] + (Memory.ROM [0x7fdd] << 8) +
	 Memory.ROM [0x7fde] + (Memory.ROM [0x7fdf] << 8)) == 0xffff)
	score += 2;

    if (Memory.ROM [0x7fda] == 0x33)
	score += 2;
    if ((Memory.ROM [0x7fd5] & 0xf) < 4)
	score += 2;
    if (!(Memory.ROM [0x7ffd] & 0x80))
	score -= 4;
    if ((1 << (Memory.ROM [0x7fd7] - 7)) > 48)
	score -= 1;
    if (!AllASCII (&Memory.ROM [0x7fb0], 6))
	score -= 1;
    if (!AllASCII (&Memory.ROM [0x7fc0], ROM_NAME_LEN - 1))
	score -= 1;

    return (score);
}
	
char *CMemory::Safe (const char *s)
{
    static char *safe = NULL;
    static int safe_len = 0;

    int len = strlen (s);
    if (!safe || len + 1 > safe_len)
    {
	delete safe;
	safe = new char [safe_len = len + 1];
    }

    for (int i = 0; i < len; i++)
    {
	if (s [i] >= 32 && s [i] < 127)
	    safe [i] = s[i];
	else
	    safe [i] = '?';
    }
    safe [len] = 0;
    return (safe);
}

/**********************************************************************************************/
/* Init()                                                                                     */
/* This function allocates all the memory needed by the emulator                              */
/**********************************************************************************************/
bool8 CMemory::Init ()
{
    RAM	    = new uint8 [0x20000];
    SRAM    = new uint8 [0x10000];
    VRAM    = new uint8 [0x10000];
    ROM     = new uint8 [MAX_ROM_SIZE + 0x200 + 0x8000];
    FillRAM = NULL;

    IPPU.TileCache [TILE_2BIT] = new uint8 [MAX_2BIT_TILES * 128];
    IPPU.TileCache [TILE_4BIT] = new uint8 [MAX_4BIT_TILES * 128];
    IPPU.TileCache [TILE_8BIT] = new uint8 [MAX_8BIT_TILES * 128];
    
    IPPU.TileCached [TILE_2BIT] = new uint8 [MAX_2BIT_TILES];
    IPPU.TileCached [TILE_4BIT] = new uint8 [MAX_4BIT_TILES];
    IPPU.TileCached [TILE_8BIT] = new uint8 [MAX_8BIT_TILES];
    
    if( !RAM || !SRAM || !VRAM || !ROM || !IPPU.TileCache [TILE_2BIT] ||
	!IPPU.TileCache [TILE_4BIT] || !IPPU.TileCache [TILE_8BIT] ||
	!IPPU.TileCached [TILE_2BIT] || !IPPU.TileCached [TILE_4BIT] ||
	!IPPU.TileCached [TILE_8BIT])
	return (FALSE);
	
    // FillRAM uses first 32K of ROM image area, otherwise space just
    // wasted. Might be read by the SuperFX code.

    FillRAM = ROM;

    // Add 0x8000 to ROM image pointer to stop SuperFX code accessing
    // unallocated memory (can cause crash on some ports).
    ROM += 0x8000;
    ::ROM = ROM;
    ::SRAM = SRAM;
    ::RegRAM = FillRAM;

#ifndef ZSNES_FX
    SuperFX.pvRegisters = &Memory.FillRAM [0x3000];
    SuperFX.nRamBanks = 2;
    SuperFX.pvRam = ROM + 1024 * 1024 * 4;
    SuperFX.nRomBanks = (2 * 1024 * 1024) / (32 * 1024);
    SuperFX.pvRom = (uint8 *) ROM;
#endif

    memset (SRAM, 0xaa, 0x10000);
    ZeroMemory (IPPU.TileCached [TILE_2BIT], MAX_2BIT_TILES);
    ZeroMemory (IPPU.TileCached [TILE_4BIT], MAX_4BIT_TILES);
    ZeroMemory (IPPU.TileCached [TILE_8BIT], MAX_8BIT_TILES);
    
    return (TRUE);
}

void CMemory::Deinit ()
{
    ROM -= 0x8000;

    delete RAM;
    delete SRAM;
    delete VRAM;
    delete ROM;

    delete IPPU.TileCache [TILE_2BIT];
    delete IPPU.TileCache [TILE_4BIT];
    delete IPPU.TileCache [TILE_8BIT];

    delete IPPU.TileCached [TILE_2BIT];
    delete IPPU.TileCached [TILE_4BIT];
    delete IPPU.TileCached [TILE_8BIT];
}

/**********************************************************************************************/
/* LoadROM()                                                                                  */
/* This function loads a Snes-Backup image                                                    */
/**********************************************************************************************/
bool8 CMemory::LoadROM (const char *filename)
{
    unsigned long FileSize = 0;
    int retry_count = 0;
    STREAM ROMFile;
    bool8 Interleaved = FALSE;
    bool8 Tales = FALSE;
    char dir [_MAX_DIR];
    char drive [_MAX_DRIVE];
    char name [_MAX_FNAME];
    char ext [_MAX_EXT];
    char fname [_MAX_PATH];

    CalculatedSize = 0;

again:
    _splitpath (filename, drive, dir, name, ext);
    _makepath (fname, drive, dir, name, ext);

    unsigned long TotalFileSize = 0;

#ifdef UNZIP_SUPPORT
    if (strcasecmp (ext, "zip") == 0)
    {
	bool8 LoadZip (const char *, unsigned long *, int *);

	if (!LoadZip (fname, &TotalFileSize, &HeaderCount))
	    return (FALSE);

	strcpy (Memory.ROMFilename, fname);
    }
    else
#endif
    {
	if ((ROMFile = OPEN_STREAM (fname, "rb")) == NULL)
	    return (FALSE);

	strcpy (Memory.ROMFilename, fname);

	HeaderCount = 0;
	uint8 *ptr = ROM;
	bool8 more = FALSE;

	do
	{
	    FileSize = READ_STREAM (ptr, MAX_ROM_SIZE + 0x200 - (ptr - ROM), ROMFile);
	    CLOSE_STREAM (ROMFile);
	    int calc_size = FileSize / 0x2000;
	    calc_size *= 0x2000;

	    if ((FileSize - calc_size == 512 && !Settings.ForceNoHeader) ||
		Settings.ForceHeader)
	    {
		memmove (ptr, ptr + 512, calc_size);
		HeaderCount++;
		FileSize -= 512;
	    }
	    ptr += FileSize;
	    TotalFileSize += FileSize;

	    int len;
	    if (ptr - ROM < MAX_ROM_SIZE + 0x200 &&
		(isdigit (ext [0]) && ext [1] == 0 && ext [0] < '9'))
	    {
		more = TRUE;
		ext [0]++;
		_makepath (fname, drive, dir, name, ext);
	    }
	    else
	    if (ptr - ROM < MAX_ROM_SIZE + 0x200 &&
		(((len = strlen (name)) == 7 || len == 8) &&
		 strncasecmp (name, "sf", 2) == 0 &&
		 isdigit (name [2]) && isdigit (name [3]) && isdigit (name [4]) &&
		 isdigit (name [5]) && isalpha (name [len - 1])))
	    {
		more = TRUE;
		name [len - 1]++;
		_makepath (fname, drive, dir, name, ext);
	    }
	    else
		more = FALSE;
	} while (more && (ROMFile = OPEN_STREAM (fname, "rb")) != NULL);
    }

    CalculatedSize = (TotalFileSize / 0x2000) * 0x2000;

    if (HeaderCount == 0)
	S9xMessage (S9X_INFO, S9X_HEADERS_INFO, "No ROM file header found.");
    else
    {
	if (HeaderCount == 1)
	    S9xMessage (S9X_INFO, S9X_HEADERS_INFO,
			"Found ROM file header (and ignored it).");
	else
	    S9xMessage (S9X_INFO, S9X_HEADERS_INFO,
			"Found multiple ROM file headers (and ignored them).");
    }

    ZeroMemory (ROM + CalculatedSize, MAX_ROM_SIZE - CalculatedSize);
    int orig_hi_score, orig_lo_score;
    int hi_score, lo_score;

    orig_hi_score = hi_score = ScoreHiROM ();
    orig_lo_score = lo_score = ScoreLoROM ();

    Interleaved = Settings.ForceInterleaved || Settings.ForceInterleaved2;
    if (Settings.ForceLoROM || (!Settings.ForceHiROM && lo_score >= hi_score))
    {
	Memory.LoROM = TRUE;
	Memory.HiROM = FALSE;

	switch (ROM [0x7fd5] & 0xf)
	{
	case 1:
	    Interleaved = TRUE;
	    break;
	case 2:
	    if (!Settings.ForceLoROM)
	    {
		Memory.LoROM = FALSE;
		Memory.HiROM = TRUE;
	    }
	    break;
	case 5:
	    Interleaved = TRUE;
	    Tales = TRUE;
	    break;
	}
    }
    else
    {
	switch (ROM [0xffd5] & 0xf)
	{
	case 0:
	case 3:
	    Interleaved = TRUE;
	    break;
	}
	Memory.LoROM = FALSE;
	Memory.HiROM = TRUE;
    }

    if (!Settings.ForceNotInterleaved && Interleaved)
    {
	S9xMessage (S9X_INFO, S9X_ROM_INTERLEAVED_INFO,
		    "ROM image is in interleaved format - converting...");

	int nblocks = CalculatedSize >> 15;
	int step = 64;

	while (nblocks <= step)
	    step >>= 1;
	    
	nblocks = step;
	uint8 blocks [256];
	int i;
	if (Tales)
	{
	    nblocks = 0x60;
	    for (i = 0; i < 0x40; i += 2)
	    {
		blocks [i + 0] = (i >> 1) + 0x20;
		blocks [i + 1] = (i >> 1);
	    }
	    for (i = 0; i < 0x80; i += 2)
	    {
		blocks [i + 0x40] = (i >> 1) + 0x80;
		blocks [i + 0x41] = (i >> 1) + 0x40;
	    }
	    Memory.LoROM = FALSE;
	    Memory.HiROM = TRUE;
	}
	else
	if (Settings.ForceInterleaved2)
	{
	    for (i = 0; i < nblocks * 2; i++)
	    {
		blocks [i] = (i & ~0x1e) | ((i & 2) << 2) | ((i & 4) << 2) |
			     ((i & 8) >> 2) | ((i & 16) >> 2);
	    }
	}
	else
	{
	    bool8 t = Memory.LoROM;

	    Memory.LoROM = Memory.HiROM;
	    Memory.HiROM = t;

	    for (i = 0; i < nblocks; i++)
	    {
		blocks [i * 2] = i + nblocks;
		blocks [i * 2 + 1] = i;
	    }
	}

	uint8 *tmp = new uint8 [0x8000];
	for (i = 0; i < nblocks * 2; i++)
	{
	    for (int j = i; j < nblocks * 2; j++)
	    {
		if (blocks [j] == i)
		{
		    memmove (tmp, &ROM [blocks [j] * 0x8000], 0x8000);
		    memmove (&ROM [blocks [j] * 0x8000], 
			     &ROM [blocks [i] * 0x8000], 0x8000);
		    memmove (&ROM [blocks [i] * 0x8000], tmp, 0x8000);
		    uint8 b = blocks [j];
		    blocks [j] = blocks [i];
		    blocks [i] = b;
		    break;
		}
	    }
	}
	delete tmp;
	hi_score = ScoreHiROM ();
	lo_score = ScoreLoROM ();

	if ((Memory.HiROM &&
	     (lo_score >= hi_score || orig_lo_score > hi_score ||
	      hi_score < 0)) ||
	    (Memory.LoROM && 
	     (hi_score > lo_score || orig_hi_score > lo_score ||
	      lo_score < 0)))
	{

	    if (retry_count == 0)
	    {
		S9xMessage (S9X_INFO, S9X_ROM_CONFUSING_FORMAT_INFO,
			    "ROM lied about its type! Trying again.");
		Settings.ForceNotInterleaved = TRUE;
		Settings.ForceInterleaved = FALSE;
		retry_count++;
		goto again;
	    }
	}
    }

#ifndef ZSNES_FX
    SuperFX.nRomBanks = CalculatedSize >> 15;
#endif
    Settings.MultiPlayer5Master = Settings.MultiPlayer5;
    Settings.MouseMaster = Settings.Mouse;
    Settings.SuperScopeMaster = Settings.SuperScope;
    Settings.DSP1Master = Settings.ForceDSP1;
    Settings.SuperFX = FALSE;

    ZeroMemory (BlockIsRAM, 0x800);
    ZeroMemory (BlockIsROM, 0x800);

    ::SRAM = SRAM;

    if (Memory.HiROM)
    {
	Memory.SRAMSize = ROM [0xffd8];
	strncpy (ROMName, (char *) &ROM[0xffc0], ROM_NAME_LEN - 1);
	ROMSpeed = ROM [0xffd5];
	ROMType = ROM [0xffd6];
	ROMSize = ROM [0xffd7];
	// Try to auto-detect the DSP1 chip
	if (!Settings.ForceNoDSP1 &&
	    (ROMType & 0xf) >= 3 && (ROMType & 0xf0) == 0)
	    Settings.DSP1Master = TRUE;

	if ((ROMSpeed & ~0x10) == 0x25)
	{
	    TalesROMMap ();
	}
#if 1
	else if ((ROMSpeed & 0xf) == 2 && 
		 strncmp (ROMName, "Super Street Fighter", 20) != 0)
	    AlphaROMMap ();
#endif
	else
	    HiROMMap ();
    }
    else
    {
	Memory.HiROM = FALSE;
	Memory.SRAMSize = ROM [0x7fd8];
	ROMSpeed = ROM [0x7fd5];
	ROMType = ROM [0x7fd6];
	ROMSize = ROM [0x7fd7];
	strncpy (ROMName, (char *) &ROM[0x7fc0], ROM_NAME_LEN - 1);
	Settings.SuperFX = Settings.ForceSuperFX;

	if ((ROMType & 0xf0) == 0x10)
	    Settings.SuperFX = !Settings.ForceNoSuperFX;

	// Try to auto-detect the DSP1 chip
	if (!Settings.ForceNoDSP1 &&
	    (ROMType & 0xf) >= 3 && (ROMType & 0xf0) == 0)
	    Settings.DSP1Master = TRUE;

	if (Settings.SuperFX)
	{
	    ::SRAM = ROM + 1024 * 1024 * 4;
	    SuperFXROMMap ();
	    Settings.MultiPlayer5Master = FALSE;
	    Settings.MouseMaster = FALSE;
	    Settings.SuperScopeMaster = FALSE;
	    Settings.DSP1Master = FALSE;
	}
	else
	if ((ROMSpeed & ~0x10) == 0x25)
	    TalesROMMap ();
	else
	if ((ROMSpeed & ~0x10) == 0x23 && 
	    (ROMType & 0xf) > 3 && (ROMType & 0xf0) == 0x30)
	    SA1ROMMap ();
	else
	    LoROMMap ();
    }

    if (Settings.ForceNTSC)
	Settings.PAL = FALSE;
    else
    if (Settings.ForcePAL)
	Settings.PAL = TRUE;
    else
    if (Memory.HiROM)
	// Country code
	Settings.PAL = ROM [0xffd9] >= 2;
    else
	Settings.PAL = ROM [0x7fd9] >= 2;
    
    if (Settings.PAL)
    {
	Settings.FrameTime = Settings.FrameTimePAL;
	Memory.ROMFramesPerSecond = 50;
    }
    else
    {
	Settings.FrameTime = Settings.FrameTimeNTSC;
	Memory.ROMFramesPerSecond = 60;
    }
	
    ROMName[ROM_NAME_LEN - 1] = 0;
    if (strlen (ROMName))
    {
	char *p = ROMName + strlen (ROMName) - 1;

	while (p > ROMName && *(p - 1) == ' ')
	    p--;
	*p = 0;
    }

    if (Settings.SuperFX)
    {
	SRAMMask = 0xffff;
	Memory.SRAMSize = 16;
    }
    else
    {
	SRAMMask = Memory.SRAMSize ?
		    ((1 << (Memory.SRAMSize + 3)) * 128) - 1 : 0;
    }

    ApplyROMFixes ();
    sprintf (ROMName, "%s", Safe (ROMName));

    sprintf (String, "\"%s\" %s, %s, %s, Type: %s, Mode: %s, TV: %s, S-RAM: %s",
	    ROMName, MapType (),
	    Size (),
	    Speed (),
	    KartContents (),
	    MapMode (),
	    TVStandard (),
	    StaticRAMSize ());

    S9xMessage (S9X_INFO, S9X_ROM_INFO, String);
	
    S9xApplyCheats ();
    S9xReset ();
    return TRUE;
}

bool8 CMemory::LoadSRAM (const char *filename)
{
    int size = Memory.SRAMSize ?
	       (1 << (Memory.SRAMSize + 3)) * 128 : 0;

    memset (::SRAM, 0xaa, 0x10000);

    if (size > 0x10000)
	size = 0x10000;
    if (size)
    {
	FILE *file;
	if (file = fopen (filename, "rb"))
	{
	    int len = fread ((char*) ::SRAM, 1, 0x10000, file);
	    fclose (file);
	    if (len - size == 512)
	    {
		// S-RAM file has a header - remove it
		memmove (::SRAM, ::SRAM + 512, size);
	    }
	    return (TRUE);
	}
	return (FALSE);
    }
    return (TRUE);
}

bool8 CMemory::SaveSRAM (const char *filename)
{
    int size = Memory.SRAMSize ?
	       (1 << (Memory.SRAMSize + 3)) * 128 : 0;
    if (size > 0x10000)
	size = 0x10000;

    if (size && *Memory.ROMFilename)
    {
	FILE *file;
	if (file = fopen (filename, "wb"))
	{
	    fwrite ((char *) ::SRAM, size, 1, file);
	    fclose (file);
#if defined(__linux)
	    chown (filename, getuid (), getgid ());
#endif
	    return (TRUE);
	}
	return (FALSE);
    }
    return (TRUE);
}

void CMemory::FixROMSpeed ()
{
    int c;

    for (c = 0x400; c < 0x800; c++)
    {
	if (BlockIsROM [c])
	    MemorySpeed [c] = CPU.FastROMSpeed;
    }
}

void CMemory::WriteProtectROM ()
{
    memmove ((void *) WriteMap, (void *) Map, sizeof (Map));
    for (int c = 0; c < 0x800; c++)
    {
	if (BlockIsROM [c])
	    WriteMap [c] = (uint8 *) MAP_NONE;
    }
}

void CMemory::MapRAM ()
{
    int c;

    // Banks 7e->7f, RAM
    for (c = 0; c < 8; c++)
    {
	Map [c + 0x3f0] = RAM;
	Map [c + 0x3f8] = RAM + 0x10000;
	BlockIsRAM [c + 0x3f0] = TRUE;
	BlockIsRAM [c + 0x3f8] = TRUE;
	BlockIsROM [c + 0x3f0] = FALSE;
	BlockIsROM [c + 0x3f8] = FALSE;
    }

    // Banks 70->77, S-RAM
    for (c = 0; c < 0x40; c++)
    {
	Map [c + 0x380] = (uint8 *) MAP_LOROM_SRAM;
	BlockIsRAM [c + 0x380] = TRUE;
	BlockIsROM [c + 0x380] = FALSE;
    }
}

void CMemory::LoROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	if (Settings.DSP1Master)
	    Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_DSP;
	else
	    Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_NONE;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [(c >> 1) << 13] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }

    if (Settings.DSP1Master)
    {
	// Banks 30->3f and b0->bf
	for (c = 0x180; c < 0x200; c += 8)
	{
	    Map [c + 4] = Map [c + 0x404] = (uint8 *) MAP_DSP;
	    Map [c + 5] = Map [c + 0x405] = (uint8 *) MAP_DSP;
	    Map [c + 6] = Map [c + 0x406] = (uint8 *) MAP_DSP;
	    Map [c + 7] = Map [c + 0x407] = (uint8 *) MAP_DSP;
	    BlockIsROM [c + 4] = BlockIsROM [c + 0x404] = FALSE;
	    BlockIsROM [c + 5] = BlockIsROM [c + 0x405] = FALSE;
	    BlockIsROM [c + 6] = BlockIsROM [c + 0x406] = FALSE;
	    BlockIsROM [c + 7] = BlockIsROM [c + 0x407] = FALSE;
	}
    }

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 4; i++)
	    Map [i + 0x200] = Map [i + 0x600] = &ROM [((c >> 1) << 13) + 0x200000];

	for (i = c + 4; i < c + 8; i++)
	    Map [i + 0x200] = Map [i + 0x600] = &ROM [((c >> 1) << 13) + 0x200000 - 0x8000];

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    BlockIsROM [i + 0x200] = BlockIsROM [i + 0x600] = TRUE;
	}
    }

    if (Settings.DSP1Master)
    {
	for (c = 0; c < 0x80; c++)
	{
	    Map [c + 0x700] = (uint8 *) MAP_DSP;
	    MemorySpeed [c + 0x700] = 8;
	    BlockIsROM [c + 0x700] = FALSE;
	}
    }
    MapRAM ();
    WriteProtectROM ();
}

void CMemory::HiROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	if (Settings.DSP1Master)
	    Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_DSP;
	else
	    Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_NONE;
	    
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [c << 13];
	    BlockIsROM [i] = BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }

    // Banks 30->3f address ranges 6000->7ffff is S-RAM.
    for (c = 0; c < 16; c++)
    {
	Map [0x183 + (c << 3)] = (uint8 *) MAP_HIROM_SRAM;
	Map [0x583 + (c << 3)] = (uint8 *) MAP_HIROM_SRAM;
	BlockIsRAM [0x183 + (c << 3)] = TRUE;
	BlockIsRAM [0x583 + (c << 3)] = TRUE;
    }

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 8; i++)
	{
	    Map [i + 0x200] = Map [i + 0x600] = &ROM [c << 13];
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    BlockIsROM [i + 0x200] = BlockIsROM [i + 0x600] = TRUE;
	}
    }

    MapRAM ();
    WriteProtectROM ();
}

void CMemory::TalesROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_NONE;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = &ROM [c << 13];
	    Map [i + 0x400] = &ROM [c << 13];
	    BlockIsROM [i] = TRUE;
	    BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }
    
    // Banks 30->3f address ranges 6000->7ffff is S-RAM.
    for (c = 0; c < 16; c++)
    {
	Map [0x583 + (c << 3)] = (uint8 *) MAP_HIROM_SRAM;
	BlockIsRAM [0x583 + (c << 3)] = TRUE;
    }

#define OFFSET0 0x100000
#define OFFSET1 0x000000
#define OFFSET2 0x400000
#define OFFSET3 0x200000

    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 4; i++)
	{
	    Map [i + 0x200] = &ROM [c << 13];
	    Map [i + 0x204] = &ROM [c << 13];
	    Map [i + 0x600] = &ROM [c << 13] + OFFSET3;
	    Map [i + 0x604] = &ROM [c << 13] + OFFSET3;
	    BlockIsROM [i + 0x200] = TRUE;
	    BlockIsROM [i + 0x204] = TRUE;
	    BlockIsROM [i + 0x600] = TRUE;
	    BlockIsROM [i + 0x604] = TRUE;
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    MemorySpeed [i + 0x204] = MemorySpeed [i + 0x604] = 8;
	}
    }
    MapRAM ();

    WriteProtectROM ();
}

void CMemory::AlphaROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
#if 0
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_DSP;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [c << 13];
	    BlockIsROM [i] = BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }
#else
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_DSP;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [(c >> 1) << 13] - 0x8000;
	    BlockIsROM [i] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }
#endif    
    // Banks 40->7f and c0->ff

#undef OFFSET
#define OFFSET 0x000000

#if 0
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 4; i++)
	{
	    Map [i + 0x200] = Map [i + 0x600] = &ROM [((c >> 1) << 13) + 0x200000];
	    Map [i + 0x204] = Map [i + 0x604] = &ROM [((c >> 1) << 13) + 0x200000 - 0x8000];
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    MemorySpeed [i + 0x204] = MemorySpeed [i + 0x604] = 8;
	    BlockIsROM [i + 0x200] = BlockIsROM [i + 0x204] = TRUE;
	    BlockIsROM [i + 0x600] = BlockIsROM [i + 0x604] = TRUE;
    }
#else
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 8; i++)
	{
	    Map [i + 0x200] = (uint8 *) MAP_DEBUG;
	    Map [i + 0x600] = &ROM [c << 13] + OFFSET;
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    BlockIsROM [i + 0x200] = BlockIsROM [i + 0x600] = TRUE;
	}
    }
#endif

    MapRAM ();
    WriteProtectROM ();
}

void CMemory::SuperFXROMMap ()
{
    int c;
    int i;
    int num_blocks = (1024 * 1024 * 2) / 0x2000;
    
    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	Map [c + 3] = Map [c + 0x403] = (uint8 *) MAP_DSP;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [(c >> 1) << 13] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }
    
    // Banks 40->7f and c0->ff
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 8; i++)
	{
	    Map [i + 0x200] = Map [i + 0x600] = &ROM [c << 13];
	    MemorySpeed [i + 0x200] = MemorySpeed [i + 0x600] = 8;
	    BlockIsROM [i + 0x200] = BlockIsROM [i + 0x600] = TRUE;
	}
    }

    // Banks 7e->7f, RAM
    for (c = 0; c < 8; c++)
    {
	Map [c + 0x3f0] = RAM;
	Map [c + 0x3f8] = RAM + 0x10000;
	BlockIsRAM [c + 0x3f0] = TRUE;
	BlockIsRAM [c + 0x3f8] = TRUE;
	BlockIsROM [c + 0x3f0] = FALSE;
	BlockIsROM [c + 0x3f8] = FALSE;
    }

    // Banks 70->71, S-RAM
    for (c = 0; c < 0x10; c++)
    {
	Map [c + 0x380] = ::SRAM + (((c >> 3) & 1) << 16);
	BlockIsRAM [c + 0x380] = TRUE;
	BlockIsROM [c + 0x380] = FALSE;
    }

    // Banks 00->3f and 80->bf address ranges 6000->7fff is RAM.
    for (c = 0; c < 0x40; c++)
    {
	Map [0x003 + (c << 3)] = (uint8 *) ::SRAM - 0x6000;
	Map [0x403 + (c << 3)] = (uint8 *) ::SRAM - 0x6000;
	BlockIsRAM [0x003 + (c << 3)] = TRUE;
	BlockIsRAM [0x403 + (c << 3)] = TRUE;
    }

    WriteProtectROM ();
}

void CMemory::SA1ROMMap ()
{
    int c;
    int i;

    // Banks 00->3f and 80->bf
    for (c = 0; c < 0x200; c += 8)
    {
	Map [c + 0] = Map [c + 0x400] = RAM;
	BlockIsRAM [c + 0] = BlockIsRAM [c + 0x400] = TRUE;

	Map [c + 1] = Map [c + 0x401] = (uint8 *) MAP_PPU;
	Map [c + 2] = Map [c + 0x402] = (uint8 *) MAP_CPU;
	Map [c + 3] = Map [c + 0x403] = (uint8 *) SRAM;
	for (i = c + 4; i < c + 8; i++)
	{
	    Map [i] = Map [i + 0x400] = &ROM [(c >> 1) << 13] - 0x8000;
	    BlockIsROM [i] = BlockIsROM [i + 0x400] = TRUE;
	}

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0] = 
		MemorySpeed [i + 0x400] = (i & 7) == 1 || (i & 7) == 2 ? 
					    ONE_CYCLE : 8;
	}
    }

    // Banks 40->7f
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 8; i++)
	    Map [i + 0x200] = &SRAM [(c << 13) & 0xffff];

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0x200] = 8;
	    BlockIsROM [i + 0x200] = TRUE;
	}
    }

    // c0->ff
    for (c = 0; c < 0x200; c += 8)
    {
	for (i = c; i < c + 4; i++)
	    Map [i + 0x600] = &ROM [c << 13];

	for (i = c + 4; i < c + 8; i++)
	    Map [i + 0x600] = &ROM [c << 13];

	for (i = c; i < c + 8; i++)
	{
	    MemorySpeed [i + 0x600] = 8;
	    BlockIsROM [i + 0x600] = TRUE;
	}
    }

    for (c = 0; c < 8; c++)
    {
	Map [c + 0x3f0] = RAM;
	Map [c + 0x3f8] = RAM + 0x10000;
	BlockIsRAM [c + 0x3f0] = TRUE;
	BlockIsRAM [c + 0x3f8] = TRUE;
	BlockIsROM [c + 0x3f0] = FALSE;
	BlockIsROM [c + 0x3f8] = FALSE;
    }
    WriteProtectROM ();
}

const char *CMemory::TVStandard ()
{
    return (Settings.PAL ? "PAL" : "NTSC");
}

const char *CMemory::Speed ()
{
    return (ROMSpeed & 0x10 ? "120ns" : "200ns");
}

const char *CMemory::MapType ()
{
    return (HiROM ? "HiROM" : "LoROM");
}

const char *CMemory::StaticRAMSize ()
{
    static char tmp [20];

    if (Memory.SRAMSize > 16)
	return ("Corrupt");
    sprintf (tmp, "%dKb", (SRAMMask + 1) / 1024);
    return (tmp);
}

const char *CMemory::Size ()
{
    static char tmp [20];

    if (ROMSize < 7 || ROMSize - 7 > 23)
	return ("Corrupt");
    sprintf (tmp, "%dMbits", 1 << (ROMSize - 7));
    return (tmp);
}

const char *CMemory::KartContents ()
{
    static char tmp [30];
    static const char *CoPro [16] = {
	"DSP1", "SuperFX", "OBC1", "SA-1", "S-DD1", "CoPro#5", "CoPro#6",
	"CoPro#7", "CoPro#8", "CoPro#9", "CoPro#10", "CoPro#11", "CoPro#12",
	"CoPro#13", "CoPro#14", "CoPro-Custom"
    };
    static const char *Contents [3] = {
	"ROM", "ROM+RAM", "ROM+RAM+BAT"
    };
    if (ROMType == 0)
	return ("ROM only");

    sprintf (tmp, "%s", Contents [(ROMType & 0xf) % 3]);

    if ((ROMType & 0xf) >= 3)
	sprintf (tmp, "%s+%s", tmp, CoPro [(ROMType & 0xf0) >> 4]);

    return (tmp);
}

const char *CMemory::MapMode ()
{
    static char tmp [4];
    sprintf (tmp, "%02x", ROMSpeed & ~0x10);
    return (tmp);
}

const char *CMemory::ROMID ()
{
    return (ROMId);
}

void CMemory::ApplyROMFixes ()
{
    // Breath of Fire 2 has a Hacker Intro that writes behond the limits of
    // the S-RAM the ROM header says the cartridge has and destroys saved
    // games in S-RAM without the hack below.
    if (strcmp (ROMName, "BREATH OF FIRE 2") == 0)
	SRAMMask = 0xffff;

    // XXX: Battle Toads
    Settings.BattleToadsNMIHack = FALSE;
}

#ifdef NO_INLINE_SET_GET
#define INLINE
#include "getset.h"
#endif
