#include <stdlib.h>
#include <unistd.h>
#include <sys/param.h>

#include <tcl.h>
#include <caml/mlvalues.h>
#include <caml/alloc.h>
#include <caml/memory.h>

/* The Tcl interpretor */
Tcl_Interp *cltclinterp = NULL;

/* Exception bucket for TkError */
static value tkerror_exn;

/* Initialization of TkError */
value camltk_install_tkerror_exn(bucket) /* ML */
  value bucket;
{
  tkerror_exn = Field(bucket, 0);
  register_global_root(&tkerror_exn);
  return Val_unit;
}

/* Closure for the callback dispatcher */
value handler_code;

/* Initialization: has to be called (once) with the global Caml function
 * implementing the callback dispatcher 
 */
value camltk_install_callback_handler(handler) /* ML */
  value handler;
{
  handler_code = handler;
  register_global_root(&handler_code);
  return Val_unit;
}

/* Copy a list of strings from the C heap to Caml */
value copy_string_list(argc, argv)
     int argc;
     char ** argv;
{
  value res;
  int i;
  Push_roots(r, 2);
#define oldres r[0]
#define str r[1]
  res = Val_int(0); /* [] */
  for (i = argc-1; i >= 0; i--) {
    oldres = res;
    str = copy_string(argv[i]);
    res = alloc(2, 0);
    Field(res, 0) = str;
    Field(res, 1) = oldres;
  }
  Pop_roots();
#undef oldres
#undef str
  return res;
}

/* Note: raise_with_string WILL copy the error message */
void tk_error(errmsg)
     char *errmsg;
{
  raise_with_string(tkerror_exn, errmsg);
}

/*
 * Calling Tcl from Caml
 *   this version works on an arbitrary Tcl command
 */
value camltk_tcl_eval(str) /* ML */
value str; 
{
  int code;

  if (!cltclinterp) tk_error("Tcl/Tk not initialised");

  code = Tcl_Eval(cltclinterp,String_val(str));
  switch (code) {
  case TCL_OK:
    return copy_string(cltclinterp->result);
  case TCL_ERROR:
    tk_error(cltclinterp->result);
  default:  /* TCL_BREAK, TCL_CONTINUE, TCL_RETURN */
    tk_error("bad tcl result");
  }
}


/* 
 * Calling Tcl from Caml
 *   direct call, argument is TkArgs vect
  type TkArgs =
      TkToken of string
    | TkTokenList of TkArgs list		(* to be expanded *)
    | TkQuote of TkArgs 	                (* mapped to Tcl list *)
 */

/* 
 * Compute the size of the argument (of type TkArgs). 
 * TkTokenList must be expanded,
 * TkQuote count for one.
 */
int argv_size(v)
value v;
{
  switch (Tag_val(v)) {
  case 0:			/* TkToken */
    return 1;
  case 1:			/* TkTokenList */
    { int n;
      value l;
      for (l=Field(v,0), n=0; Is_block(l); l=Field(l,1))
	n+=argv_size(Field(l,0));
      return n;
    }
  case 2:			/* TkQuote */
    return 1;
  }
}

/* 
 * Memory of allocated Tcl lists.
 * We should not need more than MAX_LIST
 */
#define MAX_LIST 256
static char *tcllists[MAX_LIST];

static int startfree = 0;
/* If size is lower, do not allocate */
static char *quotedargv[16];

/* Fill a preallocated vector arguments, doing expansion and all.
 * Assumes Tcl will 
 *  not tamper with our strings
 *  make copies if strings are "persistent"
 */
int fill_args (argv, where, v) 
char ** argv;
int where;
value v;
{
  switch (Tag_val(v)) {
  case 0:
    argv[where] = String_val(Field(v,0));
    return (where + 1);
  case 1:
    { value l;
      for (l=Field(v,0); Is_block(l); l=Field(l,1))
	where = fill_args(argv,where,Field(l,0));
      return where;
    }
  case 2:
    { char **tmpargv;
      int size = argv_size(Field(v,0));
      if (size < 16)
	tmpargv = &quotedargv[0];
      else
	tmpargv = (char **)stat_alloc((size + 1) * sizeof(char *));
      fill_args(tmpargv,0,Field(v,0));
      tmpargv[size] = NULL;
      argv[where] = Tcl_Merge(size,tmpargv);
      tcllists[startfree++] = argv[where]; /* so we can free it later */
      if (size >= 16) 
	stat_free((char *)tmpargv);
      return (where + 1);
    }
  }
}


value camltk_tcl_direct_eval(v) /* ML */
value v; 
{
  int i;
  int size;			/* size of argv */
  char **argv;
  int result;
  Tcl_CmdInfo info;
  int wherewasi,whereami;       /* positions in tcllists array */

  if (!cltclinterp) tk_error("Tcl/Tk not initialised");

  /* walk the array to compute final size for Tcl */
  for(i=0,size=0;i<Wosize_val(v);i++)
    size += argv_size(Field(v,i));

  /* +4: one slot for NULL
         one slot for "unknown" if command not found
	 two slots for chaining local roots */
  argv = (char **)stat_alloc((size + 4) * sizeof(char *));

  wherewasi = startfree; /* should be zero except when nested calls */
  Assert(startfree < MAX_LIST);

  /* Copy */
  {
    int where;
    for(i=0, where=0;i<Wosize_val(v);i++)
      where = fill_args(argv,where,Field(v,i));
    argv[size] = NULL;
    argv[size + 1] = NULL;
  }

  /* Register argv as local roots for the GC (cf. Push_roots in memory.h) */
  argv[size + 2] = (char *)(size + 2);
  argv[size + 3] = (char *) local_roots;
  local_roots = (value *)&(argv[size + 2]);

  whereami = startfree;

  /* Eval */
  Tcl_ResetResult(cltclinterp);
  if (Tcl_GetCommandInfo(cltclinterp,argv[0],&info)) { /* command found */
    result = (*info.proc)(info.clientData,cltclinterp,size,argv);
  } else /* implement the autoload stuff */
    if (Tcl_GetCommandInfo(cltclinterp,"unknown",&info)) { /* unknown found */
      for (i = size; i >= 0; i--)
	argv[i+1] = argv[i];
      argv[0] = "unknown";
      result = (*info.proc)(info.clientData,cltclinterp,size+1,argv);
    } else { /* ah, it isn't there at all */
      result = TCL_ERROR;
      Tcl_AppendResult(cltclinterp, "Unknown command \"", argv[0], "\"", NULL);
    };

  /* Remove argv from the local roots */
  Pop_roots();

  /* Free the various things we allocated */
  stat_free((char *)argv);
  for (i=wherewasi; i<whereami; i++)
    free(tcllists[i]);
  startfree = wherewasi;
  
  switch (result) {
  case TCL_OK:
    return copy_string (cltclinterp->result);
  case TCL_ERROR:
    tk_error(cltclinterp->result);
  default:  /* TCL_BREAK, TCL_CONTINUE, TCL_RETURN */
    tk_error("bad tcl result");
  }
}

